package ru.yandex.direct.core.entity.communication.repository;

import java.io.IOException;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.RecordMapper;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.communication.model.CommunicationEventType;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.dbschema.ppcdict.enums.CommunicationEventVersionsStatus;
import ru.yandex.direct.dbschema.ppcdict.tables.records.CommunicationEventVersionsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.schema.yt.tables.Eventversionconfiguration;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytcomponents.service.CommunicationEventVersionConfigDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.dbschema.ppcdict.tables.CommunicationEventVersions.COMMUNICATION_EVENT_VERSIONS;
import static ru.yandex.direct.dbschema.ppcdict.tables.CommunicationEvents.COMMUNICATION_EVENTS;
import static ru.yandex.direct.grid.schema.yt.Tables.EVENTVERSIONCONFIGURATION;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;

@Repository
@ParametersAreNonnullByDefault
public class CommunicationEventVersionsRepository {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationEventVersionsRepository.class);

    private final DslContextProvider dslContextProvider;
    private final CommunicationEventVersionMapper mapper;

    public CommunicationEventVersionsRepository(
            DslContextProvider dslContextProvider,
            CommunicationEventVersionConfigDynContextProvider dynContextProvider) {
        this.dslContextProvider = dslContextProvider;
        mapper = new CommunicationEventVersionMapper(dynContextProvider);
    }

    public List<CommunicationEventVersion> getVersions(
            Map<Long, List<Long>> eventIdToIterations) {
        if (eventIdToIterations.isEmpty()) {
            return Collections.emptyList();
        }

        var query = dslContextProvider.ppcdict()
                .select(CommunicationEventVersionMapper.ALL_COLUMNS)
                .from(COMMUNICATION_EVENT_VERSIONS);

        Condition iterCondition = null;
        for (var e: eventIdToIterations.entrySet()) {
            if (CollectionUtils.isEmpty(e.getValue())) {
                continue;
            }
            var orCondition = COMMUNICATION_EVENT_VERSIONS.EVENT_ID.eq(e.getKey())
                    .and(COMMUNICATION_EVENT_VERSIONS.ITER.in(e.getValue()));
            iterCondition = iterCondition == null ? orCondition : iterCondition.or(orCondition);
        }

        return iterCondition == null ?
                query.fetch(mapper) :
                query.where(iterCondition)
                .fetch(mapper);
    }

    public List<CommunicationEventVersion> getVersionsByStatuses(List<CommunicationEventVersionStatus> statuses) {
        return dslContextProvider.ppcdict()
                .select(CommunicationEventVersionMapper.ALL_COLUMNS)
                .from(COMMUNICATION_EVENT_VERSIONS)
                .where(COMMUNICATION_EVENT_VERSIONS.STATUS.in(
                        mapList(statuses, CommunicationEventVersionStatus::toSource)))
                .fetch(mapper);
    }

    public List<CommunicationEventVersion> getVersionByStatusesAndEventType(
            List<CommunicationEventVersionStatus> statuses,
            CommunicationEventType eventType
    ){
        return dslContextProvider.ppcdict()
                .select(CommunicationEventVersionMapper.ALL_COLUMNS)
                .from(COMMUNICATION_EVENT_VERSIONS)
                .innerJoin(COMMUNICATION_EVENTS)
                .on(COMMUNICATION_EVENTS.EVENT_ID.eq(COMMUNICATION_EVENT_VERSIONS.EVENT_ID))
                .where(COMMUNICATION_EVENT_VERSIONS.STATUS.in(
                        mapList(statuses, CommunicationEventVersionStatus::toSource)))
                .and(COMMUNICATION_EVENTS.TYPE.eq(CommunicationEventType.toSource(eventType)))
                .fetch(mapper);
    }
    public CommunicationEventVersion getVersion(Long eventId, Long iter) {
        return getOptionalVersion(eventId, iter).get();
    }

    public Optional<CommunicationEventVersion> getOptionalVersion(Long eventId, Long iter) {
        return dslContextProvider.ppcdict()
                .select(CommunicationEventVersionMapper.ALL_COLUMNS)
                .from(COMMUNICATION_EVENT_VERSIONS)
                .where(COMMUNICATION_EVENT_VERSIONS.EVENT_ID.eq(eventId))
                .and(COMMUNICATION_EVENT_VERSIONS.ITER.eq(iter))
                .fetch(mapper)
                .stream().findAny();
    }

    public List<CommunicationEventVersion> getVersionsByEvent(Long eventId) {
        return dslContextProvider.ppcdict()
                .select(CommunicationEventVersionMapper.ALL_COLUMNS)
                .from(COMMUNICATION_EVENT_VERSIONS)
                .where(COMMUNICATION_EVENT_VERSIONS.EVENT_ID.eq(eventId))
                .orderBy(COMMUNICATION_EVENT_VERSIONS.ITER.desc())
                .fetch(mapper);
    }

    /**
     * Создает итерацию события
     * @param cev
     * @return номер итерации
     */
    public boolean create(CommunicationEventVersion cev) {
        return dslContextProvider.ppcdict()
                .insertInto(COMMUNICATION_EVENT_VERSIONS)
                .set(mapper.map(cev))
                .execute() > 0;
    }

    public boolean update(CommunicationEventVersion newState, CommunicationEventVersionStatus prevStatus) {
        return update(newState, prevStatus, true);
    }

    public boolean update(CommunicationEventVersion newState, Set<CommunicationEventVersionStatus> prevStatuses) {
        return update(newState, prevStatuses, true);
    }

    /**
     * Обновляет поля итерации, если текущий статус итерации с базе совпадает с ожидаемый
     * @param newState - новое состояние итерации
     * @param prevStatus - ожидаемый статус итерации
     * @param updateLastChangeStatus - требуется ли обновлять время последнего изменения статуса
     * @return true, если изменения были успешно применены
     */
    public boolean update(CommunicationEventVersion newState, CommunicationEventVersionStatus prevStatus, boolean updateLastChangeStatus) {
        return update(newState, Set.of(prevStatus), true);
    }

    /**
     * Обновляет поля итерации, если текущий статус итерации с базе является ожидаемым
     * @param newState - новое состояние итерации
     * @param prevStatuses - список ожидаемых статусов итерации
     * @param updateLastChangeStatus - требуется ли обновлять время последнего изменения статуса
     * @return true, если изменения были успешно применены
     */
    public boolean update(CommunicationEventVersion newState, Set<CommunicationEventVersionStatus> prevStatuses, boolean updateLastChangeStatus) {
        if (updateLastChangeStatus && !prevStatuses.contains(newState.getStatus())) {
            newState.setLastChangeStatus(Instant.now().getEpochSecond() + "");
        }
        return dslContextProvider.ppcdict()
                .update(COMMUNICATION_EVENT_VERSIONS)
                .set(mapper.map(newState))
                .where(COMMUNICATION_EVENT_VERSIONS.EVENT_ID.eq(newState.getEventId()))
                .and(COMMUNICATION_EVENT_VERSIONS.ITER.eq(newState.getIter()))
                .and(COMMUNICATION_EVENT_VERSIONS.STATUS.in(mapList(prevStatuses, CommunicationEventVersionStatus::toSource)))
                .execute() > 0;
    }

    /**
     * @return для каждого статуса количество итераций и уникальных событий с такими итерациями
     */
    public Map<CommunicationEventVersionStatus, Pair<Integer, Integer>> getStatusesCount() {
        return dslContextProvider.ppcdict()
                .select(COMMUNICATION_EVENT_VERSIONS.STATUS,
                        DSL.count(),
                        DSL.countDistinct(COMMUNICATION_EVENT_VERSIONS.EVENT_ID))
                .from(COMMUNICATION_EVENT_VERSIONS)
                .groupBy(COMMUNICATION_EVENT_VERSIONS.STATUS)
                .fetchMap(
                        r -> CommunicationEventVersionStatus.fromSource(r.getValue(COMMUNICATION_EVENT_VERSIONS.STATUS)),
                        r -> Pair.of(r.value2(), r.value3())
                );
    }

    public List<CommunicationEventVersionStatus> getStatusesByEventId(Long eventId) {
        return dslContextProvider.ppcdict()
                .select(COMMUNICATION_EVENT_VERSIONS.STATUS)
                .from(COMMUNICATION_EVENT_VERSIONS)
                .where(COMMUNICATION_EVENT_VERSIONS.EVENT_ID.eq(eventId))
                .fetch()
                .stream()
                .map(result -> CommunicationEventVersionStatus.fromSource(result.value1()))
                .collect(Collectors.toList());
    }

    private static class CommunicationEventVersionMapper implements RecordMapper<Record, CommunicationEventVersion> {

        private static final List<TableField> ALL_COLUMNS = new ArrayList<>();
        private static final Map<ModelProperty, TableField> MODEL_TO_COLUMNS = new HashMap();
        private static final Map<TableField, ModelProperty> COLUMN_TO_ONE_FIELD = new HashMap();
        private static final Map<TableField<CommunicationEventVersionsRecord, String>,List<ModelProperty<CommunicationEventVersion, String>>>
                COLUMN_TO_LIST_OF_FIELD= new HashMap();
        static {
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.EVENT_ID, COMMUNICATION_EVENT_VERSIONS.EVENT_ID);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.ITER, COMMUNICATION_EVENT_VERSIONS.ITER);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.STATUS, COMMUNICATION_EVENT_VERSIONS.STATUS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.START_TIME, COMMUNICATION_EVENT_VERSIONS.START_TIME);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.EXPIRED, COMMUNICATION_EVENT_VERSIONS.EXPIRED);

            MODEL_TO_COLUMNS.put(CommunicationEventVersion.USERS, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.TITLE, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.TEXT, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.BUTTON_TEXT, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.BUTTON_HREF, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.IMAGE_HREF, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.ON_CLIENT_CREATE, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.CHECK_ACTUAL, COMMUNICATION_EVENT_VERSIONS.PARAMS);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.FORMAT_NAME, COMMUNICATION_EVENT_VERSIONS.PARAMS);

            MODEL_TO_COLUMNS.put(CommunicationEventVersion.CLUSTER, COMMUNICATION_EVENT_VERSIONS.INFO);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.EVENTS_TABLE_PATH, COMMUNICATION_EVENT_VERSIONS.INFO);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.ROLLBACK_EVENTS_TABLE_PATH, COMMUNICATION_EVENT_VERSIONS.INFO);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.LAST_CHANGE_STATUS, COMMUNICATION_EVENT_VERSIONS.INFO);
            MODEL_TO_COLUMNS.put(CommunicationEventVersion.USER_TABLE_HASH, COMMUNICATION_EVENT_VERSIONS.INFO);

            Map<TableField,List<ModelProperty>> temp = EntryStream.of(MODEL_TO_COLUMNS)
                    .invert()
                    .grouping();

            COLUMN_TO_ONE_FIELD.putAll(
                    EntryStream.of(temp)
                            .filterValues(list -> list.size() == 1)
                            .mapValues(list -> list.get(0))
                            .toMap()
            );

            COLUMN_TO_LIST_OF_FIELD.putAll(
                    (Map) EntryStream.of(temp)
                            .filterValues(list -> list.size() > 1)
                            .toMap()
            );

            ALL_COLUMNS.addAll(COLUMN_TO_ONE_FIELD.keySet());
            ALL_COLUMNS.addAll(COLUMN_TO_LIST_OF_FIELD.keySet());
        }

        private static final Eventversionconfiguration CONFIG = EVENTVERSIONCONFIGURATION.as("CONFIG");
        private static final Field<String> CONFIG_DATA = aliased(CONFIG.DATA);

        private static final TableSchema TABLE_SCHEMA = new TableSchema.Builder()
                .addKey(CONFIG.EVENT_ID.getName(), ColumnValueType.INT64)
                .addKey(CONFIG.EVENT_VERSION_ID.getName(), ColumnValueType.INT64)
                .addValue(CONFIG.STATUS.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.DESCRIPTION.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.CREATION_TIMESTAMP.getName(), ColumnValueType.INT64)
                .addValue(CONFIG.ST_TICKET.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.START_TIME.getName(), ColumnValueType.INT64)
                .addValue(CONFIG.FINISH_TIME.getName(), ColumnValueType.INT64)
                .addValue(CONFIG.SLOTS.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.FEATURES.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.FILTER_FEATURES.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.TEST_IDS.getName(), ColumnValueType.STRING)
                .addValue(CONFIG.DATA.getName(), ColumnValueType.STRING)
                .build();

        private final CommunicationEventVersionConfigDynContextProvider provider;

        private CommunicationEventVersionMapper(CommunicationEventVersionConfigDynContextProvider provider) {
            this.provider = provider;
        }

        @Override
        public CommunicationEventVersion map(Record record) {
            CommunicationEventVersion cev = readCommunicationEventVersionFromYt(
                    record.getValue(COMMUNICATION_EVENT_VERSIONS.EVENT_ID),
                    record.getValue(COMMUNICATION_EVENT_VERSIONS.ITER)
            );
            EntryStream.of(COLUMN_TO_ONE_FIELD)
                    .forEach(e -> {
                        var value = record.getValue(e.getKey());
                        if (e.getKey().equals(COMMUNICATION_EVENT_VERSIONS.STATUS)) {
                            value = CommunicationEventVersionStatus.fromSource((CommunicationEventVersionsStatus) value);
                        }
                        e.getValue().set(cev, value);
                    });
            EntryStream.of(COLUMN_TO_LIST_OF_FIELD)
                    .forEach(e ->
                        fromDb(cev, record.getValue(e.getKey()), e.getValue())
                    );
            return cev;
        }

        private CommunicationEventVersion readCommunicationEventVersionFromYt(Long eventId, Long version) {
            var query = YtDSL.ytContext().select(CONFIG_DATA)
                    .from(CONFIG)
                    .where(CONFIG.EVENT_ID.eq(eventId))
                    .and(CONFIG.EVENT_VERSION_ID.eq(version));
            var rows = provider.getContext().executeSelect(query);
            return rows.getRows().stream()
                    .findAny()
                    .map(stringValueGetter(rows.getSchema(), CONFIG_DATA))
                    .map(data -> {
                        try {
                            return JsonUtils.getObjectMapper().readValue(data, CommunicationEventVersion.class);
                        } catch (IOException e) {
                            logger.error(String.format("Can't read EventVersion config for eventId = %d, version = %d", eventId, version), e);
                            return new CommunicationEventVersion();
                        }
                    })
                    .orElse(new CommunicationEventVersion());
        }

        private void writeCommunicationEventVersionToYt(CommunicationEventVersion config) {
            String tablePath = provider.getYtConfigurationTablePath();
            provider.getYtWriteOperator().runInTransaction(
                    transaction -> upsertConfig(transaction, tablePath, config),
                    new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
            );
        }

        private void upsertConfig(ApiServiceTransaction transaction, String tablePath, CommunicationEventVersion config) {
            String data;
            try {
                data = JsonUtils.getObjectMapper().writeValueAsString(config);
            } catch (JsonProcessingException e) {
                logger.error(String.format("Can't write EventVersion config for eventId = %d, version = %d", config.getEventId(), config.getIter()), e);
                throw new RuntimeException(e);
            }
            String slots = config.getSlots() == null || config.getSlots().isEmpty() ?
                    null : String.join(",", mapList(config.getSlots(), l -> l.toString()));
            var request = new ModifyRowsRequest(tablePath, TABLE_SCHEMA)
                    .setRequireSyncReplica(false)
                    .addInsert(Arrays.asList(
                            config.getEventId(),
                            config.getIter(),
                            config.getStatus().name(),
                            config.getDescription(),
                            config.getCreationTimestamp(),
                            config.getStTicket(),
                            config.getStartTime().toEpochSecond(ZoneOffset.UTC),
                            config.getExpired().toEpochSecond(ZoneOffset.UTC),
                            listToString(config.getSlots()),
                            listToString(config.getFeatures()),
                            listToString(config.getFilterFeatures()),
                            listToString(config.getTestIds()),
                            data
                    ));

            try {
                transaction.modifyRows(request).get(60, TimeUnit.SECONDS);
            } catch (InterruptedException | ExecutionException | TimeoutException ex) {
                throw new RuntimeException(ex);
            }
        }

        private String listToString(List list) {
            if (list == null || list.isEmpty()) {
                return null;
            }
            return String.join(",", mapList(list, Objects::toString));
        }

        public Record map(CommunicationEventVersion cev) {
            writeCommunicationEventVersionToYt(cev);
            CommunicationEventVersionsRecord record = new CommunicationEventVersionsRecord();
            EntryStream.of(COLUMN_TO_ONE_FIELD)
                    .forEach(e -> {
                        var value = e.getValue().get(cev);
                        if (e.getKey().equals(COMMUNICATION_EVENT_VERSIONS.STATUS)) {
                            value = CommunicationEventVersionStatus.toSource((CommunicationEventVersionStatus) value);
                        }
                        record.setValue(e.getKey(), value);
                    });
            EntryStream.of(COLUMN_TO_LIST_OF_FIELD)
                    .forEach(e ->
                        record.setValue(e.getKey(), toDb(cev, e.getValue()))
                    );
            return record;
        }

        private static void fromDb(
                CommunicationEventVersion cev,
                String mapString,
                List<ModelProperty<CommunicationEventVersion, String>> modelProperties) {
            var map = parseToMap(mapString);
            modelProperties.forEach(property -> {
                var value = map.get(property.name());
                property.set(cev, value);
            });
        }

        private static final String KEY_VALUE_SEPARATOR = "=";
        private static final String ENTRY_SEPARATOR = ";;;";

        private static String toDb(
                CommunicationEventVersion cev,
                List<ModelProperty<CommunicationEventVersion, String>> modelProperties) {
            return StreamEx.of(modelProperties)
                    .sortedBy(ModelProperty::name)
                    .map(property -> {
                        var value = property.get(cev);
                        return value == null ? null : property.name() + KEY_VALUE_SEPARATOR + value;
                    })
                    .filter(Objects::nonNull)
                    .collect(Collectors.joining(ENTRY_SEPARATOR));
        }

        private static Map<String, String> parseToMap(String source) {
            return StringUtils.isEmpty(source)
                    ? Collections.emptyMap()
                    : StreamEx.of(source.split(ENTRY_SEPARATOR))
                    .map(x -> x.split(KEY_VALUE_SEPARATOR, 2))
                    .filter(x -> x.length > 1)
                    .toMap(x -> x[0], x -> x[1]);
        }
    }
}
