package ru.yandex.direct.core.entity.communication.repository;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventType;
import ru.yandex.direct.dbschema.ppcdict.enums.CommunicationEventsStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.grid.schema.yt.tables.Eventconfiguration;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytcomponents.service.CommunicationEventConfigDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.dbschema.ppcdict.tables.CommunicationEvents.COMMUNICATION_EVENTS;
import static ru.yandex.direct.grid.schema.yt.Tables.EVENTCONFIGURATION;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.stringValueGetter;

@Repository
@ParametersAreNonnullByDefault
public class CommunicationEventsRepository {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationEventsRepository.class);
    private static final Eventconfiguration CONFIG = EVENTCONFIGURATION.as("CONFIG");
    private static final Field<String> CONFIG_DATA = aliased(CONFIG.DATA);

    private static final TableSchema TABLE_SCHEMA = new TableSchema.Builder()
            .addKey(CONFIG.ID.getName(), ColumnValueType.INT64)
            .addValue(CONFIG.TYPE.getName(), ColumnValueType.STRING)
            .addValue(CONFIG.STATUS.getName(), ColumnValueType.STRING)
            .addValue(CONFIG.NAME.getName(), ColumnValueType.STRING)
            .addValue(CONFIG.DESCRIPTION.getName(), ColumnValueType.STRING)
            .addValue(CONFIG.CREATION_TIMESTAMP.getName(), ColumnValueType.INT64)
            .addValue(CONFIG.ST_TICKET.getName(), ColumnValueType.STRING)
            .addValue(CONFIG.DATA.getName(), ColumnValueType.STRING)
            .build();

    private final JooqMapperWithSupplier<CommunicationEvent> communicationEventMapper;
    private final DslContextProvider dslContextProvider;
    private final CommunicationEventConfigDynContextProvider dynContextProvider;

    @Autowired
    public CommunicationEventsRepository(
            DslContextProvider dslContextProvider,
            CommunicationEventConfigDynContextProvider dynContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.dynContextProvider = dynContextProvider;
        this.communicationEventMapper = JooqMapperWithSupplierBuilder.builder(CommunicationEvent::new)
                .map(property(CommunicationEvent.EVENT_ID, COMMUNICATION_EVENTS.EVENT_ID))
                .map(convertibleProperty(CommunicationEvent.TYPE, COMMUNICATION_EVENTS.TYPE,
                        CommunicationEventType::fromSource, CommunicationEventType::toSource))
                .map(property(CommunicationEvent.NAME, COMMUNICATION_EVENTS.NAME))
                .map(convertibleProperty(CommunicationEvent.OWNERS, COMMUNICATION_EVENTS.OWNERS,
                        CommunicationEventsRepository::ownersFromDb, CommunicationEventsRepository::ownersToDb))
                .map(convertibleProperty(CommunicationEvent.STATUS, COMMUNICATION_EVENTS.STATUS,
                        CommunicationEventStatus::fromSource, CommunicationEventStatus::toSource))
                .map(property(CommunicationEvent.ACTIVATE_TIME, COMMUNICATION_EVENTS.ACTIVATE_TIME))
                .map(property(CommunicationEvent.ARCHIVED_TIME, COMMUNICATION_EVENTS.ARCHIVED_TIME))
                .build();
    }

    public Long addCommunicationEvent(CommunicationEvent event) {
        Long id = InsertHelper.addModelsAndReturnIds(dslContextProvider.ppcdict(), COMMUNICATION_EVENTS,
                communicationEventMapper, List.of(event), COMMUNICATION_EVENTS.EVENT_ID).get(0);
        writeCommunicationEventToYt(event.withEventId(id));
        return id;
    }
    public void updateCommunicationEvent(CommunicationEvent newState) {
        dslContextProvider.ppcdict()
                .update(COMMUNICATION_EVENTS)
                .set(COMMUNICATION_EVENTS.OWNERS, ownersToDb(newState.getOwners()))
                .set(COMMUNICATION_EVENTS.STATUS, CommunicationEventStatus.toSource(newState.getStatus()))
                .set(COMMUNICATION_EVENTS.ACTIVATE_TIME, newState.getActivateTime())
                .set(COMMUNICATION_EVENTS.ARCHIVED_TIME, newState.getArchivedTime())
                .where(COMMUNICATION_EVENTS.EVENT_ID.eq(newState.getEventId()))
                .execute();
        writeCommunicationEventToYt(newState);
    }

    public void updateCommunicationEventOwners(Long eventId, Set<String> owners) {
        var event = getCommunicationEventsById(eventId);
        if (event.isEmpty()) {
            return;
        }
        dslContextProvider.ppcdict()
                .update(COMMUNICATION_EVENTS)
                .set(COMMUNICATION_EVENTS.OWNERS, ownersToDb(owners))
                .where(COMMUNICATION_EVENTS.EVENT_ID.eq(eventId))
                .execute();
        writeCommunicationEventToYt(event.get().withOwners(owners));
    }

    public void archiveCommunicationEvent(Long eventId, LocalDateTime time) {
        var event = getCommunicationEventsById(eventId);
        if (event.isEmpty()) {
            return;
        }
        dslContextProvider.ppcdict()
                .update(COMMUNICATION_EVENTS)
                .set(COMMUNICATION_EVENTS.STATUS, CommunicationEventsStatus.archived)
                .set(COMMUNICATION_EVENTS.ARCHIVED_TIME, time)
                .where(COMMUNICATION_EVENTS.EVENT_ID.eq(eventId))
                .execute();
        writeCommunicationEventToYt(event.get()
                .withStatus(CommunicationEventStatus.ARCHIVED)
                .withArchivedTime(time));
    }

    public void activateCommunicationEvent(Long eventId, LocalDateTime time) {
        var optionalEvent = getCommunicationEventsById(eventId);
        if (optionalEvent.isEmpty()) {
            return;
        }
        var event = optionalEvent.get();
        var updateSetStep = dslContextProvider.ppcdict()
                .update(COMMUNICATION_EVENTS)
                .set(COMMUNICATION_EVENTS.STATUS, CommunicationEventsStatus.active);
        event.setStatus(CommunicationEventStatus.ACTIVE);
        if (time != null) {
            updateSetStep.set(COMMUNICATION_EVENTS.ACTIVATE_TIME, time);
            event.setActivateTime(time);
        }
        updateSetStep
                .where(COMMUNICATION_EVENTS.EVENT_ID.eq(eventId))
                .execute();
        writeCommunicationEventToYt(event);
    }

    public List<CommunicationEvent> getCommunicationEventsByStatuses(List<CommunicationEventStatus> statuses) {
        List<CommunicationEvent> events = dslContextProvider.ppcdict()
                .select(communicationEventMapper.getFieldsToRead())
                .from(COMMUNICATION_EVENTS)
                .where(COMMUNICATION_EVENTS.STATUS.in(mapList(statuses, CommunicationEventStatus::toSource)))
                .orderBy(COMMUNICATION_EVENTS.EVENT_ID.desc())
                .fetch()
                .map(communicationEventMapper::fromDb);
        return StreamEx.of(events)
                .map(e -> readCommunicationEventFromYt(e.getEventId()).orElse(e))
                .toList();
    }

    public List<CommunicationEvent> getAllCommunicationEvents() {
        List<CommunicationEvent> events = dslContextProvider.ppcdict()
                .select(communicationEventMapper.getFieldsToRead())
                .from(COMMUNICATION_EVENTS)
                .fetch()
                .map(communicationEventMapper::fromDb);
        return StreamEx.of(events)
                .map(e -> readCommunicationEventFromYt(e.getEventId()).orElse(e))
                .toList();
    }

    public Optional<CommunicationEvent> getCommunicationEventsById(Long eventId) {
        return getCommunicationEventsByIds(List.of(eventId)).stream().findAny();
    }

    public List<CommunicationEvent> getCommunicationEventsByIds(Collection<Long> eventIds) {
        List<CommunicationEvent> events = dslContextProvider.ppcdict()
                .select(communicationEventMapper.getFieldsToRead())
                .from(COMMUNICATION_EVENTS)
                .where(COMMUNICATION_EVENTS.EVENT_ID.in(eventIds))
                .fetch()
                .map(communicationEventMapper::fromDb);
        return StreamEx.of(events)
                .map(e -> readCommunicationEventFromYt(e.getEventId()).orElse(e))
                .toList();
    }

    private void writeCommunicationEventToYt(CommunicationEvent config) {
        String tablePath = dynContextProvider.getYtConfigurationTablePath();
        dynContextProvider.getYtWriteOperator().runInTransaction(
                transaction -> upsertConfig(transaction, tablePath, config),
                new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
        );
    }

    private void upsertConfig(ApiServiceTransaction transaction, String tablePath, CommunicationEvent config) {
        String data;
        try {
            data = JsonUtils.getObjectMapper().writeValueAsString(config);
        } catch (JsonProcessingException e) {
            logger.error(String.format("Can't write Event config for eventId = %d", config.getEventId()), e);
            throw new RuntimeException(e);
        }
        var request = new ModifyRowsRequest(tablePath, TABLE_SCHEMA)
                .setRequireSyncReplica(false)
                .addInsert(Arrays.asList(
                        config.getEventId(),
                        config.getType().name(),
                        config.getStatus().name(),
                        config.getName(),
                        config.getDescription(),
                        config.getCreationTimestamp(),
                        config.getStTicket(),
                        data
                ));

        try {
            transaction.modifyRows(request).get(60, TimeUnit.SECONDS);
        } catch (InterruptedException | ExecutionException | TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    private Optional<CommunicationEvent> readCommunicationEventFromYt(Long eventId) {
        var query = YtDSL.ytContext().select(CONFIG_DATA)
                .from(CONFIG)
                .where(CONFIG.ID.eq(eventId));
        var rows = dynContextProvider.getContext().executeSelect(query);
        return rows.getRows().stream()
                .findAny()
                .map(stringValueGetter(rows.getSchema(), CONFIG_DATA))
                .map(data -> {
                    try {
                        return JsonUtils.getObjectMapper().readValue(data, CommunicationEvent.class);
                    } catch (IOException e) {
                        logger.error(String.format("Can't read Event config for eventId = %d", eventId), e);
                        return null;
                    }
                });
    }

    private static String ownersToDb(@Nullable Set<String> owners) {
        return owners != null ? StringUtils.join(owners, ",") : null;
    }

    private static Set<String> ownersFromDb(@Nullable String owners) {
        return owners != null
                ? StreamEx.split(owners, ",").collect(Collectors.toSet())
                : null;
    }
}
