package ru.yandex.direct.core.entity.contentpromotion;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.type.ContentPromotionCoreTypeSupportFacade;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.contentNotFound;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;

/**
 * Операция, по получению и добавлении в базу при необходимости продвигаемого контента
 */
public class ContentPromotionAddOrGetOperation {

    private final ContentPromotionRepository contentPromotionRepository;
    private final ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade;

    private List<ContentPromotionContentBasicData> basicDataList;
    private ListValidationBuilder<ContentPromotionContentBasicData, Defect> listValidationBuilder;
    private Map<Integer, String> externalIdsByIndex;
    private Map<Integer, Long> contentIdsByIndex;
    private Map<Integer, ContentPromotionSingleObjectRequest> requestsByIndex;
    private ClientId clientId;

    public ContentPromotionAddOrGetOperation(ClientId clientId,
                                             List<ContentPromotionSingleObjectRequest> contentToAddOrGet,
                                             ContentPromotionRepository contentPromotionRepository,
                                             ContentPromotionCoreTypeSupportFacade
                                                     contentPromotionCoreTypeSupportFacade) {
        this.clientId = clientId;
        this.contentPromotionRepository = contentPromotionRepository;
        this.contentPromotionCoreTypeSupportFacade = contentPromotionCoreTypeSupportFacade;
        init(contentToAddOrGet);
    }

    private void init(List<ContentPromotionSingleObjectRequest> contentToAddOrGet) {
        contentToAddOrGet.forEach(URLHelper::buildCorrectUrl);
        basicDataList = contentToAddOrGet.stream()
                .map(t -> (ContentPromotionContentBasicData) null)
                .collect(Collectors.toList());
        requestsByIndex = EntryStream.of(contentToAddOrGet).toMap();
        contentIdsByIndex = new HashMap<>();
    }

    /**
     * Непосредственно добавление контента и его валидация
     * По результатам выполнения каждому индексу запросов должен быть сопоставлен subResult у listValidationBuilder
     */
    public ContentPromotionOperationResult prepareAndExecute() {
        if (requestsByIndex.isEmpty()) {
            listValidationBuilder = ListValidationBuilder.of(basicDataList);
            return new ContentPromotionOperationResult(contentIdsByIndex, listValidationBuilder.getResult());
        }
        //Пытаемся получить внешние идентификаторы контента
        externalIdsByIndex = contentPromotionCoreTypeSupportFacade.calcExternalIds(requestsByIndex);

        Set<Integer> basicDataIndexesToFill = EntryStream.of(externalIdsByIndex).nonNullValues().keys().toSet();

        //По индексам с ненулевыми внешними идентификаторами сделаем запрос за самим контентом
        //Сначала в базу
        Set<Integer> indexesWithDbData = fillDataFromDbIfPossible(basicDataIndexesToFill);
        basicDataIndexesToFill.removeAll(indexesWithDbData);

        //Затем во внешний сервис по оставшимся индексам
        Set<Integer> indexesWithExternalData = fillDataFromExternalServiceIfPossible(basicDataIndexesToFill);

        //Сделаем валидация сначала контента, который лежит в базе, затем контента, полученного извне
        listValidationBuilder = ListValidationBuilder.of(basicDataList);
        listValidationBuilder.checkSublistBy(sublist ->
                        contentPromotionCoreTypeSupportFacade.validateBasicDataFromDb(sublist),
                (vr, ind) -> indexesWithDbData.contains(ind));
        listValidationBuilder.checkSublistBy(sublist ->
                        contentPromotionCoreTypeSupportFacade.validateBasicDataFromExternalService(sublist),
                (vr, ind) -> indexesWithExternalData.contains(ind));

        Map<Integer, Boolean> subResultHasAnyErrors = EntryStream.of(listValidationBuilder.getResult().getSubResults())
                .filterKeys(t -> t instanceof PathNode.Index)
                .mapKeys(t -> ((PathNode.Index) t).getIndex())
                .mapValues(ValidationResult::hasAnyErrors)
                .toMap();
        Set<Integer> externalIndexesToInsert = StreamEx.of(indexesWithExternalData)
                .filter(ind -> !subResultHasAnyErrors.getOrDefault(ind, false))
                .toSet();

        //Если контента нет, считаем его ненайденным
        listValidationBuilder.checkEach(fromPredicateOfNullable(Objects::nonNull, contentNotFound()));

        //Запишем в базу полученный извне контент, прошедший валидацию
        Map<Integer, Long> newContentIds = insertExternalDataToDb(externalIndexesToInsert);
        contentIdsByIndex.putAll(newContentIds);
        return new ContentPromotionOperationResult(contentIdsByIndex, listValidationBuilder.getResult());
    }

    /**
     * Попытка заполнить информацию об объектах по данным о продвижении в базе
     *
     * @param indexesToFill индексы, по которым пытаться заполнить
     * @return Набор индексов, по которым заполнили
     */
    private Set<Integer> fillDataFromDbIfPossible(Set<Integer> indexesToFill) {
        List<String> externalIdsToUse = EntryStream.of(externalIdsByIndex)
                .filterKeys(indexesToFill::contains)
                .values()
                .toList();

        Map<String, ContentPromotionContent> contentPromotionContentsByExternalIds =
                contentPromotionRepository.getContentPromotionByExternalIds(clientId, externalIdsToUse);

        Map<Integer, ContentPromotionContent> contentsInDbByIndex = EntryStream.of(externalIdsByIndex)
                .filterKeys(indexesToFill::contains)
                .mapValues(contentPromotionContentsByExternalIds::get)
                .nonNullValues()
                .toMap();

        Map<Integer, ContentPromotionContentBasicData> contentPromotionContentBasicDataByIndex =
                contentPromotionCoreTypeSupportFacade.buildBasicDataFromDbData(contentsInDbByIndex, requestsByIndex);

        contentPromotionContentBasicDataByIndex.forEach((ind, val) -> basicDataList.set(ind, val));
        contentIdsByIndex.putAll(EntryStream.of(contentsInDbByIndex).mapValues(ContentPromotionContent::getId).toMap());
        return contentPromotionContentBasicDataByIndex.keySet();
    }

    /**
     * Попытка заполнить информацию об объектах по данным о продвижении во внешнем сервисе
     *
     * @param indexesToFill индексы, по которым пытаться заполнить
     * @return Набор индексов, по которым заполнили
     */
    private Set<Integer> fillDataFromExternalServiceIfPossible(Set<Integer> indexesToFill) {
        Map<Integer, String> externalIdsToUse = EntryStream.of(externalIdsByIndex)
                .filterKeys(indexesToFill::contains)
                .toMap();
        Map<Integer, ContentPromotionSingleObjectRequest> requestsToUse = EntryStream.of(requestsByIndex)
                .filterKeys(indexesToFill::contains)
                .toMap();
        Map<String, ContentPromotionContentBasicData> basicDataByExternalId = contentPromotionCoreTypeSupportFacade
                .getBasicDataFromExternalService(externalIdsToUse, requestsToUse);

        Map<Integer, ContentPromotionContentBasicData> basicDataByIndex = EntryStream.of(externalIdsByIndex)
                .filterKeys(indexesToFill::contains)
                .mapValues(basicDataByExternalId::get)
                .nonNullValues()
                .toMap();

        basicDataByIndex.forEach((ind, val) -> basicDataList.set(ind, val));
        return basicDataByIndex.keySet();
    }

    /**
     * Добавить в базу данные, полученные от внешнего сервиса
     */
    private Map<Integer, Long> insertExternalDataToDb(Set<Integer> indexesToInsert) {
        Map<String, ContentPromotionContent> contentPromotionContentsByExternalIds = EntryStream.of(basicDataList)
                .filterKeys(indexesToInsert::contains)
                .mapToValue((ind, basicData) -> constructContentPromotionContentToInsert(basicData.getContentType(),
                        basicData, requestsByIndex.get(ind), externalIdsByIndex.get(ind), clientId))
                .mapToKey((ind, content) -> content.getExternalId())
                .toMap((firstContent, secondContent) -> firstContent);
        contentPromotionRepository.insertContentPromotions(clientId, contentPromotionContentsByExternalIds.values());
        return EntryStream.of(externalIdsByIndex)
                .mapValues(contentPromotionContentsByExternalIds::get)
                .nonNullValues()
                .mapValues(ContentPromotionContent::getId)
                .toMap();
    }

    private ContentPromotionContent constructContentPromotionContentToInsert(
            ContentPromotionContentType contentType,
            ContentPromotionContentBasicData basicData,
            ContentPromotionSingleObjectRequest request,
            String externalId,
            ClientId clientId) {
        return new ContentPromotionContent()
                .withUrl(request.getUrl())
                .withClientId(clientId.asLong())
                .withIsInaccessible(false)
                .withType(contentType)
                .withExternalId(externalId)
                .withPreviewUrl(basicData.getPreviewUrl())
                .withMetadata(basicData.getMetadataJson())
                .withMetadataHash(contentPromotionCoreTypeSupportFacade
                        .calcMetadataHash(contentType, basicData.getMetadataJson()));
    }
}
