package ru.yandex.direct.core.entity.contentpromotion;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.contentpromotion.model.CollectionExternalIdHolder;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.libs.collections.CollectionsClient;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpData;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.utils.HashingUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.contentpromotion.type.collection.ContentPromotionCollectionUtils.convertToPreviewUrl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

/**
 * Операция, по получению и добавлении в базу при необходимости продвигаемой коллекции
 * В отличие от ContentPromotionAddOrGetOperation без валидации, добавление по external_id вместо ссылки
 * ссылка(url) берётся из serp_data
 */
public class ContentPromotionCollectionsTruncatedOperation extends AbstractAddOperation<CollectionExternalIdHolder,
        Long> {
    private final ContentPromotionRepository contentPromotionRepository;
    private final CollectionsClient collectionsClient;
    private final EnvironmentType environmentType;
    private ClientId clientId;
    int shard;

    private Map<String, CollectionSerpData> serpDataMap;
    private Map<String, Long> externalIdsInDbValid;

    public ContentPromotionCollectionsTruncatedOperation(Applicability applicability,
                                                         List<String> externalIds,
                                                         ContentPromotionRepository contentPromotionRepository,
                                                         CollectionsClient collectionsClient,
                                                         EnvironmentType environmentType,
                                                         ClientId clientId,
                                                         int shard) {
        super(applicability, StreamEx.of(externalIds)
                .map(t -> new CollectionExternalIdHolder().withExternalId(t)).toList());
        this.contentPromotionRepository = contentPromotionRepository;
        this.collectionsClient = collectionsClient;
        this.environmentType = environmentType;
        this.clientId = clientId;
        this.shard = shard;
    }


    @Override
    protected void onPreValidated(ModelsPreValidatedStep<CollectionExternalIdHolder> modelsPreValidatedStep) {
        List<String> externalIds = StreamEx.of(modelsPreValidatedStep.getModels())
                .map(CollectionExternalIdHolder::getExternalId)
                .toList();
        externalIdsInDbValid = EntryStream.of(contentPromotionRepository
                .getValidContentPromotionByExternalIdsAndType(clientId, externalIds,
                        ContentPromotionContentType.COLLECTION))
                .mapValues(ContentPromotionContent::getId)
                .toMap();
        Set<String> externalIdsToUpdate = new HashSet<>(externalIds);
        externalIdsToUpdate.removeAll(externalIdsInDbValid.keySet());
        serpDataMap = StreamEx.of(externalIdsToUpdate)
                .mapToEntry(collectionsClient::getCollectionSerpData)
                .nonNullValues()
                .toMap();
    }

    @Override
    protected void validate(ValidationResult<List<CollectionExternalIdHolder>, Defect> preValidationResult) {
        new ListValidationBuilder<>(preValidationResult).checkEachBy(t -> validateSingleExtId(t));
    }

    private ValidationResult<CollectionExternalIdHolder, Defect> validateSingleExtId(
            CollectionExternalIdHolder collectionExternalIdHolder) {
        ItemValidationBuilder<CollectionExternalIdHolder, Defect> vb =
                ItemValidationBuilder.of(collectionExternalIdHolder);

        vb.item(collectionExternalIdHolder.getExternalId(), CollectionExternalIdHolder.EXTERNAL_ID.name())
                .check(notNull())
                .check(fromPredicate(extId -> externalIdsInDbValid.containsKey(extId) || serpDataMap.containsKey(extId),
                        invalidValue()));

        return vb.getResult();
    }

    @Override
    protected Map<Integer, Long> execute(Map<Integer, CollectionExternalIdHolder> validModelsMapToApply) {
        Map<Integer, String> validModelsExtIdMap = EntryStream.of(validModelsMapToApply)
                .mapValues(CollectionExternalIdHolder::getExternalId)
                .toMap();
        List<ContentPromotionContent> contentToInsert = EntryStream.of(validModelsExtIdMap)
                .values()
                .distinct()
                .filter(serpDataMap::containsKey)
                .map(t -> constructContentPromotionContentToInsert(serpDataMap.get(t), t, clientId))
                .toList();

        contentPromotionRepository.insertOrUpdateContentPromotionContents(contentToInsert, shard);
        Map<String, Long> idByExtId = EntryStream.of(contentPromotionRepository
                .getContentPromotionByExternalIdsAndType(clientId, new HashSet<>(validModelsExtIdMap.values()),
                        ContentPromotionContentType.COLLECTION))
                .mapValues(ContentPromotionContent::getId)
                .toMap();

        Map<Integer, Long> result = new HashMap<>(EntryStream.of(validModelsExtIdMap)
                .mapValues(externalIdsInDbValid::get)
                .nonNullValues()
                .toMap());
        result.putAll(EntryStream.of(validModelsExtIdMap).mapValues(idByExtId::get).nonNullValues().toMap());
        return result;
    }

    private ContentPromotionContent constructContentPromotionContentToInsert(
            CollectionSerpData serpData,
            String externalId,
            ClientId clientId) {
        return new ContentPromotionContent()
                .withUrl(serpData.getUrl())
                .withClientId(clientId.asLong())
                .withIsInaccessible(false)
                .withType(ContentPromotionContentType.COLLECTION)
                .withExternalId(externalId)
                .withPreviewUrl(convertToPreviewUrl(serpData.getThumbId(), environmentType))
                .withMetadata(serpData.getNormalizedJson())
                .withMetadataHash(HashingUtils.getMd5HalfHashUtf8(serpData.getNormalizedJson()));
    }
}
