package ru.yandex.direct.core.entity.contentpromotion;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.core.entity.contentpromotion.type.ContentPromotionCoreTypeSupportFacade;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.libs.collections.CollectionsClient;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.contentpromotion.validation.converters.ToApiResultConverter.convertToApiValidationResult;
import static ru.yandex.direct.core.entity.contentpromotion.validation.converters.ToWebResultConverter.convertToGetMetaValidationResult;
import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.contentInaccessible;
import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.contentNotFound;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Сервис для получения метаданных продвигаемого контента.
 */
@Service
@ParametersAreNonnullByDefault
public class ContentPromotionService {

    private final ContentPromotionRepository contentPromotionRepository;
    private final ShardHelper shardHelper;
    private final ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade;
    private final CollectionsClient collectionsClient;
    private final EnvironmentType environmentType;

    public ContentPromotionService(ContentPromotionRepository contentPromotionRepository,
                                   ShardHelper shardHelper,
                                   ContentPromotionCoreTypeSupportFacade contentPromotionCoreTypeSupportFacade,
                                   CollectionsClient collectionsClient,
                                   EnvironmentType environmentType) {
        this.contentPromotionRepository = contentPromotionRepository;
        this.shardHelper = shardHelper;
        this.contentPromotionCoreTypeSupportFacade = contentPromotionCoreTypeSupportFacade;
        this.collectionsClient = collectionsClient;
        this.environmentType = environmentType;
    }

    /**
     * Возвращает объект {@link ContentPromotionMetaWithStatus} для заданного контента
     *
     * @param url         ссылка на продвигаемый контент
     * @param requestId   id запроса
     * @param contentType тип контента, по которому идёт запрос
     * @param campaignId  id кампании
     * @param adGroupId   id группы
     * @param clientLogin логин клиента
     * @return объект {@link ContentPromotionMetaWithStatus} для заданного контента
     */
    @Nonnull
    public ContentPromotionMetaWithStatus getMeta(String url,
                                                  String requestId,
                                                  ContentPromotionContentType contentType,
                                                  Long campaignId,
                                                  @Nullable Long adGroupId,
                                                  String clientLogin) {
        ClientId clientId = ClientId.fromLong(shardHelper.getClientIdByLogin(clientLogin));
        ContentPromotionSingleObjectRequest singleObjectRequest = new ContentPromotionSingleObjectRequest()
                .withUrl(url)
                .withAdGroupId(adGroupId)
                .withCampaignId(campaignId)
                .withRequestId(requestId)
                .withClientLogin(clientLogin)
                .withContentType(contentType);
        return getMeta(clientId, singleObjectRequest);
    }

    /**
     * Возвращает объект {@link ContentPromotionMetaWithStatus} для заданного контента
     *
     * @param clientId       идентификатор клиента
     * @param contentRequest запрос с данными по продвигаемому контенту
     * @return объект {@link ContentPromotionMetaWithStatus} для заданного контента
     */
    @Nonnull
    private ContentPromotionMetaWithStatus getMeta(ClientId clientId,
                                                   ContentPromotionSingleObjectRequest contentRequest) {
        ValidationResult<List<DefaultContentPromotionMeta>, Defect> vr =
                addContentPromotionInternal(clientId, singletonList(contentRequest));
        DefaultContentPromotionMeta meta = vr.getValue().get(0);

        ValidationResult.ValidationResultTransformer<Defect> subResultTransformer =
                new ValidationResult.ValidationResultTransformer<>() {
                };
        //Здесь используем то, что vr.getSubResults().get(index(0)) определён, то есть не null
        ValidationResult<DefaultContentPromotionMeta, Defect> singleVr =
                vr.getSubResults().get(index(0)).transformUnchecked(subResultTransformer);
        return new ContentPromotionMetaWithStatus(meta, calcContentStatus(singleVr), singleVr);
    }

    private ContentStatus calcContentStatus(ValidationResult<DefaultContentPromotionMeta, Defect> singleVr) {
        if (singleVr.flattenErrors().stream().anyMatch(err -> err.getDefect().defectId().getCode().equals(
                contentNotFound().defectId().getCode()))) {
            return ContentStatus.NOT_FOUND;
        } else if (singleVr.flattenWarnings().stream().anyMatch(err -> err.getDefect().defectId().getCode().equals(
                contentInaccessible().defectId().getCode()))) {
            return ContentStatus.INACCESSIBLE;
        } else {
            return ContentStatus.OK;
        }
    }

    /**
     * Добавление контента и возврат результата валидации по нему для множественного запроса контента
     * При наличии соответствующего контента в базе вернёт идентификатор из базы
     *
     * @param clientId          идентификатор клиента
     * @param contentToAddOrGet запросы с данными по продвигаемому контенту
     */
    public ValidationResult<List<Long>, Defect> addContentPromotion(
            ClientId clientId, @Nonnull List<ContentPromotionSingleObjectRequest> contentToAddOrGet) {
        var vr = addContentPromotionInternal(clientId, contentToAddOrGet);
        ValidationResult<List<Long>, Defect> vrTransformed = convertToApiValidationResult(vr);
        return vrTransformed;
    }

    /**
     * Метод получения идентификаторов контента по ссылке и, при необходимости,
     * добавления данных о недостающем контенте в базу. Также производит валидацию контента
     */
    public ValidationResult<List<DefaultContentPromotionMeta>, Defect> addContentPromotionInternal(
            ClientId clientId, @Nonnull List<ContentPromotionSingleObjectRequest> contentToAddOrGet) {
        ContentPromotionAddOrGetOperation container = new ContentPromotionAddOrGetOperation(clientId, contentToAddOrGet,
                contentPromotionRepository, contentPromotionCoreTypeSupportFacade);
        var operationResult = container.prepareAndExecute();
        return convertToGetMetaValidationResult(operationResult);
    }


    /**
     * Метод получения идентификаторов контента коллекций по board_id и,
     * При необходимости добавляет данных о недостающем контенте в базу
     */
    public MassResult<Long> addContentPromotionCollectionsTruncated(
            ClientId clientId, int shard, @Nonnull List<String> externalIds) {
        ContentPromotionCollectionsTruncatedOperation operation = new ContentPromotionCollectionsTruncatedOperation(
                Applicability.PARTIAL, externalIds,
                contentPromotionRepository, collectionsClient, environmentType, clientId, shard);
        return operation.prepareAndApply();
    }
}
