package ru.yandex.direct.core.entity.contentpromotion;

import java.net.URISyntaxException;
import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType.VIDEO;
import static ru.yandex.direct.libs.video.VideoClientUtils.isEfirUrl;
import static ru.yandex.direct.libs.video.VideoClientUtils.modifyEfirUrl;

@ParametersAreNonnullByDefault
class URLHelper {

    private static final Logger logger = LoggerFactory.getLogger(URLHelper.class);

    private static final String WWW = "www.";
    private static final String HOST_YOUTUBE = "youtube.com";
    private static final String HOST_YOUTUBE_SHORT = "youtu.be";

    private URLHelper() {
    }

    static void buildCorrectUrl(ContentPromotionSingleObjectRequest contentToAddOrGet) {
        if (contentToAddOrGet.getContentType() != VIDEO || contentToAddOrGet.getUrl() == null) {
            return;
        }
        String str = contentToAddOrGet.getUrl().trim();
        str = str.startsWith("http://") || str.startsWith("https://") ? str : "https://" + str;
        try {
            URIBuilder uriBuilder = new URIBuilder(str);
            String host = uriBuilder.getHost().toLowerCase();
            if (host.endsWith(HOST_YOUTUBE_SHORT)) {
                URLHelper.modifyYoutubeShortLink(uriBuilder);
            } else if (host.endsWith(HOST_YOUTUBE)) {
                URLHelper.modifyYoutubeLink(uriBuilder);
            } else if (isEfirUrl(uriBuilder)) {
                modifyEfirUrl(uriBuilder);
            }
            contentToAddOrGet.withUrl(uriBuilder.toString());
        } catch (URISyntaxException | NullPointerException e) {
            logger.warn("Failed to create correct URL by {}. Exception: {}", contentToAddOrGet.getUrl(), e);
        }
    }

    private static void modifyYoutubeShortLink(URIBuilder uriBuilder) {
        uriBuilder.setHost(WWW + HOST_YOUTUBE);
        if (!uriBuilder.getPath().isEmpty()) {
            String videoKey = Arrays.stream(uriBuilder.getPath().split("/"))
                    .filter(s -> !s.isEmpty())
                    .findFirst()
                    .orElse("");
            uriBuilder.setCustomQuery("v=" + videoKey);
            uriBuilder.setPath("/watch");
        }
    }

    private static void modifyYoutubeLink(URIBuilder uriBuilder) {
        uriBuilder.setHost(WWW + HOST_YOUTUBE);
        String videoKey = uriBuilder.getQueryParams().stream()
                .filter(q -> q.getName().equals("v"))
                .map(NameValuePair::getValue)
                .findFirst()
                .orElse("");
        if (!videoKey.isEmpty()) {
            uriBuilder.setCustomQuery("v=" + videoKey);
        }
    }
}
