package ru.yandex.direct.core.entity.contentpromotion.repository;

import java.time.LocalDateTime;
import java.time.temporal.TemporalAmount;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.types.ULong;
import org.jooq.types.UNumber;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.dbschema.ppc.tables.records.ContentPromotionRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.enums.ContentPromotionType.collection;
import static ru.yandex.direct.dbschema.ppc.tables.ContentPromotion.CONTENT_PROMOTION;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ContentPromotionRepository {
    protected final DslContextProvider dslContextProvider;
    protected final ShardHelper shardHelper;

    private final JooqMapperWithSupplier<ContentPromotionContent> contentPromotionJooqMapper;
    private final Set<Field<?>> contentPromotionFields;

    public ContentPromotionRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;

        contentPromotionJooqMapper = JooqMapperWithSupplierBuilder.builder(ContentPromotionContent::new)
                .map(property(ContentPromotionContent.ID, CONTENT_PROMOTION.ID))
                .map(property(ContentPromotionContent.CLIENT_ID, CONTENT_PROMOTION.CLIENT_ID))
                .map(convertibleProperty(ContentPromotionContent.TYPE, CONTENT_PROMOTION.TYPE,
                        ContentPromotionContentType::fromSource, ContentPromotionContentType::toSource))
                .map(convertibleProperty(ContentPromotionContent.IS_INACCESSIBLE, CONTENT_PROMOTION.IS_INACCESSIBLE,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong))
                .map(property(ContentPromotionContent.URL, CONTENT_PROMOTION.URL))
                .map(property(ContentPromotionContent.EXTERNAL_ID, CONTENT_PROMOTION.EXTERNAL_ID))
                .map(property(ContentPromotionContent.PREVIEW_URL, CONTENT_PROMOTION.PREVIEW_URL))
                .map(property(ContentPromotionContent.METADATA, CONTENT_PROMOTION.METADATA))
                .map(convertibleProperty(ContentPromotionContent.METADATA_HASH, CONTENT_PROMOTION.METADATA_HASH,
                        t -> ifNotNull(t, UNumber::toBigInteger), t -> ifNotNull(t, ULong::valueOf)))
                .map(property(ContentPromotionContent.METADATA_CREATE_TIME, CONTENT_PROMOTION.METADATA_CREATE_TIME))
                .map(property(ContentPromotionContent.METADATA_MODIFY_TIME, CONTENT_PROMOTION.METADATA_MODIFY_TIME))
                .map(property(ContentPromotionContent.METADATA_REFRESH_TIME, CONTENT_PROMOTION.METADATA_REFRESH_TIME))
                .build();

        contentPromotionFields = contentPromotionJooqMapper.getFieldsToRead();
    }

    /**
     * Возвращает объекты {@link ContentPromotionContent} для заданного клиента с заданными id
     *
     * @param clientId            id клиента
     * @param contentPromotionIds id контента
     * @return список объектов {@link ContentPromotionContent}
     */
    public List<ContentPromotionContent> getContentPromotion(ClientId clientId, List<Long> contentPromotionIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getContentPromotion(shard, clientId, contentPromotionIds);
    }

    private List<ContentPromotionContent> getContentPromotion(int shard, ClientId clientId,
                                                              List<Long> contentPromotionIds) {
        if (contentPromotionIds.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.ID.in(contentPromotionIds))
                .and(CONTENT_PROMOTION.CLIENT_ID.eq(clientId.asLong()))
                .fetch(contentPromotionJooqMapper::fromDb);
    }

    public Map<Long, ContentPromotionContent> getContentPromotionsByContentIds(int shard,
                                                                               Collection<Long> contentPromotionIds) {
        if (contentPromotionIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.ID.in(contentPromotionIds))
                .fetchMap(CONTENT_PROMOTION.ID, contentPromotionJooqMapper::fromDb);
    }

    public List<ContentPromotionContent> getContentPromotionsByContentIdsAndTypes(
            ClientId clientId,
            @Nonnull Collection<Long> contentPromotionIds,
            @Nonnull Collection<ContentPromotionContentType> types) {
        checkArgument(!contentPromotionIds.isEmpty() || !types.isEmpty(), "Selection criteria has to be valid");
        List<Condition> conditions = new ArrayList<>();
        if (!contentPromotionIds.isEmpty()) {
            conditions.add(CONTENT_PROMOTION.ID.in(contentPromotionIds));
        }
        conditions.add(CONTENT_PROMOTION.CLIENT_ID.eq(clientId.asLong()));
        if (!types.isEmpty()) {
            conditions.add(CONTENT_PROMOTION.TYPE.in(mapList(types, ContentPromotionContentType::toSource)));
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(conditions)
                .fetch(contentPromotionJooqMapper::fromDb);
    }

    /**
     * Вставляет объект {@link ContentPromotionContent} в таблицу.
     *
     * @param clientId                id клиента
     * @param contentPromotionContent вставляемый объект
     * @return id продвигаемого контента
     */
    public Long insertContentPromotion(ClientId clientId, ContentPromotionContent contentPromotionContent) {
        return insertContentPromotions(clientId, List.of(contentPromotionContent)).get(0);
    }

    /**
     * Обновляет объекты {@link ContentPromotionContent} в таблице или добавляет при отсутствии
     */
    public void insertOrUpdateContentPromotionContents(List<ContentPromotionContent> contents,
                                                       int shard) {
        if (contents.isEmpty()) {
            return;
        }
        //выделяю с запасом,  потому что почему бы и нет
        List<Long> idsToInsert = shardHelper.generateContentPromotionIds(contents.size());
        EntryStream.zip(contents, idsToInsert).forKeyValue(ContentPromotionContent::setId);
        LocalDateTime now = LocalDateTime.now();
        contents.forEach(c -> c
                .withMetadataCreateTime(nvl(c.getMetadataCreateTime(), now))
                .withMetadataModifyTime(nvl(c.getMetadataModifyTime(), now))
                .withMetadataRefreshTime(nvl(c.getMetadataRefreshTime(), now)));

        new InsertHelper<>(dslContextProvider.ppc(shard), CONTENT_PROMOTION)
                .addAll(contentPromotionJooqMapper, contents)
                .onDuplicateKeyUpdate()
                //id не обновляю
                .set(CONTENT_PROMOTION.IS_INACCESSIBLE, MySQLDSL.values(CONTENT_PROMOTION.IS_INACCESSIBLE))
                .set(CONTENT_PROMOTION.PREVIEW_URL, MySQLDSL.values(CONTENT_PROMOTION.PREVIEW_URL))
                .set(CONTENT_PROMOTION.METADATA, MySQLDSL.values(CONTENT_PROMOTION.METADATA))
                .set(CONTENT_PROMOTION.METADATA_HASH, MySQLDSL.values(CONTENT_PROMOTION.METADATA_HASH))
                .set(CONTENT_PROMOTION.METADATA_MODIFY_TIME, MySQLDSL.values(CONTENT_PROMOTION.METADATA_MODIFY_TIME))
                .set(CONTENT_PROMOTION.METADATA_REFRESH_TIME, MySQLDSL.values(CONTENT_PROMOTION.METADATA_REFRESH_TIME))
                .executeIfRecordsAdded();
    }

    /**
     * Вставляет набор объектов {@link ContentPromotionContent} в таблицу.
     *
     * @param clientId                 id клиента
     * @param contentPromotionContents вставляемые объекты
     * @return список id продвигаемого контента
     */
    public List<Long> insertContentPromotions(ClientId clientId,
                                              Collection<ContentPromotionContent> contentPromotionContents) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Long> idsToInsert = shardHelper.generateContentPromotionIds(contentPromotionContents.size());

        Iterator<Long> iterator = idsToInsert.iterator();
        contentPromotionContents.forEach(content -> {
            LocalDateTime now = LocalDateTime.now();
            content.withId(iterator.next())
                    .withClientId(clientId.asLong())
                    .withMetadataCreateTime(nvl(content.getMetadataCreateTime(), now))
                    .withMetadataModifyTime(nvl(content.getMetadataModifyTime(), now))
                    .withMetadataRefreshTime(nvl(content.getMetadataRefreshTime(), now));
        });

        new InsertHelper<>(dslContextProvider.ppc(shard), CONTENT_PROMOTION)
                .addAll(contentPromotionJooqMapper, contentPromotionContents)
                .executeIfRecordsAdded();

        return idsToInsert;
    }

    /**
     * Обновляет объекты {@link ContentPromotionContent} в таблице
     *
     * @param dsl     контекст базы данных
     * @param changes изменения, которые нужно внести, в виде {@link AppliedChanges}
     */
    public void updateContentPromotionContents(DSLContext dsl,
                                               Collection<AppliedChanges<ContentPromotionContent>> changes) {
        JooqUpdateBuilder<ContentPromotionRecord, ContentPromotionContent> ub =
                new JooqUpdateBuilder<>(CONTENT_PROMOTION.ID, changes);
        ub.processProperty(ContentPromotionContent.PREVIEW_URL, CONTENT_PROMOTION.PREVIEW_URL);
        ub.processProperty(ContentPromotionContent.METADATA, CONTENT_PROMOTION.METADATA);
        ub.processProperty(ContentPromotionContent.METADATA_HASH, CONTENT_PROMOTION.METADATA_HASH, ULong::valueOf);
        ub.processProperty(ContentPromotionContent.METADATA_MODIFY_TIME, CONTENT_PROMOTION.METADATA_MODIFY_TIME);
        ub.processProperty(ContentPromotionContent.METADATA_REFRESH_TIME, CONTENT_PROMOTION.METADATA_REFRESH_TIME);
        ub.processProperty(ContentPromotionContent.IS_INACCESSIBLE, CONTENT_PROMOTION.IS_INACCESSIBLE,
                RepositoryUtils::booleanToLong);

        dsl.update(CONTENT_PROMOTION)
                .set(ub.getValues())
                .where(CONTENT_PROMOTION.ID.in(ub.getChangedIds()))
                .execute();
    }

    /**
     * Обновляет статус доступности коллекции с заданным external id.
     *
     * @param shard          шард
     * @param externalId     внешний id
     * @param isInaccessible новое значение доступности
     */
    public void updateCollectionAccessibilityStatus(int shard, String externalId, boolean isInaccessible) {
        dslContextProvider.ppc(shard)
                .update(CONTENT_PROMOTION)
                .set(CONTENT_PROMOTION.IS_INACCESSIBLE, booleanToLong(isInaccessible))
                .where(CONTENT_PROMOTION.EXTERNAL_ID.eq(externalId))
                .and(CONTENT_PROMOTION.TYPE.eq(collection))
                .execute();
    }

    /**
     * Получает объекты {@link ContentPromotionContent}, которые надо обновить - которые уже минимум gracePeriod не
     * обновлялись
     *
     * @param shard                       обрабатываемый шард
     * @param contentPromotionContentType тип продвигаемого контента
     * @param count                       максимальное количество контента, обрабатываемое за раз
     * @param gracePeriod                 период, в течение которого контент можно не проверять/не обновлять
     * @return список контента, нуждающегося в обновлении
     */
    public List<ContentPromotionContent> getContentPromotionContentToUpdate(int shard,
                                                                            ContentPromotionContentType
                                                                                    contentPromotionContentType,
                                                                            int count,
                                                                            TemporalAmount gracePeriod) {
        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.METADATA_REFRESH_TIME.lessThan(
                        LocalDateTime.now().minus(gracePeriod)))
                .and(CONTENT_PROMOTION.TYPE.eq(ContentPromotionContentType.toSource(contentPromotionContentType)))
                .orderBy(CONTENT_PROMOTION.METADATA_REFRESH_TIME.asc())
                .limit(count)
                .fetch(contentPromotionJooqMapper::fromDb);
    }

    /**
     * Возвращает список объектов {@link ContentPromotionContent} с заданным внешним идентификатором.
     *
     * @param clientId    id клиента
     * @param externalIds список внешних идентификаторов контента в вертикали
     * @return список объектов {@link ContentPromotionContent}
     */
    public Map<String, ContentPromotionContent> getContentPromotionByExternalIds(ClientId clientId,
                                                                                 Collection<String> externalIds) {
        if (externalIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.CLIENT_ID.eq(clientId.asLong()))
                .and(CONTENT_PROMOTION.EXTERNAL_ID.in(externalIds))
                .fetchMap(CONTENT_PROMOTION.EXTERNAL_ID, contentPromotionJooqMapper::fromDb);
    }

    /**
     * Возвращает объекты {@link ContentPromotionContent} (коллекции) из заданного шарда с заданным external id.
     *
     * @param shard      шард
     * @param externalId внешний id контента
     * @return список объектов {@link ContentPromotionContent} (коллекций)
     */
    public List<ContentPromotionContent> getCollections(int shard, String externalId) {
        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.EXTERNAL_ID.eq(externalId))
                .and(CONTENT_PROMOTION.TYPE.eq(collection))
                .fetch(contentPromotionJooqMapper::fromDb);
    }

    /**
     * Возвращает список объектов {@link ContentPromotionContent} с заданным внешним идентификатором и типом продвижения
     *
     * @param clientId    id клиента
     * @param externalIds список внешних идентификаторов контента в вертикали
     * @param type        тип продвигаемого контента
     * @return список объектов {@link ContentPromotionContent}
     */
    public Map<String, ContentPromotionContent> getContentPromotionByExternalIdsAndType(ClientId clientId,
                                                                                        Collection<String>
                                                                                                externalIds,
                                                                                        ContentPromotionContentType
                                                                                                type) {
        if (externalIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.CLIENT_ID.eq(clientId.asLong()))
                .and(CONTENT_PROMOTION.EXTERNAL_ID.in(externalIds))
                .and(CONTENT_PROMOTION.TYPE.eq(ContentPromotionContentType.toSource(type)))
                .fetchMap(CONTENT_PROMOTION.EXTERNAL_ID, contentPromotionJooqMapper::fromDb);
    }

    /**
     * Возвращает список объектов {@link ContentPromotionContent} с заданным внешним идентификатором.
     *
     * @param clientId    id клиента
     * @param externalIds список внешних идентификаторов контента в вертикали
     * @param type        тип продвижения
     * @return список объектов {@link ContentPromotionContent}
     */
    public Map<String, ContentPromotionContent> getValidContentPromotionByExternalIdsAndType(
            ClientId clientId,
            List<String> externalIds,
            @Nonnull ContentPromotionContentType type) {
        if (externalIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return dslContextProvider.ppc(shard)
                .select(contentPromotionFields)
                .from(CONTENT_PROMOTION)
                .where(CONTENT_PROMOTION.CLIENT_ID.eq(clientId.asLong()))
                .and(CONTENT_PROMOTION.EXTERNAL_ID.in(externalIds))
                .and(CONTENT_PROMOTION.TYPE.eq(ContentPromotionContentType.toSource(type)))
                .and(CONTENT_PROMOTION.IS_INACCESSIBLE.eq(0L))
                .fetchMap(CONTENT_PROMOTION.EXTERNAL_ID, contentPromotionJooqMapper::fromDb);
    }
}
