package ru.yandex.direct.core.entity.contentpromotion.type;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionContentBasicData;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionSingleObjectRequest;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.type.collection.ContentPromotionCollectionCoreTypeSupport;
import ru.yandex.direct.core.entity.contentpromotion.type.eda.ContentPromotionEdaCoreTypeSupport;
import ru.yandex.direct.core.entity.contentpromotion.type.service.ContentPromotionServiceCoreTypeSupport;
import ru.yandex.direct.core.entity.contentpromotion.type.video.ContentPromotionVideoCoreTypeSupport;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Диспетчер, помогающий сопоставить продвигаемый контент (таблицы ppc.content_promotion и ppc.content_promotion_video)
 * и группы объявлений, ему соответствующие (ppc.adgroup_content_promotion) типоспецифичным обработчикам
 *
 * @see AbstractContentPromotionCoreTypeSupport
 */
@Service
public class ContentPromotionCoreTypeSupportFacade {
    private final Map<ContentPromotionContentType, AbstractContentPromotionCoreTypeSupport> supportMap;

    @Autowired
    public ContentPromotionCoreTypeSupportFacade(
            ContentPromotionCollectionCoreTypeSupport contentPromotionCollectionCoreTypeSupport,
            ContentPromotionVideoCoreTypeSupport contentPromotionVideoCoreTypeSupport,
            ContentPromotionServiceCoreTypeSupport contentPromotionServiceCoreTypeSupport,
            ContentPromotionEdaCoreTypeSupport contentPromotionEdaCoreTypeSupport) {
        this(StreamEx.of(contentPromotionCollectionCoreTypeSupport,
                contentPromotionVideoCoreTypeSupport,
                contentPromotionServiceCoreTypeSupport,
                contentPromotionEdaCoreTypeSupport)
                .collect(toMap(AbstractContentPromotionCoreTypeSupport::getType, identity()))
        );
    }

    ContentPromotionCoreTypeSupportFacade(
            Map<ContentPromotionContentType, AbstractContentPromotionCoreTypeSupport> supportMap) {
        this.supportMap = supportMap;
    }

    /**
     * Подсчитать хэш от метаданных для хранения в базе в ppc.content_promotion.metadata
     * в зависимости от типа контента
     *
     * @param contentType  тип контента
     * @param metadataJson метаданные
     */
    public BigInteger calcMetadataHash(ContentPromotionContentType contentType, String metadataJson) {
        return supportMap.get(contentType).calcMetadataHash(metadataJson);
    }

    /**
     * Построить экземпляры моделей ContentPromotionContentBasicData по данным в базе
     *
     * @param contentByIndex мапа индексов в контент в базе
     * @param requests       мапа индексов в запросы на добавление контента
     */
    @Nonnull
    public Map<Integer, ContentPromotionContentBasicData> buildBasicDataFromDbData(
            Map<Integer, ContentPromotionContent> contentByIndex,
            Map<Integer, ContentPromotionSingleObjectRequest> requests) {
        Map<Integer, ContentPromotionContentBasicData> result = new HashMap<>();
        supportMap.forEach((type, typeSupport) -> {
            Map<Integer, ContentPromotionSingleObjectRequest> requestsOfGivenType = EntryStream.of(requests)
                    .filterValues(val -> val.getContentType() == type)
                    .toMap();
            Map<Integer, ContentPromotionContent> contentOfGivenType = EntryStream.of(contentByIndex)
                    .filterKeys(requestsOfGivenType::containsKey)
                    .filterValues(val -> val.getType() == type)
                    .toMap();
            result.putAll(typeSupport.buildBasicDataFromDbData(contentOfGivenType, requestsOfGivenType));
        });
        return result;
    }

    /**
     * Подсчитать внешние идентификаторы контента для переданных запросов
     *
     * @param requestsByIndex мапа индексов в запросы на добавление контента
     */
    @Nonnull
    public Map<Integer, String> calcExternalIds(Map<Integer, ContentPromotionSingleObjectRequest> requestsByIndex) {
        Map<Integer, String> result = new HashMap<>();
        supportMap.forEach((type, typeSupport) -> {
            Map<Integer, ContentPromotionSingleObjectRequest> requestsOfGivenType = EntryStream.of(requestsByIndex)
                    .filterValues(val -> val.getContentType() == type)
                    .toMap();
            result.putAll(typeSupport.calcExternalIds(requestsOfGivenType));
        });
        return result;
    }

    /**
     * Сходить во внешний сервис с внешними идентификаторами контента и информацией
     * из исходного запроса (ссылкой на контент) при необходимости
     *
     * @param externalIds мапа индексов во внешние идентификаторы
     * @param requests    мапа индексов в запрос на добавление контента
     */
    public Map<String, ContentPromotionContentBasicData> getBasicDataFromExternalService(
            Map<Integer, String> externalIds, Map<Integer, ContentPromotionSingleObjectRequest> requests) {
        Map<String, ContentPromotionContentBasicData> result = new HashMap<>();
        supportMap.forEach((type, typeSupport) -> {
            Map<Integer, ContentPromotionSingleObjectRequest> requestsOfGivenType = EntryStream.of(requests)
                    .filterValues(val -> val.getContentType() == type)
                    .toMap();
            Map<Integer, String> externalIdsOfGivenType = EntryStream.of(externalIds)
                    .filterKeys(requestsOfGivenType::containsKey)
                    .toMap();
            result.putAll(typeSupport.getBasicDataFromExternalRequest(externalIdsOfGivenType, requestsOfGivenType));
        });
        return result;
    }

    /**
     * Провалидировать данные по контенту, полученные походом во внешний сервис
     *
     * @param contentPromotionContentBasicData список полученного во внешнем сервисе контента
     * @return результат валидации контента
     */
    public ValidationResult<List<ContentPromotionContentBasicData>, Defect> validateBasicDataFromExternalService(
            List<ContentPromotionContentBasicData> contentPromotionContentBasicData) {
        ListValidationBuilder<ContentPromotionContentBasicData, Defect> lvb = ListValidationBuilder.of(
                contentPromotionContentBasicData, Defect.class);
        supportMap.forEach((type, typeSupport) ->
                lvb.checkSublistBy(typeSupport::validateBasicDataFromExternalService,
                        When.isValidAnd(When.valueIs(g -> g.getContentType() == type)))
        );
        return lvb.getResult();
    }

    /**
     * Провалидировать данные по контенту, полученные походом в базу
     *
     * @param contentPromotionContentBasicData список полученного из базы контента
     * @return результат валидации контента
     */
    public ValidationResult<List<ContentPromotionContentBasicData>, Defect> validateBasicDataFromDb(
            List<ContentPromotionContentBasicData> contentPromotionContentBasicData) {
        ListValidationBuilder<ContentPromotionContentBasicData, Defect> lvb = ListValidationBuilder.of(
                contentPromotionContentBasicData, Defect.class);
        supportMap.forEach((type, typeSupport) ->
                lvb.checkSublistBy(typeSupport::validateBasicDataFromDb,
                        When.isValidAnd(When.valueIs(g -> g.getContentType() == type)))
        );
        return lvb.getResult();
    }
}
