package ru.yandex.direct.core.entity.contentpromotion.type;

import java.math.BigInteger;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionContentBasicData;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionSingleObjectRequest;
import ru.yandex.direct.core.entity.contentpromotion.DefaultContentPromotionMeta;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.utils.HashingUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.contentInaccessible;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public abstract class DefaultContentPromotionCoreTypeSupport implements AbstractContentPromotionCoreTypeSupport {

    @Override
    public BigInteger calcMetadataHash(String metadataJson) {
        return HashingUtils.getMd5HalfHashUtf8(metadataJson);
    }

    @Override
    public Map<Integer, String> calcExternalIds(Map<Integer, ContentPromotionSingleObjectRequest> requests) {
        return EntryStream.of(requests)
                .mapValues(t -> calcSingleExternalId(t.getUrl()))
                .nonNullValues()
                .toMap();
    }

    @Nullable
    protected String calcSingleExternalId(String url) {
        return String.valueOf(HashingUtils.getMd5HalfHashUtf8(url));
    }

    @Override
    public Map<String, ContentPromotionContentBasicData> getBasicDataFromExternalRequest(
            Map<Integer, String> externalIdsByIndex,
            Map<Integer, ContentPromotionSingleObjectRequest> requestsByIndex) {
        requestsByIndex.keySet().forEach(ind -> checkState(externalIdsByIndex.get(ind) != null,
                "Only requests with nonnull external ids have to be present here"));
        return EntryStream.of(requestsByIndex)
                .mapKeys(externalIdsByIndex::get)
                .mapToValue((externalId, content) -> getSingleBasicDataFromExternalRequest(externalId, content))
                .nonNullValues()
                .toMap((basicDataFirst, basicDataSecond) -> basicDataFirst);
    }

    @Nullable
    protected ContentPromotionContentBasicData getSingleBasicDataFromExternalRequest(
            String externalId,
            ContentPromotionSingleObjectRequest request) {
        return null;
    }

    @Override
    public Map<Integer, ContentPromotionContentBasicData> buildBasicDataFromDbData(
            Map<Integer, ContentPromotionContent> contentByIndex,
            Map<Integer, ContentPromotionSingleObjectRequest> requests) {
        return EntryStream.of(contentByIndex)
                .mapToValue((ind, content) -> buildSingleBasicDataFromDbData(content, requests.get(ind)))
                .toMap();
    }

    @Nonnull
    protected ContentPromotionContentBasicData buildSingleBasicDataFromDbData(
            @Nonnull ContentPromotionContent content,
            @Nonnull ContentPromotionSingleObjectRequest request) {
        return new ContentPromotionContentBasicData() {
            @Override
            public String getMetadataJson() {
                return content.getMetadata();
            }

            @Override
            public String getPreviewUrl() {
                return content.getPreviewUrl();
            }

            @Override
            public String getUrl() {
                return content.getUrl();
            }

            @Nonnull
            @Override
            public DefaultContentPromotionMeta toWebResponse(Long contentId) {
                return new DefaultContentPromotionMeta()
                        .withContentId(contentId)
                        .withPreviewUrl(content.getPreviewUrl())
                        .withUrl(content.getUrl());
            }

            @Override
            public boolean isAccessible() {
                return !content.getIsInaccessible();
            }

            @Override
            public ContentPromotionContentType getContentType() {
                return getType();
            }
        };
    }

    @Override
    public <T extends ContentPromotionContentBasicData> ValidationResult<List<T>, Defect>
    validateBasicDataFromExternalService(List<T> contentPromotionContentBasicData) {
        return ListValidationBuilder.of(contentPromotionContentBasicData, Defect.class)
                .checkEachBy(t -> validateSingleBasicDataFromExternalService(t))
                .getResult();
    }

    protected <T extends ContentPromotionContentBasicData> ValidationResult<T,
            Defect> validateSingleBasicDataFromExternalService(T basicData) {
        return ValidationResult.success(basicData);
    }

    @Override
    public <T extends ContentPromotionContentBasicData> ValidationResult<List<T>, Defect> validateBasicDataFromDb(
            List<T> contentPromotionContentBasicData) {
        return ListValidationBuilder.of(contentPromotionContentBasicData, Defect.class)
                .checkEachBy(t -> validateSingleBasicDataFromDb(t))
                .getResult();
    }

    protected <T extends ContentPromotionContentBasicData> ValidationResult<T,
            Defect> validateSingleBasicDataFromDb(T basicData) {
        return ItemValidationBuilder.of(basicData, Defect.class)
                .weakCheck(fromPredicate(t -> t.isAccessible(), contentInaccessible()))
                .getResult();
    }
}
