package ru.yandex.direct.core.entity.contentpromotion.type.collection;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionContentBasicData;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionSingleObjectRequest;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.type.DefaultContentPromotionCoreTypeSupport;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.libs.collections.CollectionsClient;
import ru.yandex.direct.libs.collections.model.serpdata.CollectionSerpData;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.contentpromotion.type.collection.ContentPromotionCollectionUtils.convertToPreviewUrl;
import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.contentInaccessible;
import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.notEnoughCardsInExistingCollection;
import static ru.yandex.direct.core.entity.contentpromotion.validation.defects.ContentPromotionDefects.notEnoughCardsInNewCollection;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class ContentPromotionCollectionCoreTypeSupport extends DefaultContentPromotionCoreTypeSupport {
    private static final Integer MINIMAL_CARD_COUNT_VALUE = 10;
    private final CollectionsClient collectionsClient;
    private final EnvironmentType environmentType;

    public ContentPromotionCollectionCoreTypeSupport(EnvironmentType environmentType,
                                                     CollectionsClient collectionsClient) {
        this.environmentType = environmentType;
        this.collectionsClient = collectionsClient;
    }

    @Override
    public ContentPromotionContentType getType() {
        return ContentPromotionContentType.COLLECTION;
    }

    @Override
    public String calcSingleExternalId(String url) {
        return collectionsClient.getCollectionId(url);
    }

    @Override
    public ContentPromotionContentBasicData buildSingleBasicDataFromDbData(
            @Nonnull ContentPromotionContent content,
            @Nonnull ContentPromotionSingleObjectRequest request) {
        CollectionSerpData collectionSerpData = CollectionSerpData.fromJson(content.getMetadata());
        return new ContentPromotionContentCollectionData(content.getMetadata(), content.getPreviewUrl(),
                content.getUrl(), !content.getIsInaccessible(), collectionSerpData.getCardsCount());
    }

    @Override
    public ContentPromotionContentBasicData getSingleBasicDataFromExternalRequest(String externalId,
                                                                                  ContentPromotionSingleObjectRequest request) {
        CollectionSerpData collectionSerpData = collectionsClient.getCollectionSerpData(externalId);
        if (collectionSerpData == null) {
            return null;
        }
        String metadataJson = collectionSerpData.getNormalizedJson();
        String thumbIdPreviewUrl = convertToPreviewUrl(collectionSerpData.getThumbId(), environmentType);
        return new ContentPromotionContentCollectionData(metadataJson, thumbIdPreviewUrl, request.getUrl(),
                collectionSerpData.getCardsCount());
    }


    @Override
    public ValidationResult<ContentPromotionContentCollectionData, Defect> validateSingleBasicDataFromExternalService(
            ContentPromotionContentBasicData contentPromotionContentBasicData) {
        ItemValidationBuilder<ContentPromotionContentCollectionData, Defect> ivb =
                ItemValidationBuilder.of((ContentPromotionContentCollectionData) contentPromotionContentBasicData);
        ivb.check(Constraint.fromPredicate(data -> data.getCardsCount() >= MINIMAL_CARD_COUNT_VALUE,
                notEnoughCardsInNewCollection()), When.isValid());
        return ivb.getResult();
    }

    @SuppressWarnings("checkstyle:MethodParamPad")
    //Пока не используется
    private ValidationResult<ContentPromotionContentCollectionData, Defect> validateBasicDataFromDbInternal(
            ContentPromotionContentBasicData contentPromotionContentBasicData) {
        ItemValidationBuilder<ContentPromotionContentCollectionData, Defect> ivb =
                ItemValidationBuilder.of((ContentPromotionContentCollectionData) contentPromotionContentBasicData);
        ivb.weakCheck(fromPredicate(t-> t.isAccessible(), contentInaccessible()));
        ivb.weakCheck(Constraint.fromPredicate(data -> data.getCardsCount() >= MINIMAL_CARD_COUNT_VALUE,
                notEnoughCardsInExistingCollection()), When.isValid());
        return ivb.getResult();
    }
}
