package ru.yandex.direct.core.entity.contentpromotion.type.video;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionContentBasicData;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionSingleObjectRequest;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.type.DefaultContentPromotionCoreTypeSupport;
import ru.yandex.direct.libs.video.VideoClient;
import ru.yandex.direct.libs.video.VideoClientUtils;
import ru.yandex.direct.libs.video.model.VideoBanner;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Component
@ParametersAreNonnullByDefault
public class ContentPromotionVideoCoreTypeSupport extends DefaultContentPromotionCoreTypeSupport {
    private static final String PROTOCOL_PREFIX = "https:";

    private final VideoClient videoClient;

    public ContentPromotionVideoCoreTypeSupport(VideoClient videoClient) {
        this.videoClient = videoClient;
    }

    @Override
    public ContentPromotionContentType getType() {
        return ContentPromotionContentType.VIDEO;
    }

    @Override
    public ContentPromotionContentBasicData buildSingleBasicDataFromDbData(
            @Nonnull ContentPromotionContent content,
            @Nonnull ContentPromotionSingleObjectRequest request) {
        VideoBanner videoBanner = fromJson(content.getMetadata(), VideoBanner.class);
        checkState(content.getUrl().equals(request.getUrl()),
                "md5 hash in content_promotion_video url collision " + content.getUrl() + " " +
                        request.getUrl());
        return new ContentPromotionContentVideoData(content.getMetadata(), content.getPreviewUrl(), content.getUrl(),
                !content.getIsInaccessible(), videoBanner.getTitle(),
                StreamEx.of(videoBanner.getPassage()).joining("\n"));
    }

    @Override
    public ContentPromotionContentBasicData getSingleBasicDataFromExternalRequest(String externalId,
                                                                                  ContentPromotionSingleObjectRequest
                                                                                          request) {
        List<VideoBanner> videoBanners = videoClient.getMeta(singletonList(request.getUrl()), request.getRequestId(),
                request.getCampaignId(), request.getAdGroupId(), request.getClientLogin());
        Optional<VideoBanner> videoBannerOpt = StreamEx.of(videoBanners)
                .nonNull()
                .filter(vb -> vb.getUrl() != null)
                .filter(vb -> request.getUrl().endsWith(VideoClientUtils.getUniformUrl(vb)))
                .findFirst();
        if (videoBannerOpt.isEmpty()) {
            return null;
        }
        VideoBanner videoBanner = videoBannerOpt.get();
        String videoBannerJson = JsonUtils.toJson(videoBanner);
        String previewUrl = PROTOCOL_PREFIX + videoBanner.getThmbHref();
        return new ContentPromotionContentVideoData(videoBannerJson, previewUrl, request.getUrl(),
                videoBanner.getTitle(), StreamEx.of(videoBanner.getPassage()).joining("\n"));
    }
}
