package ru.yandex.direct.core.entity.contentpromotion.validation.converters;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionContentBasicData;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionOperationResult;
import ru.yandex.direct.core.entity.contentpromotion.DefaultContentPromotionMeta;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Конвертация ядрового результата добавления контента в адаптированный под веб
 * Проставляет айдишники возвращаемым моделям в соответствии с Map<Integer, Long> contentIds
 * Конвертирует внутри себя экземпляры ContentPromotionContentBasicData в экземпляры DefaultContentPromotionMeta,
 * вызывая ContentPromotionContentBasicData::toWebResponse поэлементно
 */
public class ToWebResultConverter {
    public static ValidationResult<List<DefaultContentPromotionMeta>, Defect> convertToGetMetaValidationResult(
            ContentPromotionOperationResult operationResult) {
        return operationResult.getValidationResult().transformUnchecked(
                new ToMetaResponseTransformer(operationResult.getContentIdsByIndex()));
    }

    private static class ToMetaResponseTransformer implements ValidationResult.ValidationResultTransformer<Defect> {
        Map<Integer, Long> contentIds;

        ToMetaResponseTransformer(Map<Integer, Long> contentIds) {
            this.contentIds = contentIds;
        }

        /**
         * Трансформирует значение внутри ValidationResult в зависимости от пути
         * Для пустого пути конвертируем весь список oldValue поэлементно
         * Для пути длины один, соответствующего индексу, конвертируем один элемент ContentPromotionContentBasicData
         * в DefaultContentPromotionMeta
         */
        @SuppressWarnings("unchecked")
        @Override
        public <OV> Object transformValue(Path path, @Nullable OV oldValue) {
            if (path.isEmpty()) {
                checkState(oldValue instanceof List, "value has to be instance of List");
                List<ContentPromotionContentBasicData> castedList = (List<ContentPromotionContentBasicData>) oldValue;
                return EntryStream.of(castedList)
                        .mapKeyValue((ind, val) -> constructContentPromotionMetaItem(contentIds.get(ind), val))
                        .toList();
            } else if (path.getNodes().size() == 1 && path.getNodes().get(0) instanceof PathNode.Index &&
                    oldValue instanceof ContentPromotionContentBasicData) {
                Integer ind = ((PathNode.Index) path.getNodes().get(0)).getIndex();
                return constructContentPromotionMetaItem(contentIds.get(ind),
                        (ContentPromotionContentBasicData) oldValue);
            } else {
                return oldValue;
            }
        }

        /**
         * Трансформация экземпляра ContentPromotionContentBasicData в DefaultContentPromotionMeta
         * Проставляет полученной модели переданный contentId в качестве идентификатора
         */
        private DefaultContentPromotionMeta constructContentPromotionMetaItem(
                @Nullable Long contentId, @Nullable ContentPromotionContentBasicData basicData) {
            // Для корректной отдачи ошибок на фронт нужно вернуть модельку,
            // поэтому возвращаем модельку без id в случае ошибки
            return ifNotNull(basicData, t -> t.toWebResponse(contentId));
        }
    }
}
