package ru.yandex.direct.core.entity.conversionsource.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.conversionsource.model.ConversionAction
import ru.yandex.direct.core.entity.conversionsource.model.ConversionActionValue
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSourceSettings
import ru.yandex.direct.core.entity.conversionsource.model.Destination
import ru.yandex.direct.core.entity.conversionsource.model.MetrikaGoalSelection
import ru.yandex.direct.core.entity.conversionsource.model.ProcessingInfo
import ru.yandex.direct.core.entity.conversionsource.model.ProcessingStatus
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceTypeCode
import ru.yandex.direct.core.entity.goal.repository.MetrikaConversionAdGoalsRepository
import ru.yandex.direct.core.entity.goal.repository.MetrikaCountersRepository
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsConversionService
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter
import ru.yandex.direct.metrika.client.model.response.CounterGoal
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.utils.mapToSet
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.math.BigDecimal

@Service
class ConversionCenterMetrikaGoalsService(
    private val metrikaClient: MetrikaClient,
    private val conversionSourceService: ConversionSourceService,
    private val rbacService: RbacService,
    private val metrikaConversionAdGoalsRepository: MetrikaConversionAdGoalsRepository,
    private val metrikaCountersRepository: MetrikaCountersRepository,
) {
    fun addCampaignGoalsToConversionCenter(clientId: ClientId, campaigns: Collection<BaseCampaign>, useMetrikaClient: Boolean) {
        val operatorUid = rbacService.getChiefByClientId(clientId)
        val goalIds = getGoalIdsFromCampaigns(campaigns)
        val counterIds = getCounterIdsFromCampaigns(campaigns).mapToSet { it.toLong() }

        updateMetrikaGoalsSelection(clientId, operatorUid, goalIds, counterIds, null, useMetrikaClient)
    }

    fun updateMetrikaGoalsSelection(clientId: ClientId, operatorUid: Long, selections: List<MetrikaGoalSelection>): ValidationResult<*, Defect<*>> {
        // проверка уникальности выбранных целей
        check(selections.size == selections.mapToSet { it.goalId!! }.size)

        val goalIds = selections.mapToSet { it.goalId!! }
        val counterIds = selections.mapToSet { it.metrikaCounterId!! }

        return updateMetrikaGoalsSelection(clientId, operatorUid, goalIds, counterIds, selections, true)
    }

    private fun updateMetrikaGoalsSelection(clientId: ClientId, operatorUid: Long, goalIds: Set<Long>,
                                                counterIds: Set<Long>, selections: List<MetrikaGoalSelection>?,
                                                useMetrikaClient: Boolean): ValidationResult<*, Defect<*>> {
        // достаем информацию о целях и счетчиках метрики
        val countersInfoById = if (useMetrikaClient) {
            val filter = UserCountersExtendedFilter().withCounterIds(counterIds.toList())
            val uids = rbacService.getClientRepresentativesUids(clientId).toList()
            metrikaClient.getUsersCountersNumExtended2(uids, filter).users
                .map { it.counters }
                .flatten()
                .associateBy { it.id }
        } else {
            metrikaCountersRepository.getCountersByIds(counterIds.map { it.toInt() })
        }

        val availableCounterIds = countersInfoById.keys

        val goalsInfo = if (useMetrikaClient) {
            metrikaClient.getMassCountersGoalsFromMetrika(availableCounterIds)
        } else {
            metrikaConversionAdGoalsRepository.getCounterGoalsByIds(availableCounterIds, goalIds)
        }

        val hasPriceByGoalId = if (useMetrikaClient) {
            metrikaClient.getGoalsConversionInfoByCounterIds(
                availableCounterIds,
                MetrikaGoalsConversionService.DAYS_WITH_CONVERSION_VISITS
            ).values.associate { it.goalId to it.isHasPrice }
        } else {
            goalIds.associateWith { false }
        }

        val countedSelections = if (selections != null) {
            // валидация существования целей и счетчиков в метрике
            val validationService = ConversionCenterMetrikaGoalsValidationService()
            val vr = validationService.validate(selections, goalsInfo)
            if (vr.hasAnyErrors()) {
                return vr
            }
            selections
        } else {
            val tempSelections = mutableListOf<MetrikaGoalSelection>()
            goalsInfo.forEach { (counterId, goals) ->
                val filteredGoals = goals.filter {
                    val goalType = Goal.computeType(it.id.toLong())
                    goalIds.contains(it.id.toLong()) && goalType == GoalType.GOAL
                        && !CampaignConstants.SPECIAL_GOAL_IDS.contains(it.id.toLong())
                }
                filteredGoals.forEach {
                    tempSelections.add(MetrikaGoalSelection(
                        goalId = it.id.toLong(),
                        metrikaCounterId = counterId.toLong(),
                        value = null,
                        isSelected = true
                    ))
                }
            }
            tempSelections
        }

        // вычисление новых значений источников конверсий
        val selectionGoals = countedSelections.mapToSet { it.goalId!! }
        val selectionByCounterId = countedSelections.groupBy { it.metrikaCounterId!! }

        val currentConversionSourcesByCounter =
            conversionSourceService.getByClientId(clientId, ConversionSourceTypeCode.METRIKA)
                .associateBy { (it.settings as ConversionSourceSettings.Metrika).counterId }
        val modifiedConversionSourcesByCounter = computeModifiedConversionSources(
            clientId,
            operatorUid,
            selectionGoals,
            selectionByCounterId,
            currentConversionSourcesByCounter,
            goalsInfo,
            countersInfoById,
            hasPriceByGoalId
        )

        deleteMetrikaConversionSources(clientId, modifiedConversionSourcesByCounter)
        addMetrikaConversionSources(clientId, modifiedConversionSourcesByCounter, useMetrikaClient)
        updateMetrikaConversionSources(clientId, modifiedConversionSourcesByCounter, currentConversionSourcesByCounter, useMetrikaClient)

        return ValidationResult.success(selections)
    }

    private fun getCounterIdsFromCampaigns(campaigns: Collection<BaseCampaign>): Set<Int> {
        val counterIds = mutableSetOf<Int>()
        for (campaign in campaigns) {
            if (campaign is CampaignWithMetrikaCounters) {
                counterIds.addAll(campaign.metrikaCounters?.map { it.toInt() } ?: listOf())
            }
        }
        return counterIds
    }

    private fun getGoalIdsFromCampaigns(campaigns: Collection<BaseCampaign>): Set<Long> {
        val goalIds = mutableSetOf<Long>()
        for (campaign in campaigns) {
            if (campaign is CampaignWithMeaningfulGoals) {
                goalIds.addAll(campaign.meaningfulGoals?.map { it.goalId } ?: listOf())
            }
            if (campaign is CampaignWithStrategy) {
                val goalId = campaign.strategy?.strategyData?.goalId
                if (goalId != null) {
                    goalIds.add(goalId)
                }
            }
        }
        return goalIds
    }

    private fun deleteMetrikaConversionSources(
        clientId: ClientId,
        modifiedConversionSourcesByCounter: Map<Long, ConversionSource>,
    ) {
        val deletingMetrikaConversionSourcesIds =
            modifiedConversionSourcesByCounter.filter { it.value.actions.isEmpty() }.values.map { it.id!! }

        val result = conversionSourceService.remove(clientId, deletingMetrikaConversionSourcesIds)

        check(!result.validationResult.hasAnyErrors()) {
            // Не ожидаем каких-либо ошибок валидации, т.к. мы сами составили запрос
            // на удаление существующих источников. Ошибка по-идее возможна только в результате гонок
            "Unexpected validation errors during delete: ${result.validationResult.flattenErrors()}"
        }
    }

    private fun addMetrikaConversionSources(
        clientId: ClientId,
        modifiedConversionSourcesByCounter: Map<Long, ConversionSource>,
        validateCounterAccess: Boolean,
    ) {
        val addingMetrikaConversionSources =
            modifiedConversionSourcesByCounter.filter { it.value.id == null }.values.toList()

        val result = conversionSourceService.add(clientId, addingMetrikaConversionSources, validateCounterAccess)

        check(!result.validationResult.hasAnyErrors()) {
            // Не ожидаем каких-либо ошибок валидации
            "Unexpected validation errors during add: ${result.validationResult.flattenErrors()}"
        }
    }

    private fun updateMetrikaConversionSources(
        clientId: ClientId,
        modifiedConversionSourcesByCounter: Map<Long, ConversionSource>,
        currentConversionSourcesByCounter: Map<Long, ConversionSource>,
        validateCounterAccess: Boolean,
    ) {
        val updatingMetrikaConversionSources = modifiedConversionSourcesByCounter.filter {
            currentConversionSourcesByCounter.contains(it.key) && it.value.actions.isNotEmpty() &&
                it.value.actions != currentConversionSourcesByCounter[it.key]?.actions
        }.values.toList()

        val result = conversionSourceService.update(clientId, updatingMetrikaConversionSources, validateCounterAccess)

        check(!result.validationResult.hasAnyErrors()) {
            // Не ожидаем каких-либо ошибок валидации
            "Unexpected validation errors during update: ${result.validationResult.flattenErrors()}"
        }
    }

    private fun computeModifiedConversionSources(
        clientId: ClientId,
        operatorUid: Long,
        selectionGoals: Set<Long>,
        selectionByCounterId: Map<Long, List<MetrikaGoalSelection>>,
        currentConversionSourcesByCounter: Map<Long, ConversionSource>,
        goalsInfo: Map<Int, List<CounterGoal>>,
        countersInfoById: Map<Int, CounterInfoDirect>,
        hasPriceByGoalId: Map<Long, Boolean>,
    ): Map<Long, ConversionSource> {

        val modifiedConversionActionsByCounter = computeModifiedConversionActions(
            selectionGoals,
            selectionByCounterId,
            currentConversionSourcesByCounter,
            goalsInfo,
            hasPriceByGoalId
        )

        val modifiedConversionSourcesByCounter = mutableMapOf<Long, ConversionSource>()
        modifiedConversionActionsByCounter.map { (counterId, actions) ->
            if (!currentConversionSourcesByCounter.contains(counterId)) {
                val counter = countersInfoById[counterId.toInt()]
                modifiedConversionSourcesByCounter.put(
                    counterId, ConversionSource(
                    id = null,
                    typeCode = ConversionSourceTypeCode.METRIKA,
                    clientId = clientId,
                    name = getNameFromCounter(countersInfoById, counterId),
                    settings = ConversionSourceSettings.Metrika(counterId, counter?.sitePath ?: ""),
                    counterId = counterId,
                    actions = actions,
                    updatePeriodHours = 0,
                    destination = Destination.CrmApi(counterId, operatorUid),
                    processingInfo = ProcessingInfo(ProcessingStatus.SUCCESS)
                )
                )
            } else {
                modifiedConversionSourcesByCounter.put(
                    counterId,
                    currentConversionSourcesByCounter[counterId]!!.copy(actions = actions)
                )
            }
        }
        return modifiedConversionSourcesByCounter
    }

    private fun computeModifiedConversionActions(
        selectionGoals: Set<Long>,
        selectionByCounterId: Map<Long, List<MetrikaGoalSelection>>,
        currentConversionSourcesByCounter: Map<Long, ConversionSource>,
        goalsInfo: Map<Int, List<CounterGoal>>,
        hasPriceByGoalId: Map<Long, Boolean>,
    ): Map<Long, List<ConversionAction>> {
        val allCounters = currentConversionSourcesByCounter.keys + selectionByCounterId.keys
        val goalsInfoById = goalsInfo.values.flatten().associateBy { it.id }

        val modifiedConversionActionsByCounter = allCounters.associateWith { counterId ->
            val modifiedActions = mutableListOf<ConversionAction>()
            currentConversionSourcesByCounter[counterId]?.actions?.forEach { action ->
                if (!selectionGoals.contains(action.goalId)) {
                    modifiedActions.add(action)
                }
            }
            selectionByCounterId[counterId]?.forEach { selection ->
                val goalId = selection.goalId!!
                if (selection.isSelected) {
                    modifiedActions.add(
                        computeConversionAction(
                            selection,
                            goalsInfoById[goalId.toInt()]?.name,
                            hasPriceByGoalId[goalId],
                            goalsInfoById[goalId.toInt()]?.defaultPrice
                        )
                    )
                }
            }
            modifiedActions
        }
        return modifiedConversionActionsByCounter
    }

    private fun computeConversionAction(
        selection: MetrikaGoalSelection,
        name: String?,
        hasPrice: Boolean?,
        defaultPrice: BigDecimal?,
    ): ConversionAction {
        val conversionActionValue = selection.value
            ?: if (hasPrice == true || defaultPrice?.compareTo(BigDecimal.valueOf(0L)) == 1) {
                ConversionActionValue.Dynamic(
                    null,
                    null
                )
            } else null

        return ConversionAction(
            name = name ?: selection.goalId!!.toString(),
            goalId = selection.goalId,
            value = conversionActionValue
        )
    }

    private fun getNameFromCounter(countersInfoById: Map<Int, CounterInfoDirect>, counterId: Long): String {
        val counter = countersInfoById[counterId.toInt()] ?: return counterId.toString()
        return if (counter.name.isNullOrBlank()) {
            counter.sitePath ?: ""
        } else {
            counter.name
        }
    }
}
