package ru.yandex.direct.core.entity.conversionsource.service

import ru.yandex.direct.core.entity.conversionsource.model.MetrikaGoalSelection
import ru.yandex.direct.metrika.client.model.response.CounterGoal
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

class ConversionCenterMetrikaGoalsValidationService {
    fun validate(
        selections: List<MetrikaGoalSelection>,
        goalsInfo: Map<Int, List<CounterGoal>>,
    ): ValidationResult<List<MetrikaGoalSelection>, Defect<*>> {
        val selectionGoalToCounter = selections.associate { it.goalId!! to it.metrikaCounterId!! }

        val metrikaCountersSet = goalsInfo.map { it.key.toLong() }.toSet()
        val metrikaGoalToCounter =
            goalsInfo.flatMap { (counterId, goal) -> goal.map { it.id.toLong() to counterId.toLong()} }.toMap()

        return ListValidationBuilder.of<MetrikaGoalSelection, Defect<*>>(selections)
            .checkEachBy { selection -> validateCounterExistence(selection, metrikaCountersSet) }
            .checkEachBy({ selection ->
                validateGoalMatchCounter(
                    selection,
                    selectionGoalToCounter,
                    metrikaGoalToCounter
                )
            }, When.isValid())
            .result
    }

    private fun validateGoalMatchCounter(
        selections: MetrikaGoalSelection,
        selectionGoalToCounter: Map<Long, Long>,
        metrikaGoalToCounter: Map<Long, Long>
    ) = validateObject(selections) {
        property(selections::goalId) {
            check(
                Constraint.fromPredicate(
                    { goalId -> (metrikaGoalToCounter[goalId] == selectionGoalToCounter[goalId]) },
                    invalidValue()
                )
            )
        }
    }

    private fun validateCounterExistence(selections: MetrikaGoalSelection, metrikaCountersSet: Set<Long>) =
        validateObject(selections) {
            property(selections::metrikaCounterId) {
                check(inSet(metrikaCountersSet).overrideDefect(objectNotFound()))
            }
        }
}
