package ru.yandex.direct.core.entity.conversionsource.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.conversionsource.model.ConversionActionName
import ru.yandex.direct.core.entity.conversionsource.model.ConversionActionValue
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSourceId
import ru.yandex.direct.core.entity.conversionsource.validation.ConversionSourceValidationContextProvider
import ru.yandex.direct.core.entity.conversionsource.validation.ConversionSourceValidationService
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceTypeCode
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.grut.api.ClientGrutModel
import ru.yandex.direct.core.grut.api.ConversionSourceGrutApi.Companion.CONVERSION_SOURCE_FETCH_LIMIT
import ru.yandex.direct.core.grut.api.GrutApiBase
import ru.yandex.direct.core.grut.api.GrutApiBase.Companion.GRUT_GET_OBJECTS_ATTEMPTS
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.mapToSet
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.TProcessingInfo.EProcessingStatus.PS_SUCCESS

private typealias GoalId = Long

@Service
class ConversionSourceService(
    grutApiService: GrutApiService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val conversionSourceValidationService: ConversionSourceValidationService,
    private val validationContextProvider: ConversionSourceValidationContextProvider,
) {
    private val clientGrutApi = grutApiService.clientGrutDao
    private val conversionSourceGrutApi = grutApiService.conversionSourceGrutApi

    fun getByClientId(clientId: ClientId): List<ConversionSource> {
        return grutTransactionProvider.runRetryable(GRUT_GET_OBJECTS_ATTEMPTS) {
            conversionSourceGrutApi.selectByClientId(clientId)
        }
    }

    fun getByClientId(clientId: ClientId, type: ConversionSourceTypeCode): List<ConversionSource> {
        return grutTransactionProvider.runRetryable(GRUT_GET_OBJECTS_ATTEMPTS) {
            conversionSourceGrutApi.selectByClientId(clientId)
        }.filter { it.typeCode == type }
    }

    fun getByIds(ids: Collection<ConversionSourceId>) =
        grutTransactionProvider.runRetryable(GRUT_GET_OBJECTS_ATTEMPTS) {
            conversionSourceGrutApi.selectByIds(ids)
        }

    /**
     * Получить источники конверсий (не метрика), в которых не указана цель в конверсионных действиях,
     * но уже прошла обработка и она успешно завершилась
     */
    fun getSourcesWithoutGoals(chunkSize: Long = CONVERSION_SOURCE_FETCH_LIMIT): Sequence<ConversionSource> =
        sequence {
            var continuationToken: String? = null
            do {
                val sources = grutTransactionProvider.runRetryable(GRUT_GET_OBJECTS_ATTEMPTS) {
                    val result = conversionSourceGrutApi.selectByFilter(
                        "[/spec/processing_info/processing_status] = ${PS_SUCCESS.number}",
                        limit = chunkSize,
                        continuationToken = continuationToken)
                    continuationToken = result.first
                    return@runRetryable result.second
                }
                yieldAll(
                    sources.asSequence()
                        .filter { it.typeCode != ConversionSourceTypeCode.METRIKA }
                        .filter { it.actions.any { conversionAction -> conversionAction.goalId == null } }
                )
            } while (sources.size >= chunkSize)
        }

    fun add(
        clientId: ClientId,
        conversionSources: List<ConversionSource>,
        validateCounterAccess: Boolean = true,
    ): MassResult<ConversionSourceId> {
        if (conversionSources.isEmpty()) {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(conversionSources))
        }
        ensureClientIsExist(listOf(clientId))
        val validationContext =
            validationContextProvider.get(clientId, conversionSources.map { it.counterId }, validateCounterAccess)
        val validationResult = conversionSourceValidationService.validateAdd(validationContext, conversionSources)
        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), validationResult)
        }
        return grutTransactionProvider.runRetryable(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val ids = conversionSourceGrutApi.createObjects(conversionSources)
            MassResult.successfulMassAction(ids, validationResult)
        }
    }

    fun update(
        clientId: ClientId,
        conversionSources: List<ConversionSource>,
        validateCounterAccess: Boolean = true,
    ): MassResult<ConversionSourceId> {
        if (conversionSources.isEmpty()) {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(conversionSources))
        }
        ensureClientIsExist(listOf(clientId))

        val validationContext =
            validationContextProvider.get(clientId, conversionSources.map { it.counterId }, validateCounterAccess)
        val validationResult = conversionSourceValidationService.validateUpdate(validationContext, conversionSources)
        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), validationResult)
        }
        return grutTransactionProvider.runRetryable(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            conversionSourceGrutApi.updateConversionSources(conversionSources)
            MassResult.successfulMassAction(conversionSources.map { it.id }, validationResult)
        }
    }

    fun remove(clientId: ClientId, ids: List<ConversionSourceId>): MassResult<ConversionSourceId> {
        if (ids.isEmpty()) {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(ids))
        }
        val validationContext = validationContextProvider.get(clientId, null, true)
        val validationResult = conversionSourceValidationService.validateRemove(validationContext, ids)
        return if (validationResult.hasErrors()) {
            MassResult.brokenMassAction(emptyList(), validationResult)
        } else grutTransactionProvider.runRetryable(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            conversionSourceGrutApi.delete(ids)
            MassResult.successfulMassAction(ids, validationResult)
        }
    }

    /**
     * Обновляет цели в конверсионных действиях указанных источников конверсий
     *
     * Конверсионных действия идентифицируется по имени в рамках источника. Источники идентифицируются по ID.
     *
     * Игнорирует не найденные источники конверсий, конверсионные действия с отличным от заданных именем.
     */
    fun setConversionActionGoals(changes: Map<ConversionSourceId, Map<ConversionActionName, GoalId>>) {
        check(changes.size <= CONVERSION_SOURCE_FETCH_LIMIT)
        val ids = changes.keys
        return grutTransactionProvider.runRetryable(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val sources = conversionSourceGrutApi.selectByIds(ids)
            val sourceToUpdate = sources.asSequence()
                .mapNotNull { source: ConversionSource ->
                    val goalsToSet = changes[source.id]!!
                    var needToUpdate = false
                    val actions = source.actions.map {
                        if (it.name in goalsToSet && it.goalId != goalsToSet[it.name]) {
                            needToUpdate = true
                            it.copy(goalId = goalsToSet[it.name])
                        } else {
                            it
                        }
                    }
                    return@mapNotNull if (needToUpdate) {
                        source.copy(actions = actions)
                    } else {
                        null
                    }
                }
                .toList()

            conversionSourceGrutApi.updateConversionActions(sourceToUpdate)
        }
    }

    private fun ensureClientIsExist(clientIds: Collection<ClientId>) {
        check(clientIds.size <= 1000)
        grutTransactionProvider.runRetryable(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val existentClientIds = clientGrutApi.getClients(clientIds.map { it.asLong() }).mapToSet { it.meta.id }
            val clientToCreate = clientIds.filter { it.asLong() !in existentClientIds }
            if (clientToCreate.isNotEmpty()) {
                clientGrutApi.createObjects(clientToCreate.map {
                    ClientGrutModel(Client().withClientId(it.asLong()), listOf())
                })
            }
        }
    }

    /**
     * Определяет "выбранность" и ценность цели в Центре Конверсий.
     *
     * Цель является выбранной, если существует источник конверсий, с конверсионным действием, в котором
     * указана эта цель.
     *
     * Выдаёт результат для каждой цели, переданной в аргументах
     */
    fun calcSelectionInfo(clientId: ClientId, goalIds: Set<GoalId>): Map<GoalId, GoalSelectionInfo> {
        val savedGoalValues = getByClientId(clientId).asSequence()
            .flatMap { it.actions }
            .filter { it.goalId != null }
            .associate { it.goalId!! to it.value }
        return goalIds.associateWith {
            if (it !in savedGoalValues) {
                GoalSelectionInfo(false, null)
            } else {
                GoalSelectionInfo(true, savedGoalValues[it])
            }
        }
    }
}

data class GoalSelectionInfo(
    val isSelected: Boolean,
    val value: ConversionActionValue?,
)
