package ru.yandex.direct.core.entity.conversionsource.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.grut.api.ConversionSourceGrutApi
import ru.yandex.direct.core.grut.api.GrutApiBase
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.utils.mapToSet

@Service
class ConversionSourceValidationContextProvider(
    grutContext: GrutContext,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val metrikaClient: MetrikaClient,
    private val rbacService: RbacService,
) {
    private val conversionSourceGrutApi = ConversionSourceGrutApi(grutContext)

    fun get(clientId: ClientId, inputCounterIds: List<Long>? = null, validateCounterAccess: Boolean): ConversionSourceValidationContext {
        val sources = grutTransactionProvider.runRetryable(GrutApiBase.GRUT_GET_OBJECTS_ATTEMPTS) {
            conversionSourceGrutApi.selectByClientId(clientId)
        }
        val accessibleCounterIds = if (inputCounterIds == null) {
            emptySet()
        } else if (!validateCounterAccess) {
            inputCounterIds.toSet()
        } else {
            val filter = UserCountersExtendedFilter().withCounterIds(inputCounterIds)
            val clientReps = rbacService.getClientRepresentativesUids(clientId).toList()
            metrikaClient.getUsersCountersNumExtended2(clientReps, filter)
                .users
                .flatMap { it.counters }
                .mapNotNull { it.id }
                .mapToSet { it.toLong() }
        }

        return ConversionSourceValidationContext(
            owner = clientId,
            existingConversionSources = sources.associateBy { it.id!! },
            accessibleCounterIds = accessibleCounterIds
        )
    }
}
