package ru.yandex.direct.core.entity.conversionsource.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.conversionsource.model.ConversionAction
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSourceSettings
import ru.yandex.direct.core.entity.conversionsource.model.Destination
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceTypeCode
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.collectionSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.constraint.CommonConstraints.isEqual
import ru.yandex.direct.validation.constraint.NumberConstraints.inRange
import ru.yandex.direct.validation.constraint.StringConstraints.admissibleChars
import ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength
import ru.yandex.direct.validation.constraint.StringConstraints.notBlank
import ru.yandex.direct.validation.constraint.StringConstraints.validHref
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.D
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateObject

private const val MAX_NAME_LENGTH = 300
private const val MAX_UPDATE_PERIOD_HOURS = 24 * 365

const val ACTION_NAME_IN_PROGRESS = "IN_PROGRESS"
const val ACTION_NAME_PAID = "PAID"
private val VALID_LINK_ACTION_NAME = setOf(ACTION_NAME_IN_PROGRESS, ACTION_NAME_PAID)

@Service
class ConversionSourceValidationService {
    fun validateAdd(
        context: ConversionSourceValidationContext,
        sources: List<ConversionSource>,
    ): ValidationResult<List<ConversionSource>, D> {
        return validateList(sources) {
            check(notEmptyCollection())
            checkEachBy { source -> validateConversionSource(context, source) }
        }
    }

    fun validateUpdate(
        context: ConversionSourceValidationContext,
        sources: List<ConversionSource>,
    ): ValidationResult<List<ConversionSource>, D> {
        return validateList(sources) {
            check(notEmptyCollection())
            checkEachBy { source -> validateConversionSource(context, source) }
        }
    }

    fun validateRemove(
        context: ConversionSourceValidationContext,
        ids: List<Long>,
    ): ValidationResult<List<Long>, D> {
        return validateList(ids) {
            checkEach(context.sourceIdIsExisting())
        }
    }

    fun validateConversionSource(context: ConversionSourceValidationContext, conversionSource: ConversionSource) =
        validateObject(conversionSource) {
            property(ConversionSource::id) {
                check(context.sourceIdIsExisting())
            }
            property(ConversionSource::clientId) {
                check(context.clientIdIsOwner())
            }
            property(ConversionSource::name) {
                check(notBlank())
                // TODO zakhar: Нужна ли эта проверка для источников по урлу?
                check(admissibleChars(), When.isTrue(conversionSource.typeCode != ConversionSourceTypeCode.METRIKA))
                check(maxStringLength(MAX_NAME_LENGTH))
            }
            property(ConversionSource::settings) {
                check(settingsCorrespondToType(conversionSource.typeCode))
                checkBy(::validateConversionSettings)
            }
            property(ConversionSource::counterId) {
                check(context.ownerHasAccessToCounterId())
            }
            listProperty(ConversionSource::actions) {
                checkEach(
                    unique { action: ConversionAction -> action.name },
                    When.isValidAnd(When.isTrue(conversionSource.typeCode != ConversionSourceTypeCode.METRIKA))
                )
                if (conversionSource.typeCode != ConversionSourceTypeCode.METRIKA) {
                    check(collectionSize(2, 2))
                    if (!result.hasAnyErrors()) {
                        checkEachBy { action: ConversionAction ->
                            validateObject(action) {
                                property(ConversionAction::name) {
                                    check(inSet(VALID_LINK_ACTION_NAME).overrideDefect(invalidValue()))
                                }
                            }
                        }
                    }
                }
            }
            property(ConversionSource::updatePeriodHours) {
                check(inRange(0, MAX_UPDATE_PERIOD_HOURS))
            }
            property(ConversionSource::destination) {
                checkBy { destination -> validateDestination(conversionSource.counterId, destination) }
            }
        }

    private fun validateConversionSettings(settings: ConversionSourceSettings): ValidationResult<ConversionSourceSettings, D> {
        return when (settings) {
            is ConversionSourceSettings.Link ->
                validateObject(settings) {
                    property(settings::url) {
                        check(validHref())
                    }
                }
            is ConversionSourceSettings.Ftp ->
                validateObject(settings) {
                    property(settings::host) {
                        check(notBlank())
                    }
                    property(settings::encryptedPassword) {
                        check(notBlank())
                    }
                    property(settings::login) {
                        check(notBlank())
                    }
                    property(settings::path) {
                        check(notBlank())
                    }
                    property(settings::port) {
                        check(inRange(1, 65535).overrideDefect(invalidValue()))
                    }
                }
            is ConversionSourceSettings.SFtp ->
                validateObject(settings) {
                    property(settings::host) {
                        check(notBlank())
                    }
                    property(settings::encryptedPassword) {
                        check(notBlank())
                    }
                    property(settings::login) {
                        check(notBlank())
                    }
                    property(settings::path) {
                        check(notBlank())
                    }
                    property(settings::port) {
                        check(inRange(1, 65535).overrideDefect(invalidValue()))
                    }
                }
            is ConversionSourceSettings.GoogleSheets ->
                validateObject(settings) {
                    property(settings::url) {
                        check(validHref())
                    }
                }
            else ->
                ValidationResult.success(settings)

        }
    }

    private fun validateDestination(
        counterId: Long,
        destination: Destination,
    ): ValidationResult<Destination, D> {
        return if (destination !is Destination.CrmApi) {
            ValidationResult.success(destination)
        } else {
            validateObject(destination) {
                property(destination::counterId) {
                    check(isEqual(counterId, invalidValue()))
                }
            }
        }
    }
}

data class ConversionSourceValidationContext(
    val owner: ClientId,
    val existingConversionSources: Map<Long, ConversionSource>,
    val accessibleCounterIds: Set<Long>,
)

private fun ConversionSourceValidationContext.sourceIdIsExisting(): Constraint<Long?, D> {
    return inSet(existingConversionSources.keys).overrideDefect(CommonDefects.objectNotFound())
}

private fun ConversionSourceValidationContext.clientIdIsOwner(): Constraint<ClientId?, D> {
    return isEqual(owner, invalidValue())
}

private fun ConversionSourceValidationContext.ownerHasAccessToCounterId(): Constraint<Long?, D> {
    return Constraint.fromPredicate(
        { counterId: Long? -> counterId == null || counterId in accessibleCounterIds },
        { counterId: Long? -> ConversionSourceDefects.counterIsInaccessible(counterId) })
}

private fun settingsCorrespondToType(typeCode: ConversionSourceTypeCode): Constraint<ConversionSourceSettings, D> {
    val expectedSettingClass = when (typeCode) {
        ConversionSourceTypeCode.LINK -> ConversionSourceSettings.Link::class
        ConversionSourceTypeCode.METRIKA -> ConversionSourceSettings.Metrika::class
        ConversionSourceTypeCode.FTP -> ConversionSourceSettings.Ftp::class
        ConversionSourceTypeCode.SFTP -> ConversionSourceSettings.SFtp::class
        ConversionSourceTypeCode.GOOGLE_SHEETS -> ConversionSourceSettings.GoogleSheets::class
        else -> {
            null
        }
    }
    return Constraint.fromPredicate(
        { expectedSettingClass != null && expectedSettingClass.isInstance(it) },
        invalidValue(),
    )
}
