package ru.yandex.direct.core.entity.conversionsourcetype.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceType;
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceTypeCode;
import ru.yandex.direct.dbschema.ppcdict.tables.records.CcConversionSourceTypesRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.dbschema.ppcdict.tables.CcConversionSourceTypes.CC_CONVERSION_SOURCE_TYPES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class ConversionSourceTypeRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ConversionSourceType> jooqMapper;

    @Autowired
    public ConversionSourceTypeRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(ConversionSourceType::new)
                .map(property(ConversionSourceType.ID, CC_CONVERSION_SOURCE_TYPES.ID))
                .map(property(ConversionSourceType.NAME, CC_CONVERSION_SOURCE_TYPES.NAME))
                .map(property(ConversionSourceType.DESCRIPTION, CC_CONVERSION_SOURCE_TYPES.DESCRIPTION))
                .map(property(ConversionSourceType.ICON_URL, CC_CONVERSION_SOURCE_TYPES.ICON_URL))
                .map(property(ConversionSourceType.ACTIVATION_URL, CC_CONVERSION_SOURCE_TYPES.ACTIVATION_URL))
                .map(convertibleProperty(ConversionSourceType.IS_DRAFT,
                        CC_CONVERSION_SOURCE_TYPES.IS_DRAFT,
                        RepositoryUtils::booleanFromLong,
                        RepositoryUtils::booleanToLong))
                .map(property(ConversionSourceType.POSITION, CC_CONVERSION_SOURCE_TYPES.POSITION))
                .map(convertibleProperty(ConversionSourceType.CODE,
                        CC_CONVERSION_SOURCE_TYPES.CODE,
                        ConversionSourceTypeCode::fromSource,
                        ConversionSourceTypeCode::toSource))
                .map(convertibleProperty(ConversionSourceType.IS_EDITABLE,
                        CC_CONVERSION_SOURCE_TYPES.IS_EDITABLE,
                        RepositoryUtils::booleanFromLong,
                        RepositoryUtils::booleanToLong))
                .map(property(ConversionSourceType.NAME_EN, CC_CONVERSION_SOURCE_TYPES.NAME_EN))
                .map(property(ConversionSourceType.DESCRIPTION_EN, CC_CONVERSION_SOURCE_TYPES.DESCRIPTION_EN))
                .build();
    }

    @QueryWithoutIndex("Взятие всех элементов таблицы. Кол. элементов в таблице не существенно для разбития по частям")
    public List<ConversionSourceType> getAll() {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(CC_CONVERSION_SOURCE_TYPES)
                .orderBy(CC_CONVERSION_SOURCE_TYPES.POSITION)
                .fetch(jooqMapper::fromDb);
    }

    @QueryWithoutIndex("Взятие всех черновиков из таблицы. Кол. элементов в таблице не существенно для разбития по " +
            "частям")
    public List<ConversionSourceType> getNotDrafts() {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(CC_CONVERSION_SOURCE_TYPES)
                .where(CC_CONVERSION_SOURCE_TYPES.IS_DRAFT.eq(0L))
                .orderBy(CC_CONVERSION_SOURCE_TYPES.POSITION)
                .fetch(jooqMapper::fromDb);
    }

    public List<ConversionSourceType> getConversionSourceTypeByIds(List<Long> ids) {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(CC_CONVERSION_SOURCE_TYPES)
                .where(CC_CONVERSION_SOURCE_TYPES.ID.in(ids))
                .fetch(jooqMapper::fromDb);
    }

    public List<Long> add(List<ConversionSourceType> conversionSourceTypes) {
        return InsertHelper.addModelsAndReturnIds(
                dslContextProvider.ppcdict(),
                CC_CONVERSION_SOURCE_TYPES,
                jooqMapper,
                conversionSourceTypes,
                CC_CONVERSION_SOURCE_TYPES.ID
        );
    }

    public List<Long> update(Collection<AppliedChanges<ConversionSourceType>> appliedChanges) {
        JooqUpdateBuilder<CcConversionSourceTypesRecord, ConversionSourceType> updateBuilder =
                new JooqUpdateBuilder<>(CC_CONVERSION_SOURCE_TYPES.ID, appliedChanges);
        updateBuilder.processProperty(ConversionSourceType.NAME, CC_CONVERSION_SOURCE_TYPES.NAME);
        updateBuilder.processProperty(ConversionSourceType.DESCRIPTION, CC_CONVERSION_SOURCE_TYPES.DESCRIPTION);
        updateBuilder.processProperty(ConversionSourceType.ICON_URL, CC_CONVERSION_SOURCE_TYPES.ICON_URL);
        updateBuilder.processProperty(ConversionSourceType.ACTIVATION_URL, CC_CONVERSION_SOURCE_TYPES.ACTIVATION_URL);
        updateBuilder.processProperty(ConversionSourceType.IS_DRAFT, CC_CONVERSION_SOURCE_TYPES.IS_DRAFT,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(ConversionSourceType.POSITION, CC_CONVERSION_SOURCE_TYPES.POSITION);
        updateBuilder.processProperty(ConversionSourceType.CODE, CC_CONVERSION_SOURCE_TYPES.CODE,
                ConversionSourceTypeCode::toSource);
        updateBuilder.processProperty(ConversionSourceType.IS_EDITABLE, CC_CONVERSION_SOURCE_TYPES.IS_EDITABLE,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(ConversionSourceType.NAME_EN, CC_CONVERSION_SOURCE_TYPES.NAME_EN);
        updateBuilder.processProperty(ConversionSourceType.DESCRIPTION_EN, CC_CONVERSION_SOURCE_TYPES.DESCRIPTION_EN);

        dslContextProvider.ppcdict()
                .update(CC_CONVERSION_SOURCE_TYPES)
                .set(updateBuilder.getValues())
                .where(CC_CONVERSION_SOURCE_TYPES.ID.in(updateBuilder.getChangedIds()))
                .execute();

        List<Long> ids = new ArrayList<>();
        appliedChanges.forEach(appliedChange -> ids.add(appliedChange.getModel().getId()));
        return ids;
    }

    public void remove(List<Long> ids) {
        dslContextProvider.ppcdict()
                .deleteFrom(CC_CONVERSION_SOURCE_TYPES)
                .where(CC_CONVERSION_SOURCE_TYPES.ID.in(ids))
                .execute();
    }
}
