package ru.yandex.direct.core.entity.conversionsourcetype.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.IntStream;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceType;
import ru.yandex.direct.core.entity.conversionsourcetype.repository.ConversionSourceTypeRepository;
import ru.yandex.direct.core.entity.conversionsourcetype.service.validadation.ConversionSourceTypeValidationService;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для работы с типами источников конверсий для Центра Конверсий (ЦК)
 */

@Service
public class ConversionSourceTypeService {

    private final ConversionSourceTypeRepository conversionSourceTypeRepository;

    private final ConversionSourceTypeValidationService conversionSourceTypeValidationService;

    private final ModelChangesValidationTool preValidationTool;


    @Autowired
    public ConversionSourceTypeService(ConversionSourceTypeRepository conversionSourceTypeRepository,
                                       ConversionSourceTypeValidationService conversionSourceTypeValidationService) {
        this.conversionSourceTypeRepository = conversionSourceTypeRepository;
        this.conversionSourceTypeValidationService = conversionSourceTypeValidationService;

        this.preValidationTool = ModelChangesValidationTool.builder().build();
    }

    /**
     * Чтение всех существующих типов источников конверсий для Центра Конверсий (ЦК)
     *
     * @return список {@link ConversionSourceType} всех типов источников конверсий
     */
    public List<ConversionSourceType> getAll() {
        return conversionSourceTypeRepository.getAll();
    }

    /**
     * Чтение всех типов источников конверсий, кроме черновиков
     *
     * @return список {@link ConversionSourceType} типов источников конверсий, у которых is_draft=false
     */
    public List<ConversionSourceType> getNotDrafts() {
        return conversionSourceTypeRepository.getNotDrafts();
    }

    /**
     * Добавление нового типа источника конверсий
     *
     * @param conversionSourceTypes список {@link ConversionSourceType} добавляемых типов источников конвесрий
     * @return список id добавленных записей
     */
    public MassResult<Long> add(List<ConversionSourceType> conversionSourceTypes) {
        ValidationResult<List<ConversionSourceType>, Defect> validationResult =
                conversionSourceTypeValidationService.validateAdd(conversionSourceTypes);

        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(new ArrayList<>(), validationResult);
        }

        List<Long> ids = conversionSourceTypeRepository.add(conversionSourceTypes);

        IntStream.range(0, conversionSourceTypes.size()).forEach(ind ->
                conversionSourceTypes.get(ind).withId(ids.get(ind))
        );

        return MassResult.successfulMassAction(ids, validationResult);
    }

    /**
     * Обновление типов источника конверсий
     *
     * @return список id изменненых записей
     */
    public MassResult<Long> update(List<ModelChanges<ConversionSourceType>> modelChanges) {
        List<Long> ids = mapList(modelChanges, ModelChanges::getId);
        List<ConversionSourceType> existingConversionSourceTypes =
                conversionSourceTypeRepository.getConversionSourceTypeByIds(ids);

        Map<Long, ConversionSourceType> conversionSourceTypesMap =
                StreamEx.of(existingConversionSourceTypes)
                        .mapToEntry(ConversionSourceType::getId)
                        .invert()
                        .toMap();

        Map<Long, Boolean> isEditableMap = StreamEx.of(ids).toMap(id -> conversionSourceTypesMap.get(id).getIsEditable());

        List<Long> existingIds = mapList(existingConversionSourceTypes, ConversionSourceType::getId);

        ValidationResult<List<ModelChanges<ConversionSourceType>>, Defect> preValidationResult =
                new ValidationResult<>(modelChanges);
        preValidationTool.validateModelChangesList(preValidationResult, () -> new HashSet<>(existingIds));

        if (preValidationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(ids, preValidationResult);
        }

        List<AppliedChanges<ConversionSourceType>> appliedChanges =
                StreamEx.of(modelChanges).map(modelChange ->
                        modelChange.applyTo(conversionSourceTypesMap.get(modelChange.getId()))
                ).toList();

        List<ConversionSourceType> conversionSourceTypes = new ArrayList<>();

        appliedChanges.forEach(appliedChange -> {
            var conversionSourceType = appliedChange.getModel();
            conversionSourceTypes.add(new ConversionSourceType()
                    .withId(conversionSourceType.getId())
                    .withName(conversionSourceType.getName())
                    .withDescription(conversionSourceType.getDescription())
                    .withIconUrl(conversionSourceType.getIconUrl())
                    .withActivationUrl(conversionSourceType.getActivationUrl())
                    .withIsDraft(conversionSourceType.getIsDraft())
                    .withPosition(conversionSourceType.getPosition())
                    .withCode(conversionSourceType.getCode())
                    .withIsEditable(isEditableMap.get(conversionSourceType.getId()))
                    .withNameEn(conversionSourceType.getNameEn())
                    .withDescriptionEn(conversionSourceType.getDescriptionEn())
            );
        });

        ValidationResult<List<ConversionSourceType>, Defect> validationResult =
                conversionSourceTypeValidationService.validateUpdate(conversionSourceTypes);

        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(ids, validationResult);
        }

        conversionSourceTypeRepository.update(appliedChanges);
        return MassResult.successfulMassAction(ids, validationResult);
    }

    /**
     * Удаление типа ичточника конверсий по id
     *
     * @param ids список id для удаляемых значений
     * @return список id удаленных записей
     */
    public MassResult<Long> remove(List<Long> ids) {
        ValidationResult<List<ConversionSourceType>, Defect> validationResult =
                conversionSourceTypeValidationService.validateRemove(ids);

        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(ids, validationResult);
        }

        conversionSourceTypeRepository.remove(ids);
        return MassResult.successfulMassAction(ids, validationResult);
    }
}
