package ru.yandex.direct.core.entity.conversionsourcetype.service.validadation;

import java.util.List;

import com.google.common.base.CharMatcher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceType;
import ru.yandex.direct.core.entity.conversionsourcetype.repository.ConversionSourceTypeRepository;
import ru.yandex.direct.utils.TextConstants;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.defect.StringDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.unableToDelete;

@Service
public class ConversionSourceTypeValidationService {

    @Autowired
    private final ConversionSourceTypeRepository conversionSourceTypeRepository;

    public static final int MAX_NAME_LENGTH = 50;
    public static final int MAX_DESCRIPTION_LENGTH = 100;

    private static final String ENG_LET_PUNCTUATION_AND_SPECIAL_SYMBOLS = TextConstants.ENG_ALPHABET +
            TextConstants.PUNCTUATION + TextConstants.SPACE_CHARS + TextConstants.NUMBERS;

    private static Constraint<String, Defect> hasEngAlphabetLettersSpaceCharsAndPunctuationOnly() {
        return fromPredicate(CharMatcher.anyOf(ENG_LET_PUNCTUATION_AND_SPECIAL_SYMBOLS)::matchesAllOf,
                StringDefects.admissibleChars());
    }

    public ConversionSourceTypeValidationService(ConversionSourceTypeRepository conversionSourceTypeRepository) {
        this.conversionSourceTypeRepository = conversionSourceTypeRepository;
    }

    public ValidationResult<List<ConversionSourceType>, Defect> validateAdd(List<ConversionSourceType> conversionSourceTypes) {
        return ListValidationBuilder.<ConversionSourceType, Defect>of(conversionSourceTypes).checkEachBy(this::validateFieldsValues).getResult();
    }

    public ValidationResult<List<ConversionSourceType>, Defect> validateUpdate(List<ConversionSourceType> conversionSourceTypes) {
        return ListValidationBuilder.<ConversionSourceType, Defect>of(conversionSourceTypes)
                .checkEachBy(this::validateFieldsValues)
                .checkEachBy(this::validateAccess)
                .getResult();
    }

    private ValidationResult<ConversionSourceType, Defect> validateFieldsValues(ConversionSourceType conversionSourceType) {
        ModelItemValidationBuilder<ConversionSourceType> vb = ModelItemValidationBuilder.of(conversionSourceType);

        vb.item(ConversionSourceType.NAME).check(maxStringLength(MAX_NAME_LENGTH));
        vb.item(ConversionSourceType.DESCRIPTION).check(maxStringLength(MAX_DESCRIPTION_LENGTH));
        vb.item(ConversionSourceType.ICON_URL).check(Constraint.fromPredicate(StringConstraints::isValidUrl,
                invalidValue()));
        vb.item(ConversionSourceType.ACTIVATION_URL).check(Constraint.fromPredicate(StringConstraints::isValidUrl,
                invalidValue()));
        vb.item(ConversionSourceType.NAME_EN).check(maxStringLength(MAX_NAME_LENGTH));
        vb.item(ConversionSourceType.DESCRIPTION_EN).check(maxStringLength(MAX_DESCRIPTION_LENGTH));
        vb.item(ConversionSourceType.NAME_EN).check(hasEngAlphabetLettersSpaceCharsAndPunctuationOnly());
        vb.item(ConversionSourceType.DESCRIPTION_EN).check(hasEngAlphabetLettersSpaceCharsAndPunctuationOnly());

        return vb.getResult();
    }

    public ValidationResult<List<ConversionSourceType>, Defect> validateRemove(List<Long> ids) {
        List<ConversionSourceType> conversionSourceTypes = conversionSourceTypeRepository.getConversionSourceTypeByIds(ids);

        if (conversionSourceTypes.isEmpty()) {
            ValidationResult<List<ConversionSourceType>, Defect> result = new ValidationResult<>(conversionSourceTypes);
            return result.addError(unableToDelete());
        }

        return ListValidationBuilder.<ConversionSourceType, Defect>of(conversionSourceTypes)
                .checkEachBy(this::validateAccess)
                .getResult();
    }

    private ValidationResult<ConversionSourceType, Defect> validateAccess(ConversionSourceType conversionSourceType) {
        ModelItemValidationBuilder<ConversionSourceType> vb = ModelItemValidationBuilder.of(conversionSourceType);
        vb.item(ConversionSourceType.IS_EDITABLE).check(Constraint.fromPredicate(isEditable -> isEditable,
                invalidValue()));

        return vb.getResult();
    }
}
