package ru.yandex.direct.core.entity.creative.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.creative.model.BannerStorageDictItem;
import ru.yandex.direct.core.entity.creative.model.BannerStorageDictLayoutItem;
import ru.yandex.direct.core.entity.creative.model.BannerStorageDictThemeItem;
import ru.yandex.direct.core.entity.creative.model.BannerStorageItemType;
import ru.yandex.direct.core.entity.creative.model.TemplateInfo;
import ru.yandex.direct.dbschema.ppcdict.enums.BannerStorageDictType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.dbschema.ppcdict.tables.BannerStorageDict.BANNER_STORAGE_DICT;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class BannerStorageDictRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqReaderWithSupplier<BannerStorageDictItem> reader;

    public BannerStorageDictRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        reader = createReader();
    }

    private JooqReaderWithSupplier<BannerStorageDictItem> createReader() {
        return JooqReaderWithSupplierBuilder.builder(BannerStorageDictItem::new)
                .readProperty(BannerStorageDictItem.ID, fromField(BANNER_STORAGE_DICT.ID))
                .readProperty(BannerStorageDictItem.TYPE,
                        fromField(BANNER_STORAGE_DICT.TYPE).by(BannerStorageItemType::fromSource))
                .readProperty(BannerStorageDictItem.JSON_CONTENT, fromField(BANNER_STORAGE_DICT.JSON_CONTENT))
                .build();
    }

    public Map<Long, BannerStorageDictThemeItem> getThemesByIds(Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyMap();
        }
        List<BannerStorageDictItem> items = dslContextProvider.ppcdict()
                .select(reader.getFieldsToRead())
                .from(BANNER_STORAGE_DICT)
                .where(BANNER_STORAGE_DICT.TYPE.eq(BannerStorageDictType.theme))
                .and(BANNER_STORAGE_DICT.ID.in(ids))
                .fetch(reader::fromDb);
        return StreamEx.of(items)
                .mapToEntry(BannerStorageDictItem::getId,
                        i -> JsonUtils.fromJson(i.getJsonContent(), BannerStorageDictThemeItem.class))
                .toMap();
    }

    public Map<Long, BannerStorageDictLayoutItem> getLayoutsByIds(Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyMap();
        }
        List<BannerStorageDictItem> items = dslContextProvider.ppcdict()
                .select(reader.getFieldsToRead())
                .from(BANNER_STORAGE_DICT)
                .where(BANNER_STORAGE_DICT.TYPE.eq(BannerStorageDictType.layout))
                .and(BANNER_STORAGE_DICT.ID.in(ids))
                .fetch(reader::fromDb);
        return StreamEx.of(items)
                .mapToEntry(BannerStorageDictItem::getId,
                        i -> JsonUtils.fromJson(i.getJsonContent(), BannerStorageDictLayoutItem.class))
                .toMap();
    }

    private Map<Long, BannerStorageDictLayoutItem> getAllElementsByType(BannerStorageDictType bannerStorageDictType) {
        List<BannerStorageDictItem> items = dslContextProvider.ppcdict()
                .select(reader.getFieldsToRead())
                .from(BANNER_STORAGE_DICT)
                .where(BANNER_STORAGE_DICT.TYPE.in(bannerStorageDictType))
                .fetch(reader::fromDb);
        return StreamEx.of(items)
                .mapToEntry(BannerStorageDictItem::getId,
                        i -> JsonUtils.fromJson(i.getJsonContent(), BannerStorageDictLayoutItem.class))
                .toMap();
    }

    public Map<Long, BannerStorageDictLayoutItem> getAllTemplates() {
        return getAllElementsByType(BannerStorageDictType.template);
    }

    public TemplateInfo getTemplateById(int templateId) {
        BannerStorageDictItem record = dslContextProvider.ppcdict()
                .select(reader.getFieldsToRead())
                .from(BANNER_STORAGE_DICT)
                .where(BANNER_STORAGE_DICT.TYPE.eq(BannerStorageDictType.template)
                        .and(BANNER_STORAGE_DICT.ID.eq((long) templateId)))
                .fetchOne(reader::fromDb);
        if (record == null) {
            return null;
        }
        return JsonUtils.fromJson(record.getJsonContent(), TemplateInfo.class);
    }
}
