package ru.yandex.direct.core.entity.creative.repository;

import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.creative.model.YabsData;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstants.FEEDS_CREATIVES_COMPATIBILITY;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPC_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_AUDIO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_INDOOR_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_OUTDOOR_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_OVERLAY_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_VIDEO_ADDITION_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.MOBILE_CONTENT_VIDEO_ADDITION_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.MOBILE_CPC_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.TEXT_VIDEO_ADDITION_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.TEXT_VIDEO_ADDITION_TALL_LAYOUT_IDS;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

public class CreativeMappings {
    private CreativeMappings() {
    }

    public static String moderationInfoToDb(ModerationInfo moderationInfo) {
        return moderationInfo != null ? toJson(moderationInfo) : null;
    }

    public static ModerationInfo moderationInfoFromDb(String moderationInfo) {
        return moderationInfo != null ? fromJson(moderationInfo, ModerationInfo.class) : null;
    }

    public static String yabsDataToDb(YabsData yabsData) {
        return yabsData != null ? toJson(yabsData) : null;
    }

    public static YabsData yabsDataFromDb(String yabsData) {
        return yabsData != null ? fromJson(yabsData, YabsData.class) : null;
    }

    public static String additionalDataToDb(AdditionalData additionalData) {
        return additionalData != null ? toJson(additionalData) : null;
    }

    public static AdditionalData additionalDataFromDb(String additionalData) {
        return additionalData != null ? fromJson(additionalData, AdditionalData.class) : null;
    }

    @Nullable
    public static PerfCreativesCreativeType creativeTypeToDb(CreativeType value) {
        if (value == null) {
            return null;
        }

        switch (value) {
            case PERFORMANCE:
                return PerfCreativesCreativeType.performance;
            case BANNERSTORAGE:
                return PerfCreativesCreativeType.bannerstorage;
            case CANVAS:
                return PerfCreativesCreativeType.canvas;
            case HTML5_CREATIVE:
                return PerfCreativesCreativeType.html5_creative;
            case VIDEO_ADDITION_CREATIVE:
            case CPC_VIDEO_CREATIVE:
            case CPM_VIDEO_CREATIVE:
            case CPM_OUTDOOR_CREATIVE:
            case CPM_INDOOR_CREATIVE:
            case CPM_AUDIO_CREATIVE:
            case CPM_OVERLAY:
                return PerfCreativesCreativeType.video_addition;
            default:
                throw new IllegalStateException("No such value: " + value);
        }
    }

    @Nullable
    static CreativeType creativeTypeByTypeAndLayoutIdFromDb(PerfCreativesCreativeType type, Long layoutId) {
        if (type == null) {
            return null;
        }

        switch (type) {
            case canvas:
                return convertCanvasType(layoutId);
            case html5_creative:
                return CreativeType.HTML5_CREATIVE;
            case performance:
                return CreativeType.PERFORMANCE;
            case bannerstorage:
                return CreativeType.BANNERSTORAGE;
            case video_addition:
                return convertVideoType(layoutId);
            default:
                throw new IllegalStateException("No such value: " + type);
        }
    }

    public static Set<CreativeBusinessType> creativeBusinessTypesFromBusinessType(BusinessType businessType) {
        return requireNonNull(FEEDS_CREATIVES_COMPATIBILITY.get(businessType), "No such value: " + businessType);
    }

    public static CreativeType convertCanvasType(Long layoutId) {
        return CreativeType.CANVAS;
    }

    public static CreativeType convertVideoType(Long layoutId) {
        checkState(layoutId != null, "layoutId must be defined for video creatives");

        if (TEXT_VIDEO_ADDITION_LAYOUT_IDS.contains(layoutId) ||
                TEXT_VIDEO_ADDITION_TALL_LAYOUT_IDS.contains(layoutId) ||
                MOBILE_CONTENT_VIDEO_ADDITION_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.VIDEO_ADDITION_CREATIVE;
        } else if (CPC_VIDEO_LAYOUT_IDS.contains(layoutId) || MOBILE_CPC_VIDEO_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.CPC_VIDEO_CREATIVE;
        } else if (CPM_VIDEO_ADDITION_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.CPM_VIDEO_CREATIVE;
        } else if (CPM_OUTDOOR_VIDEO_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.CPM_OUTDOOR_CREATIVE;
        } else if (CPM_INDOOR_VIDEO_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.CPM_INDOOR_CREATIVE;
        } else if (CPM_OVERLAY_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.CPM_OVERLAY;
        } else if (CPM_AUDIO_LAYOUT_IDS.contains(layoutId)) {
            return CreativeType.CPM_AUDIO_CREATIVE;
        } else {
            throw new IllegalStateException("Invalid layoutId value for video type: " + layoutId);
        }
    }

    public static List<Long> sumGeoFromDb(String geo) {
        Function<String, List<Long>> converter =
                geo1 -> StreamEx.split(geo1, ",").map(Long::valueOf).toList();
        return StringUtils.isNotEmpty(geo) ? converter.apply(geo) : null;
    }

    public static String sumGeoToDb(List<Long> geo) {
        return CollectionUtils.isNotEmpty(geo) ? StringUtils.join(geo, ",") : null;
    }
}
