package ru.yandex.direct.core.entity.creative.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.DatePart;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.SelectConditionStep;
import org.jooq.SortField;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.creative.container.CreativeFilterContainer;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.SourceMediaType;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CreativeBannerStorageSyncSyncStatus;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesBusinessType;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.records.CreativeBannerStorageSyncRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.PerfCreativesRecord;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toSet;
import static org.jooq.impl.DSL.choose;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPC_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_AUDIO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_INDOOR_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_OUTDOOR_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_OVERLAY_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_VIDEO_ADDITION_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.MOBILE_CONTENT_VIDEO_ADDITION_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.MOBILE_CPC_VIDEO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.TEXT_VIDEO_ADDITION_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.TEXT_VIDEO_ADDITION_TALL_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.metrika.repository.MetrikaCreativeRepository.ADVANCE_MINUTES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CREATIVE_BANNER_STORAGE_SYNC;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;

@Repository
@ParametersAreNonnullByDefault
public class CreativeRepository {

    private final DslContextProvider dslContextProvider;
    private final ShardSupport shardSupport;
    private final JooqMapperWithSupplier<Creative> creativeJooqMapper;

    @Autowired
    public CreativeRepository(ShardSupport shardSupport, DslContextProvider dslContextProvider) {
        this.shardSupport = shardSupport;
        this.dslContextProvider = dslContextProvider;

        creativeJooqMapper = createCreativeMapper();
    }

    /**
     * Добавляет креативы. Если креатив с таким id уже существует кидает исключение (а значит в случае конкурентных
     * запросов на добавление одного и того же креатива один из запросов получит исключение).
     *
     * @param shard     шард для запроса
     * @param creatives креативы
     * @throws org.jooq.exception.DataAccessException если какой-либо креатив уже существует
     */
    public void add(int shard, Collection<Creative> creatives) {
        if (creatives.isEmpty()) {
            return;
        }
        add(dslContextProvider.ppc(shard), creatives);
    }

    public void add(DSLContext dslContext, Collection<Creative> creatives) {
        if (creatives.isEmpty()) {
            return;
        }

        prepareAddCreatives(creatives);

        Map<Long, Long> creativeIdToClientId = new HashMap<>(creatives.size());
        InsertHelper<PerfCreativesRecord> insertHelper = new InsertHelper<>(dslContext, PERF_CREATIVES);
        for (Creative creative : creatives) {
            creativeIdToClientId.put(creative.getId(), creative.getClientId());
            insertHelper.add(creativeJooqMapper, creative).newRecord();
        }

        shardSupport.saveValues(ShardKey.CREATIVE_ID, ShardKey.CLIENT_ID, creativeIdToClientId);
        insertHelper.execute();
    }

    /**
     * Изменяет данные креативов
     *
     * @param shard   шард для запроса
     * @param changes набор изменений
     */
    public void update(int shard, Collection<AppliedChanges<Creative>> changes) {
        update(dslContextProvider.ppc(shard), changes);
    }

    public void update(DSLContext dslContext, Collection<AppliedChanges<Creative>> changes) {
        JooqUpdateBuilder<PerfCreativesRecord, Creative> ub =
                new JooqUpdateBuilder<>(PERF_CREATIVES.CREATIVE_ID, changes);
        ub.processProperty(Creative.NAME, PERF_CREATIVES.NAME);
        ub.processProperty(Creative.GROUP_NAME, PERF_CREATIVES.GROUP_NAME);
        ub.processProperty(Creative.WIDTH, PERF_CREATIVES.WIDTH);
        ub.processProperty(Creative.HEIGHT, PERF_CREATIVES.HEIGHT);
        ub.processProperty(Creative.PREVIEW_URL, PERF_CREATIVES.PREVIEW_URL);
        ub.processProperty(Creative.VERSION, PERF_CREATIVES.VERSION);
        ub.processProperty(Creative.YABS_DATA, PERF_CREATIVES.YABS_DATA, CreativeMappings::yabsDataToDb);
        ub.processProperty(Creative.MODERATION_INFO, PERF_CREATIVES.MODERATE_INFO,
                CreativeMappings::moderationInfoToDb);
        ub.processProperty(Creative.LIVE_PREVIEW_URL, PERF_CREATIVES.LIVE_PREVIEW_URL);
        ub.processProperty(Creative.ARCHIVE_URL, PERF_CREATIVES.ARCHIVE_URL);
        ub.processProperty(Creative.ADDITIONAL_DATA, PERF_CREATIVES.ADDITIONAL_DATA,
                CreativeMappings::additionalDataToDb);
        ub.processProperty(Creative.HAS_PACKSHOT, PERF_CREATIVES.HAS_PACKSHOT, RepositoryUtils::booleanToLong);
        ub.processProperty(Creative.EXPANDED_PREVIEW_URL, PERF_CREATIVES.EXPANDED_PREVIEW_URL);
        ub.processProperty(Creative.IS_ADAPTIVE, PERF_CREATIVES.IS_ADAPTIVE, RepositoryUtils::booleanToLong);
        ub.processProperty(Creative.IS_BRAND_LIFT, PERF_CREATIVES.IS_BRAND_LIFT, RepositoryUtils::booleanToLong);
        ub.processProperty(Creative.STATUS_MODERATE, PERF_CREATIVES.STATUS_MODERATE, StatusModerate::toSource);
        ub.processProperty(Creative.IS_BANNERSTORAGE_PREDEPLOYED, PERF_CREATIVES.IS_BANNERSTORAGE_PREDEPLOYED,
                RepositoryUtils::nullSafeBooleanToLong);
        ub.processProperty(Creative.LAYOUT_ID, PERF_CREATIVES.LAYOUT_ID);

        dslContext.update(PERF_CREATIVES)
                .set(ub.getValues())
                .where(PERF_CREATIVES.CREATIVE_ID.in(ub.getChangedIds()))
                .execute();
    }

    /**
     * Получить креативы по клиенту, если список типов креативов пустой, то ограничения по типам нет
     */
    public Set<Long> getExistingClientCreativeIds(int shard, ClientId clientId, Collection<Long> creativeIds,
                                                  Collection<CreativeType> creativeTypes) {
        return getExistingClientCreativeIds(shard, clientId, creativeIds, creativeTypes, maxLimited());
    }

    public Set<Long> getExistingClientCreativeIds(int shard, ClientId clientId, Collection<Long> creativeIds) {
        return getExistingClientCreativeIds(shard, clientId, creativeIds, Collections.emptyList());
    }

    public Set<Long> getExistingClientCreativeIds(int shard, ClientId clientId, Collection<Long> creativeIds,
                                                  Collection<CreativeType> types, LimitOffset limitOffset) {
        SelectConditionStep<Record1<Long>> condition = dslContextProvider.ppc(shard)
                .select(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()));

        if (!creativeIds.isEmpty()) {
            condition = condition.and(PERF_CREATIVES.CREATIVE_ID.in(creativeIds));
        }
        condition = condition.and(typesCondition(types));

        return condition
                .orderBy(PERF_CREATIVES.CREATIVE_ID)
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetchSet(PERF_CREATIVES.CREATIVE_ID);
    }

    /**
     * Возвращает Id креативов, связанных с баннерами, исключая архивные кампании
     *
     * @param shard          шард для запроса
     * @param clientId       ид клиента
     * @param lastCreativeId ид последнего креатива
     * @param sort           тип сортировки - asc|desc
     * @param limitOffset    ограничение количества записей и смещение от начала выборки. Если null, то смещение
     *                       берётся за 0, а ограничение из константы CreativeConstants.GET_USED_CREATIVES_LIMIT.
     */
    @Nonnull
    public List<Long> getClientUsedCreativeIds(int shard, ClientId clientId, Collection<CreativeType> types,
                                               long lastCreativeId, SqlUtils.SortOrder sort,
                                               @Nullable LimitOffset limitOffset) {
        limitOffset = limitOffset != null ? limitOffset : limited(CreativeConstants.GET_USED_CREATIVES_LIMIT);
        SortField<Long> sortOrder;
        Condition lastCreativeIdCond;

        if (SqlUtils.SortOrder.DESC.equals(sort)) {
            sortOrder = BANNERS_PERFORMANCE.CREATIVE_ID.desc();
            lastCreativeIdCond = BANNERS_PERFORMANCE.CREATIVE_ID.lessThan(lastCreativeId);
        } else {
            sortOrder = BANNERS_PERFORMANCE.CREATIVE_ID.asc();
            lastCreativeIdCond = BANNERS_PERFORMANCE.CREATIVE_ID.greaterThan(lastCreativeId);
        }

        SelectConditionStep<Record1<Long>> selectConditionStep = dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS_PERFORMANCE.CREATIVE_ID)
                .from(BANNERS_PERFORMANCE)
                .join(PERF_CREATIVES)
                .on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .join(CAMPAIGNS)
                .on(BANNERS_PERFORMANCE.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(typesCondition(types));

        if (lastCreativeId != 0L) {
            selectConditionStep = selectConditionStep.and(lastCreativeIdCond);
        }
        return selectConditionStep
                .orderBy(sortOrder)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(BANNERS_PERFORMANCE.CREATIVE_ID);
    }

    public Set<Long> getUsedCreativeIds(int shard, ClientId clientId, Collection<Long> creativeIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS_PERFORMANCE.CREATIVE_ID)
                .from(BANNERS_PERFORMANCE)
                .join(PERF_CREATIVES)
                .on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()))
                .and(PERF_CREATIVES.CREATIVE_ID.in(creativeIds))
                .fetchSet(PERF_CREATIVES.CREATIVE_ID);
    }

    public List<Creative> getCreativesWithType(int shard, @Nullable ClientId clientId, Collection<Long> ids,
                                               CreativeType type) {
        return getCreatives(shard, ids, clientId, singleton(type));
    }

    public List<Creative> getCreativesWithTypes(int shard, ClientId clientId, Collection<Long> creativeIds,
                                                @Nullable Collection<CreativeType> types) {
        return getCreatives(shard, creativeIds, clientId, types);
    }

    public List<Creative> getPerfCreativesWithBusinessType(int shard, ClientId clientId,
                                                           Set<CreativeBusinessType> types,
                                                           @Nullable String idOrNameString) {
        Condition idOrNameLikeCondition = DSL.trueCondition();
        if (!StringUtils.isEmpty(idOrNameString)) {

            idOrNameLikeCondition = PERF_CREATIVES.NAME.contains(idOrNameString)
                    .or(PERF_CREATIVES.GROUP_NAME.contains(idOrNameString))
                    .or(PERF_CREATIVES.CREATIVE_ID.cast(String.class).contains(idOrNameString))
                    .or(PERF_CREATIVES.CREATIVE_GROUP_ID.cast(String.class).contains(idOrNameString));
        }

        return dslContextProvider.ppc(shard)
                .select(creativeJooqMapper.getFieldsToRead())
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()))
                .and(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.performance))
                .and(PERF_CREATIVES.BUSINESS_TYPE.in(types.stream().map(CreativeBusinessType::toSource).collect(toSet())))
                .and(idOrNameLikeCondition)
                .fetch(creativeJooqMapper::fromDb);
    }

    public List<Creative> getCreatives(int shard, Collection<Long> creativeIds) {
        return getCreatives(shard, creativeIds, null, null);
    }

    public List<Creative> getCreatives(int shard, ClientId clientId, Collection<Long> creativeIds) {
        return getCreatives(shard, creativeIds, clientId, null);
    }

    private List<Creative> getCreatives(int shard, Collection<Long> ids, @Nullable ClientId clientId,
                                        @Nullable Collection<CreativeType> types) {
        if (ids.isEmpty()) {
            return emptyList();
        }

        SelectConditionStep<Record> conditionStep = dslContextProvider.ppc(shard)
                .select(creativeJooqMapper.getFieldsToRead())
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CREATIVE_ID.in(ids));

        if (clientId != null) {
            conditionStep = conditionStep.and(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()));
        }
        if (types != null) {
            conditionStep = conditionStep.and(typesCondition(types));
        }

        return conditionStep.fetch(creativeJooqMapper::fromDb);
    }

    public Map<Long, List<Creative>> getCreativesByPerformanceAdGroupIds(int shard, ClientId clientId,
                                                                         @Nullable Collection<Long> campaignIds,
                                                                         @Nullable Collection<Long> adGroupIds) {
        if (campaignIds == null && adGroupIds == null) {
            return emptyMap();
        }
        Set<Field<?>> fieldsToRead = new HashSet<>(creativeJooqMapper.getFieldsToRead());
        fieldsToRead.add(BANNERS_PERFORMANCE.PID);
        SelectConditionStep<Record> conditionStep = dslContextProvider.ppc(shard)
                .selectDistinct(fieldsToRead)
                .from(PERF_CREATIVES)
                .join(BANNERS_PERFORMANCE)
                .on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .join(BANNERS)
                .on(BANNERS.BID.eq(BANNERS_PERFORMANCE.BID))
                .where(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()));

        if (campaignIds != null) {
            conditionStep = conditionStep.and(BANNERS.CID.in(campaignIds));
        }

        if (adGroupIds != null) {
            conditionStep = conditionStep.and(BANNERS.PID.in(adGroupIds));
        }

        return conditionStep
                .fetchGroups(BANNERS_PERFORMANCE.PID, creativeJooqMapper::fromDb);
    }


    /**
     * Возвращает отображения Id баннера в привязанный креатив
     *
     * @param shard     шард
     * @param bannerIds коллекция id баннеров
     * @return key - bannerId, value - креативов
     */
    public Map<Long, Creative> getCreativesByBannerIds(int shard, Collection<Long> bannerIds) {
        return getCreativesByBannerIds(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Возвращает отображения Id баннера в привязанный креатив
     *
     * @param dslContext контекст
     * @param bannerIds  коллекция id баннеров
     * @return key - bannerId, value - креативов
     */
    public Map<Long, Creative> getCreativesByBannerIds(DSLContext dslContext, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }
        return dslContext
                .select(creativeJooqMapper.getFieldsToRead())
                .select(BANNERS_PERFORMANCE.BID)
                .from(BANNERS_PERFORMANCE)
                .join(PERF_CREATIVES)
                .on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .fetchMap(BANNERS_PERFORMANCE.BID, creativeJooqMapper::fromDb);
    }

    public Map<Long, CreativeInfo> getCreativeInfoByBannerId(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(BANNERS_PERFORMANCE.BID, PERF_CREATIVES.CREATIVE_TYPE, PERF_CREATIVES.LAYOUT_ID,
                        PERF_CREATIVES.EXPANDED_PREVIEW_URL)
                .from(BANNERS_PERFORMANCE)
                .join(PERF_CREATIVES)
                .on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .fetchMap(BANNERS_PERFORMANCE.BID,
                        rec -> new CreativeInfo(
                                CreativeMappings.creativeTypeByTypeAndLayoutIdFromDb(
                                        rec.get(PERF_CREATIVES.CREATIVE_TYPE),
                                        rec.get(PERF_CREATIVES.LAYOUT_ID)),
                                rec.get(PERF_CREATIVES.EXPANDED_PREVIEW_URL) != null,
                                rec.get(PERF_CREATIVES.LAYOUT_ID))
                );
    }

    private void prepareAddCreatives(Collection<Creative> creatives) {
        // всем креативам, кроме видео-дополнений выставляем stockCreativeId равный creativeId
        creatives.stream()
                .filter(cr -> !CreativeConstants.VIDEO_TYPES.contains(cr.getType()))
                .forEach(cr -> cr
                        .withStockCreativeId(cr.getId()));

        creatives.stream()
                .filter(cr -> cr.getIsGenerated() == null)
                .forEach(cr -> cr.withIsGenerated(false));
    }

    /**
     * Возвращает флаг - есть/нет турболендинги у клиента (любые, включая не привязанные к баннерам или сайтлинкам)
     *
     * @param shard    номер шарда
     * @param clientId идентификатор клиента
     */
    public boolean clientAlreadyHasHtml5Creatives(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(DSL.val(1))
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()))
                .and(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.html5_creative))
                .limit(1)
                .fetchAny() != null;
    }

    public void updateCreativesGeo(int shard, Map<Long, String> geoByCreativeId) {
        updateCreativesGeo(dslContextProvider.ppc(shard), geoByCreativeId);
    }

    /**
     * Пересчитывает и перезаписывает гео у креативов-черновиков и отклонённых
     *
     * @param context         контекст
     * @param geoByCreativeId строка гео по ID креатива
     */
    public void updateCreativesGeo(DSLContext context, Map<Long, String> geoByCreativeId) {
        if (geoByCreativeId.isEmpty()) {
            return;
        }
        context.update(PERF_CREATIVES)
                .set(PERF_CREATIVES.SUM_GEO,
                        choose(PERF_CREATIVES.CREATIVE_ID).mapValues(geoByCreativeId).otherwise(PERF_CREATIVES.SUM_GEO))
                .where(PERF_CREATIVES.CREATIVE_ID.in(geoByCreativeId.keySet()))
                .and(PERF_CREATIVES.STATUS_MODERATE.in(PerfCreativesStatusmoderate.New,
                        PerfCreativesStatusmoderate.Error, PerfCreativesStatusmoderate.No))
                .execute();
    }

    public void sendCreativesToModeration(int shard, Collection<Long> creativeIds) {
        sendCreativesToModeration(dslContextProvider.ppc(shard), creativeIds);
    }

    public void sendCreativesToModeration(DSLContext dslContext, Collection<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            return;
        }
        LocalDateTime moderateSendTime = LocalDateTime.now().plusMinutes(ADVANCE_MINUTES);

        dslContext
                .update(PERF_CREATIVES)
                .set(PERF_CREATIVES.STATUS_MODERATE, PerfCreativesStatusmoderate.Ready)
                .set(PERF_CREATIVES.MODERATE_TRY_COUNT, 0L)
                // Это актуально только для performance и bannerstorage креативов
                .setNull(PERF_CREATIVES.MODERATION_COMMENT)
                .set(PERF_CREATIVES.MODERATE_SEND_TIME, moderateSendTime)
                .where(PERF_CREATIVES.CREATIVE_ID.in(creativeIds))
                .and(PERF_CREATIVES.STATUS_MODERATE.in(PerfCreativesStatusmoderate.New,
                        PerfCreativesStatusmoderate.Error))
                .execute();
    }

    public void sendRejectedCreativesToModeration(DSLContext dslContext, Collection<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            return;
        }
        LocalDateTime moderateSendTime = LocalDateTime.now().plusMinutes(ADVANCE_MINUTES);

        dslContext
                .update(PERF_CREATIVES)
                .set(PERF_CREATIVES.STATUS_MODERATE, PerfCreativesStatusmoderate.Ready)
                .set(PERF_CREATIVES.MODERATE_TRY_COUNT, 0L)
                // Это актуально только для performance и bannerstorage креативов
                .setNull(PERF_CREATIVES.MODERATION_COMMENT)
                .set(PERF_CREATIVES.MODERATE_SEND_TIME, moderateSendTime)
                .where(PERF_CREATIVES.CREATIVE_ID.in(creativeIds))
                .and(PERF_CREATIVES.STATUS_MODERATE.eq(PerfCreativesStatusmoderate.No))
                .execute();
    }

    /**
     * Добавляет таски на синхронизацию скриншотов из bannerstorage
     * (портировано из функции перла create_creative_sync_tasks)
     */
    public void addScreenshotsSyncTaskForPerformance(DSLContext dslContext, Collection<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            return;
        }

        var insertQuery = dslContext.insertQuery(CREATIVE_BANNER_STORAGE_SYNC);
        for (Long creativeId : creativeIds) {
            Map<TableField<CreativeBannerStorageSyncRecord, ?>, ?> fieldValues = Map.of(
                    CREATIVE_BANNER_STORAGE_SYNC.CREATIVE_ID, creativeId,
                    CREATIVE_BANNER_STORAGE_SYNC.SYNC_STATUS, CreativeBannerStorageSyncSyncStatus.New,
                    CREATIVE_BANNER_STORAGE_SYNC.ATTEMPTS, 0L,
                    CREATIVE_BANNER_STORAGE_SYNC.FULL_SYNC, 0L
            );
            insertQuery.addValues(fieldValues);
            insertQuery.newRecord();
        }
        insertQuery.onDuplicateKeyUpdate(true);
        insertQuery.addValueForUpdate(CREATIVE_BANNER_STORAGE_SYNC.SYNC_STATUS,
                CreativeBannerStorageSyncSyncStatus.New);
        insertQuery.addValueForUpdate(CREATIVE_BANNER_STORAGE_SYNC.ATTEMPTS, 0L);
        insertQuery.addValueForUpdate(CREATIVE_BANNER_STORAGE_SYNC.FULL_SYNC, 0L);
        insertQuery.execute();
    }

    private JooqMapperWithSupplier<Creative> createCreativeMapper() {
        return JooqMapperWithSupplierBuilder.builder(Creative::new)
                .map(property(Creative.ID, PERF_CREATIVES.CREATIVE_ID))
                .map(property(Creative.NAME, PERF_CREATIVES.NAME))
                .map(property(Creative.CLIENT_ID, PERF_CREATIVES.CLIENT_ID))
                .map(property(Creative.HEIGHT, PERF_CREATIVES.HEIGHT))
                .map(property(Creative.WIDTH, PERF_CREATIVES.WIDTH))
                .map(property(Creative.PREVIEW_URL, PERF_CREATIVES.PREVIEW_URL))
                .map(property(Creative.LIVE_PREVIEW_URL, PERF_CREATIVES.LIVE_PREVIEW_URL))
                .map(property(Creative.EXPANDED_PREVIEW_URL, PERF_CREATIVES.EXPANDED_PREVIEW_URL))
                .map(property(Creative.ARCHIVE_URL, PERF_CREATIVES.ARCHIVE_URL))
                .map(property(Creative.MODERATE_TRY_COUNT, PERF_CREATIVES.MODERATE_TRY_COUNT))
                .map(property(Creative.STOCK_CREATIVE_ID, PERF_CREATIVES.STOCK_CREATIVE_ID))
                .map(property(Creative.DURATION, PERF_CREATIVES.DURATION))

                .readProperty(Creative.TYPE, fromFields(PERF_CREATIVES.CREATIVE_TYPE, PERF_CREATIVES.LAYOUT_ID)
                        .by(CreativeMappings::creativeTypeByTypeAndLayoutIdFromDb))
                .writeField(PERF_CREATIVES.CREATIVE_TYPE,
                        fromProperty(Creative.TYPE).by(CreativeMappings::creativeTypeToDb))
                .map(property(Creative.LAYOUT_ID, PERF_CREATIVES.LAYOUT_ID))
                .map(property(Creative.TEMPLATE_ID, PERF_CREATIVES.TEMPLATE_ID))
                .map(property(Creative.VERSION, PERF_CREATIVES.VERSION))

                .map(booleanProperty(Creative.IS_GENERATED, PERF_CREATIVES.IS_GENERATED))
                .map(convertibleProperty(Creative.SOURCE_MEDIA_TYPE, PERF_CREATIVES.SOURCE_MEDIA_TYPE,
                        SourceMediaType::fromSource, SourceMediaType::toSource))
                .map(convertibleProperty(Creative.STATUS_MODERATE, PERF_CREATIVES.STATUS_MODERATE,
                        StatusModerate::fromSource, StatusModerate::toSource))
                .map(convertibleProperty(Creative.MODERATION_INFO, PERF_CREATIVES.MODERATE_INFO,
                        CreativeMappings::moderationInfoFromDb, CreativeMappings::moderationInfoToDb))
                .map(convertibleProperty(Creative.YABS_DATA, PERF_CREATIVES.YABS_DATA,
                        CreativeMappings::yabsDataFromDb, CreativeMappings::yabsDataToDb))
                .map(convertibleProperty(Creative.BUSINESS_TYPE, PERF_CREATIVES.BUSINESS_TYPE,
                        CreativeBusinessType::fromSource, value -> value != null
                                ? CreativeBusinessType.toSource(value)
                                : PerfCreativesBusinessType.retail))  // use database default if null
                .readProperty(Creative.SUM_GEO, fromField(PERF_CREATIVES.SUM_GEO)
                        .by(CreativeMappings::sumGeoFromDb))
                .map(property(Creative.THEME_ID, PERF_CREATIVES.THEME_ID))
                .map(property(Creative.GROUP_NAME, PERF_CREATIVES.GROUP_NAME))
                .map(convertibleProperty(Creative.ADDITIONAL_DATA, PERF_CREATIVES.ADDITIONAL_DATA,
                        CreativeMappings::additionalDataFromDb, CreativeMappings::additionalDataToDb))
                .map(booleanProperty(Creative.HAS_PACKSHOT, PERF_CREATIVES.HAS_PACKSHOT))
                .map(booleanProperty(Creative.IS_ADAPTIVE, PERF_CREATIVES.IS_ADAPTIVE))
                .map(booleanProperty(Creative.IS_BRAND_LIFT, PERF_CREATIVES.IS_BRAND_LIFT))
                .map(property(Creative.CREATIVE_GROUP_ID, PERF_CREATIVES.CREATIVE_GROUP_ID))
                .map(convertibleProperty(Creative.IS_BANNERSTORAGE_PREDEPLOYED,
                        PERF_CREATIVES.IS_BANNERSTORAGE_PREDEPLOYED,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong))
                .build();
    }

    /**
     * Условие запроса по типам с layoutId. Если не указан ни один тип, то условие не ограничивает по типу (true)
     */
    private static Condition typesCondition(Collection<CreativeType> types) {
        if (types.isEmpty()) {
            return DSL.trueCondition();
        }

        Condition condition = DSL.falseCondition();
        if (types.contains(CreativeType.CANVAS)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.canvas));
        }
        if (types.contains(CreativeType.HTML5_CREATIVE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.html5_creative));
        }
        if (types.contains(CreativeType.PERFORMANCE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.performance));
        }
        if (types.contains(CreativeType.BANNERSTORAGE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.bannerstorage));
        }
        if (types.contains(CreativeType.VIDEO_ADDITION_CREATIVE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(TEXT_VIDEO_ADDITION_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(TEXT_VIDEO_ADDITION_LAYOUT_IDS.upperEndpoint())));
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(TEXT_VIDEO_ADDITION_TALL_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(TEXT_VIDEO_ADDITION_TALL_LAYOUT_IDS.upperEndpoint())));
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(MOBILE_CONTENT_VIDEO_ADDITION_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(MOBILE_CONTENT_VIDEO_ADDITION_LAYOUT_IDS.upperEndpoint())));
        }
        if (types.contains(CreativeType.CPC_VIDEO_CREATIVE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(CPC_VIDEO_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(CPC_VIDEO_LAYOUT_IDS.upperEndpoint())));
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(MOBILE_CPC_VIDEO_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(MOBILE_CPC_VIDEO_LAYOUT_IDS.upperEndpoint())));
        }
        if (types.contains(CreativeType.CPM_VIDEO_CREATIVE)) {
            List<Condition> layoutCondition = new ArrayList<>();

            for (var cpmRange : CPM_VIDEO_ADDITION_LAYOUT_IDS.asRanges()) {
                layoutCondition.add(DSL.and(PERF_CREATIVES.LAYOUT_ID.ge(cpmRange.lowerEndpoint()),
                        PERF_CREATIVES.LAYOUT_ID.le(cpmRange.upperEndpoint())));
            }

            condition =
                    condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition).and(DSL.or(layoutCondition)));
        }
        if (types.contains(CreativeType.CPM_OUTDOOR_CREATIVE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(CPM_OUTDOOR_VIDEO_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(CPM_OUTDOOR_VIDEO_LAYOUT_IDS.upperEndpoint())));
        }
        if (types.contains(CreativeType.CPM_INDOOR_CREATIVE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(CPM_INDOOR_VIDEO_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(CPM_INDOOR_VIDEO_LAYOUT_IDS.upperEndpoint())));
        }
        if (types.contains(CreativeType.CPM_AUDIO_CREATIVE)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(CPM_AUDIO_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(CPM_AUDIO_LAYOUT_IDS.upperEndpoint())));
        }
        if (types.contains(CreativeType.CPM_OVERLAY)) {
            condition = condition.or(PERF_CREATIVES.CREATIVE_TYPE.eq(PerfCreativesCreativeType.video_addition)
                    .and(PERF_CREATIVES.LAYOUT_ID.ge(CPM_OVERLAY_LAYOUT_IDS.lowerEndpoint()))
                    .and(PERF_CREATIVES.LAYOUT_ID.le(CPM_OVERLAY_LAYOUT_IDS.upperEndpoint())));
        }

        return condition;
    }

    public List<Creative> getCreatives(int shard, ClientId clientId, CreativeFilterContainer filterContainer) {
        SelectConditionStep<Record> conditionStep = dslContextProvider.ppc(shard)
                .select(creativeJooqMapper.getFieldsToRead())
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.CLIENT_ID.eq(clientId.asLong()));

        if (CollectionUtils.isNotEmpty(filterContainer.getCreativeIds())) {
            conditionStep = conditionStep.and(PERF_CREATIVES.CREATIVE_ID.in(filterContainer.getCreativeIds()));
        }
        if (CollectionUtils.isNotEmpty(filterContainer.getCreativeTypes())) {
            conditionStep = conditionStep.and(typesCondition(filterContainer.getCreativeTypes()));
        }

        return conditionStep.fetch(creativeJooqMapper::fromDb);
    }

    public List<Long> getAllCreativeIds(
            int shardId, Long startId, @Nullable Long endId, List<CreativeType> creativeTypes, Integer chunkSize) {
        SelectConditionStep<Record1<Long>> conditionStep =
                dslContextProvider.ppc(shardId)
                        .select(PERF_CREATIVES.CREATIVE_ID)
                        .from(PERF_CREATIVES)
                        .where(PERF_CREATIVES.CREATIVE_ID.eq(PERF_CREATIVES.STOCK_CREATIVE_ID))
                        .and(typesCondition(creativeTypes))
                        .and(PERF_CREATIVES.CREATIVE_ID.greaterOrEqual(startId));

        if (endId != null) {
            conditionStep.and(PERF_CREATIVES.CREATIVE_ID.lessThan(endId));
        }

        return conditionStep
                .limit(chunkSize)
                .fetch(Record1::value1);
    }

    /**
     * Из предложенных id креатива возвращает только те, которые связаны с каким-нибудь баннером
     */
    public Set<Long> selectCreativesLinkedWithBanners(int shard, Collection<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            return Collections.emptySet();
        }

        return dslContextProvider.ppc(shard)
                .select(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES)
                .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(PERF_CREATIVES.CREATIVE_ID.in(creativeIds))
                .fetchSet(PERF_CREATIVES.CREATIVE_ID);
    }

    /**
     * Возвращает креативы bannerstorage, которые в статусе Ready, но не имеют ни одного связанного баннера,
     * и поэтому не могут отправиться в новую Модерацию
     */
    public List<Long> getModeratingBannerstorageCreativesWithoutBanners(int shard, int limit) {
        return dslContextProvider.ppc(shard)
                .select(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES)
                .leftJoin(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(PERF_CREATIVES.STATUS_MODERATE.in(
                        PerfCreativesStatusmoderate.Ready,
                        PerfCreativesStatusmoderate.Sent
                ))
                .and(PERF_CREATIVES.CREATIVE_TYPE.in(
                        PerfCreativesCreativeType.performance
                ))
                .and(BANNERS_PERFORMANCE.CREATIVE_ID.isNull())
                .orderBy(PERF_CREATIVES.CREATIVE_ID)
                .limit(limit)
                .fetch(PERF_CREATIVES.CREATIVE_ID);
    }

    /**
     * Возвращает креативы bannerstorage, которые можно считать "зависшими" на модерации
     * Такое может произойти например, если вердикт запрашивался в связке с баннером, который был позже удалён
     */
    public List<Long> getBannerstorageCreativesStaleInModeration(int shard, int hours, int limit) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES)
                .join(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(PERF_CREATIVES.STATUS_MODERATE.in(
                        PerfCreativesStatusmoderate.Ready,
                        PerfCreativesStatusmoderate.Sent
                ))
                .and(PERF_CREATIVES.CREATIVE_TYPE.in(
                        PerfCreativesCreativeType.performance
                ))
                .and(PERF_CREATIVES.MODERATE_SEND_TIME.lessThan(
                        DSL.localDateTimeSub(DSL.currentLocalDateTime(), hours, DatePart.HOUR)
                ))
                .orderBy(PERF_CREATIVES.CREATIVE_ID)
                .limit(limit)
                .fetch(PERF_CREATIVES.CREATIVE_ID);
    }

    public void setStatusModerate(int shard, Collection<Long> creativeIds, PerfCreativesStatusmoderate statusModerate) {
        if (creativeIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard)
                .update(PERF_CREATIVES)
                .set(PERF_CREATIVES.STATUS_MODERATE, statusModerate)
                .where(PERF_CREATIVES.CREATIVE_ID.in(creativeIds))
                .execute();
    }

    public List<Long> getCreativeIdsWithoutScreenshoot(int shardId, Integer chunkSize) {
        return dslContextProvider.ppc(shardId)
                .select(PERF_CREATIVES.CREATIVE_ID)
                .from(PERF_CREATIVES)
                .where(PERF_CREATIVES.MODERATE_SEND_TIME.greaterOrEqual(
                        DSL.localDateTime(LocalDateTime.of(2021, 1, 1, 0, 0))
                ))
                .and(PERF_CREATIVES.PREVIEW_URL.like("%getScreenshot%"))
                .orderBy(PERF_CREATIVES.CREATIVE_ID.desc())
                .limit(chunkSize)
                .fetch(PERF_CREATIVES.CREATIVE_ID);
    }
}
