package ru.yandex.direct.core.entity.creative.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException;
import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.bannerstorage.client.model.CreativeGroup;
import ru.yandex.direct.bannerstorage.client.model.Parameter;
import ru.yandex.direct.bannerstorage.client.model.Template;
import ru.yandex.direct.bannerstorage.client.model.TemplateInclude;
import ru.yandex.direct.bannerstorage.client.model.TnsArticle;
import ru.yandex.direct.bannerstorage.client.model.TnsBrand;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.SmartTheme;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroupsItem;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesBusinessType;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class BannerStorageCreativeService {
    private static final Logger logger = LoggerFactory.getLogger(BannerStorageCreativeService.class);

    private static final Map<SmartTheme, Integer> THEME_TO_BS_TEMPLATE_ID = Map.of(
            SmartTheme.RETAIL, 740,
            SmartTheme.HOTELS, 741,
            SmartTheme.REALTY, 778,
            SmartTheme.AUTOMOBILES, 779,
            SmartTheme.AIRLINE_TICKETS, 783,
            SmartTheme.CLOTHES, 839,
            SmartTheme.OTHER, 910,
            SmartTheme.PHARM, 1051
    );
    public static final List<TnsArticle> TNS_ARTICLES = List.of(new TnsArticle(-3));
    public static final List<TnsBrand> TNS_BRANDS = List.of(new TnsBrand(1));

    private final BannerStorageClient bannerStorageClient;
    private final BannersAddOperationFactory bannersAddOperationFactory;
    private final CreativeService creativeService;

    @Autowired
    public BannerStorageCreativeService(BannerStorageClient bannerStorageClient,
            BannersAddOperationFactory bannersAddOperationFactory,
            CreativeService creativeService) {
        this.bannerStorageClient = bannerStorageClient;
        this.bannersAddOperationFactory = bannersAddOperationFactory;
        this.creativeService = creativeService;
    }

    public boolean updateSmartAdGroupCreativeGroupsItem(ClientId clientId, Long operatorUid, boolean saveDraft,
            UpdateSmartAdGroupCreativeGroupsItem item,
            @Nullable Collection<Long> existingCreativeGroupIds) {
        long adGroupId = item.getAdGroupId();

        var parameters = List.of(
                new Parameter()
                        .withParamName("LOGO")
                        .withValues(StreamEx.ofNullable(item.getLogoFileId())
                                .map(Objects::toString)
                                .toList()),
                new Parameter() // пока что обязательный, но не используемый в показах параметр
                        .withParamName("DOMAIN_LIST")
                        .withValues(List.of(nvl(item.getDomain(), "example.com"))));

        if (!isEmpty(existingCreativeGroupIds)) {
            return updateSmartCreativeGroups(clientId, existingCreativeGroupIds, parameters);
        }

        int templateId = toBsTemplateId(item.getTheme());
        return createSmartCreativeGroupForAdGroup(clientId, operatorUid, saveDraft, adGroupId, templateId, parameters);
    }

    private boolean createSmartCreativeGroupForAdGroup(ClientId clientId, Long operatorUid, boolean saveDraft,
            Long adGroupId, Integer templateId, List<Parameter> parameters) {
        var template = bannerStorageClient.getTemplate(templateId, TemplateInclude.LAYOUT_CODES);
        var creatives = StreamEx.of(template)
                .flatCollection(Template::getLayoutCodes)
                .map(layoutCode -> new Creative()
                        .withTemplateId(template.getId())
                        .withLayoutCodeId(layoutCode.getId())
                        .withParameters(parameters)
                        .withTnsArticles(TNS_ARTICLES)
                        .withTnsBrands(TNS_BRANDS))
                .toList();
        CreativeGroup creativeGroup;
        try {
            creativeGroup = bannerStorageClient.createSmartCreativeGroup(
                    new CreativeGroup(null, "Новая группа", creatives));
        } catch (BannerStorageClientException e) {
            logger.error("Failed to create creative group", e);
            return false;
        }
        var creativeGroupId = Objects.requireNonNull(creativeGroup.getId()).longValue();
        var coreCreatives = StreamEx.of(creativeGroup)
                .flatCollection(CreativeGroup::getCreatives)
                .map(creative -> new ru.yandex.direct.core.entity.creative.model.Creative()
                        .withId(creative.getId().longValue())
                        .withClientId(clientId.asLong())
                        .withName(creative.getName())
                        .withType(CreativeType.PERFORMANCE)
                        .withStatusModerate(StatusModerate.NEW)
                        .withPreviewUrl(nvl(creative.getScreenshotUrl(), creative.getThumbnailUrl()))
                        .withLivePreviewUrl(creative.getPreview().getUrl())
                        .withWidth(creative.getWidth().longValue())
                        .withHeight(creative.getHeight().longValue())
                        .withStockCreativeId(creative.getId().longValue())
                        .withLayoutId((long) creative.getLayoutCode().getLayoutId())
                        .withTemplateId(creative.getTemplateId().longValue())
                        .withVersion(creative.getVersion().longValue())
                        .withThemeId((long) creative.getLayoutCode().getThemeId())
                        .withBusinessType(CreativeBusinessType.fromSource(
                                PerfCreativesBusinessType.valueOf(
                                        creative.getBusinessType().getDirectId())))
                        .withGroupName(creativeGroup.getName())
                        .withCreativeGroupId(creativeGroupId)
                        .withIsBannerstoragePredeployed(creative.isPredeployed())
                        .withIsAdaptive(false)
                        .withIsBrandLift(false)
                        .withHasPackshot(false)
                        .withModerateTryCount(0L))
                .toList();
        var creativesResult = creativeService.createOrUpdate(coreCreatives, clientId);
        if (!isEmpty(creativesResult.getErrors())) {
            logger.error("Failed to save creatives {}", creativesResult.getErrors());
        }
        if (!creativesResult.isSuccessful()) {
            return false;
        }
        var banners = StreamEx.of(coreCreatives)
                .map(creative -> (BannerWithAdGroupId)
                        new PerformanceBanner()
                                .withAdGroupId(adGroupId)
                                .withCreativeId(creative.getId()))
                .toList();
        var result = bannersAddOperationFactory.createFullAddOperation(banners, clientId, operatorUid, saveDraft)
                .prepareAndApply();
        if (!result.isSuccessful()) {
            logger.error("Failed to save banners {}", result.getErrors());
            return false;
        }
        return true;
    }

    private boolean updateSmartCreativeGroups(ClientId clientId, Collection<Long> creativeGroupIds,
            List<Parameter> parameters) {
        for (var creativeGroupId : creativeGroupIds) {
            var existingCreativeGroup =
                    bannerStorageClient.getSmartCreativeGroup(creativeGroupId.intValue());
            var creatives = StreamEx.of(existingCreativeGroup)
                    .flatCollection(CreativeGroup::getCreatives)
                    .map(creative -> new Creative()
                            .withId(creative.getId())
                            .withVersion(creative.getVersion())
                            .withParameters(parameters)
                            .withTnsArticles(TNS_ARTICLES)
                            .withTnsBrands(TNS_BRANDS))
                    .toList();
            CreativeGroup creativeGroup;
            try {
                creativeGroup = bannerStorageClient.editSmartCreativeGroup(creativeGroupId.intValue(),
                        new CreativeGroup(existingCreativeGroup.getId(), existingCreativeGroup.getName(), creatives));
            } catch (BannerStorageClientException e) {
                logger.error("Failed to edit creative group", e);
                return false;
            }
            var creativeById = listToMap(creativeGroup.getCreatives(),
                    creative -> creative.getId().longValue());
            var existingCreatives = creativeService.get(clientId, creativeById.keySet(),
                    List.of(CreativeType.PERFORMANCE));
            existingCreatives.forEach(existingCreative -> {
                var creative = creativeById.get(existingCreative.getId());
                existingCreative
                        .withName(creative.getName())
                        .withGroupName(creativeGroup.getName())
                        .withWidth(creative.getWidth().longValue())
                        .withHeight(creative.getHeight().longValue())
                        .withPreviewUrl(nvl(creative.getScreenshotUrl(), creative.getThumbnailUrl()))
                        .withLivePreviewUrl(creative.getPreview().getUrl())
                        .withVersion(creative.getVersion().longValue())
                        .withModerationInfo(null)
                        .withIsBannerstoragePredeployed(creative.isPredeployed())
                        .withStatusModerate(StatusModerate.NEW);
            });
            var result = creativeService.createOrUpdate(existingCreatives, clientId);
            if (!isEmpty(result.getErrors())) {
                logger.error("Failed to update creatives {}", result.getErrors());
            }
            if (!result.isSuccessful()) {
                return false;
            }
        }
        return true;
    }

    private int toBsTemplateId(SmartTheme theme) {
        var templateId = THEME_TO_BS_TEMPLATE_ID.get(theme);
        if (templateId == null) {
            throw new IllegalArgumentException("unknown theme: " + theme);
        }
        return templateId;
    }
}
