package ru.yandex.direct.core.entity.creative.service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException;
import ru.yandex.direct.bannerstorage.client.model.File;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroups;
import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroupsItem;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validIntegerId;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class BannerStorageCreativeValidationService {
    private static final int MIN_IMAGE_SIZE = 30;
    private static final int MAX_IMAGE_SIZE = 800;

    private final AdGroupService adGroupService;
    private final BannerStorageClient bannerStorageClient;

    @Autowired
    public BannerStorageCreativeValidationService(AdGroupService adGroupService,
            BannerStorageClient bannerStorageClient) {
        this.adGroupService = adGroupService;
        this.bannerStorageClient = bannerStorageClient;
    }

    public ValidationResult<UpdateSmartAdGroupCreativeGroups, Defect> validateAdGroupCreativeGroupsUpdateRequest(
            UpdateSmartAdGroupCreativeGroups request,
            ClientId clientId,
            Map<Long, List<Creative>> creativesByAdGroupId) {
        List<Long> adGroupIds = mapList(request.getUpdateItems(), UpdateSmartAdGroupCreativeGroupsItem::getAdGroupId);
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, adGroupIds);
        Set<Long> existingAdGroupIds = adGroupTypes.keySet();
        Set<Long> performanceAdGroupIds = EntryStream.of(adGroupTypes)
                .filterValues(v -> v == AdGroupType.PERFORMANCE)
                .keys()
                .toSet();
        Map<Integer, File> fileById = StreamEx.of(request.getUpdateItems())
                .map(UpdateSmartAdGroupCreativeGroupsItem::getLogoFileId)
                .distinct()
                .nonNull()
                .mapToEntry(fileId -> {
                    try {
                        return bannerStorageClient.getFile(fileId);
                    } catch (BannerStorageClientException e) { // not found
                        return null;
                    }
                })
                .nonNullValues()
                .toMap();
        ModelItemValidationBuilder<UpdateSmartAdGroupCreativeGroups> vb = ModelItemValidationBuilder.of(request);
        vb.list(UpdateSmartAdGroupCreativeGroups.UPDATE_ITEMS)
                .check(maxListSize(1)) // ограничение из-за долгих запросов в BannerStorage
                .checkEachBy(item -> validateAdGroupCreativeGroupsUpdateItem(item, existingAdGroupIds,
                        performanceAdGroupIds, fileById))
                .check(uniqueCreativeGroups(creativesByAdGroupId));
        return vb.getResult();
    }

    private ValidationResult<UpdateSmartAdGroupCreativeGroupsItem, Defect> validateAdGroupCreativeGroupsUpdateItem(
            UpdateSmartAdGroupCreativeGroupsItem item,
            Set<Long> existingAdGroupIds,
            Set<Long> performanceAdGroupIds,
            Map<Integer, File> fileById) {
        ModelItemValidationBuilder<UpdateSmartAdGroupCreativeGroupsItem> vb = ModelItemValidationBuilder.of(item);
        vb.item(UpdateSmartAdGroupCreativeGroupsItem.AD_GROUP_ID)
                .check(validId())
                .check(inSet(existingAdGroupIds), objectNotFound(), When.isValid())
                .check(inSet(performanceAdGroupIds), AdGroupDefects.unsupportedAdGroupType(), When.isValid());
        vb.item(UpdateSmartAdGroupCreativeGroupsItem.LOGO_FILE_ID)
                .check(validIntegerId(), When.notNull())
                .check(inSet(fileById.keySet()), objectNotFound(), When.notNullAnd(When.isValid()))
                .check(validImage(fileById), When.isValid());
        return vb.getResult();
    }

    private static Constraint<List<UpdateSmartAdGroupCreativeGroupsItem>, Defect> uniqueCreativeGroups(
            Map<Long, List<Creative>> creativesByAdGroupId) {
        return Constraint.fromPredicate(items -> {
            List<Long> creativeGroupIds = StreamEx.of(items)
                    .map(UpdateSmartAdGroupCreativeGroupsItem::getAdGroupId)
                    .map(creativesByAdGroupId::get)
                    .flatCollection(creatives ->
                            mapAndFilterToSet(creatives, Creative::getCreativeGroupId, Objects::nonNull))
                    .toList();
            Set<Long> uniqueCreativeGroupIds = new HashSet<>();
            for (Long creativeGroupId : creativeGroupIds) {
                if (!uniqueCreativeGroupIds.add(creativeGroupId)) {
                    return false;
                }
            }
            return true;
        }, AdGroupDefects.mutuallyExclusive());
    }

    private static Constraint<Integer, Defect> validImage(Map<Integer, File> fileById) {
        return Constraint.fromPredicate(fileId -> {
            File file = fileById.get(fileId);
            Integer height = file.getHeight();
            Integer width = file.getWidth();
            return file.getDuration() == null // не видео
                    && height != null && height >= MIN_IMAGE_SIZE && height <= MAX_IMAGE_SIZE
                    && width != null && width >= MIN_IMAGE_SIZE && width <= MAX_IMAGE_SIZE;
        }, invalidValue());
    }
}
