package ru.yandex.direct.core.entity.creative.service;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.canvas.client.CanvasClient;
import ru.yandex.direct.canvas.client.model.exception.CanvasClientException;
import ru.yandex.direct.canvas.client.model.video.CreativeResponse;
import ru.yandex.direct.canvas.client.model.video.UcCreative;
import ru.yandex.direct.canvas.client.model.video.UcCreativeResponse;
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeConverter;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.SourceMediaType;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.creative.repository.CreativeConstants;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.creative.service.add.validation.CreativeValidationService;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.creative.repository.CreativeMappings.creativeBusinessTypesFromBusinessType;
import static ru.yandex.direct.core.entity.creative.service.add.validation.CreativeDefects.createAdditionFromVideoFailed;
import static ru.yandex.direct.core.entity.creative.service.add.validation.CreativeDefects.uploadVideoFailed;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
@ParametersAreNonnullByDefault
public class CreativeService {
    private final static Logger logger = LoggerFactory.getLogger(CreativeService.class);

    // медиа типы из которых html5 конструтор генерит креативы сам (сейчас все типы)
    public static final Set<SourceMediaType> HTML_5_MEDIA_TYPES_FOR_GENERATED_CREATIVE = new HashSet<>(Arrays.asList(
            SourceMediaType.GIF, SourceMediaType.JPG, SourceMediaType.PNG
    ));

    public static final Set<CreativeType> ADDITIONAL_DATA_SUPPORTED_TYPES = ImmutableSet.of(
            CreativeType.CPM_OUTDOOR_CREATIVE,
            CreativeType.CPM_INDOOR_CREATIVE,
            CreativeType.PERFORMANCE
    );

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final CreativeRepository creativeRepository;
    private final CreativeValidationService creativeValidationService;
    private final CanvasClient canvasClient;
    private final BannerCreativeRepository bannerCreativeRepository;
    private final ClientGeoService clientGeoService;
    private final ModerationReasonRepository moderationReasonRepository;
    private final PpcProperty<Boolean> enableCreateEarlyCreativeProp;

    @Autowired
    public CreativeService(ShardHelper shardHelper,
                           DslContextProvider dslContextProvider,
                           CreativeRepository creativeRepository,
                           BannerCreativeRepository bannerCreativeRepository,
                           CreativeValidationService creativeValidationService, CanvasClient canvasClient,
                           ClientGeoService clientGeoService,
                           ModerationReasonRepository moderationReasonRepository,
                           PpcPropertiesSupport ppcPropertiesSupport) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.creativeRepository = creativeRepository;
        this.bannerCreativeRepository = bannerCreativeRepository;
        this.creativeValidationService = creativeValidationService;
        this.canvasClient = canvasClient;
        this.clientGeoService = clientGeoService;
        this.moderationReasonRepository = moderationReasonRepository;
        this.enableCreateEarlyCreativeProp = ppcPropertiesSupport.get(
                PpcPropertyNames.ENABLE_CREATE_EARLY_CREATIVE_IN_API,
                Duration.ofMinutes(5));
    }

    public List<Creative> get(ClientId clientId,
                              Collection<Long> creativeIds,
                              @Nullable Collection<CreativeType> types) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return creativeRepository.getCreativesWithTypes(shard, clientId, creativeIds, types);
    }

    public List<Creative> getVideoAdditionFromCanvasHandlingException(ClientId clientId,
                                                                      List<Long> creativeIds) {
        // фильтруем некорректные идентификаторы, чтобы не делать лишнее обращение в сторонний сервис
        List<Long> idsToRequest = filterList(creativeIds, Objects::nonNull);
        if (idsToRequest.isEmpty()) {
            return emptyList();
        }

        try {
            List<CreativeResponse> videoAdditions =
                    canvasClient.getVideoAdditions(clientId.asLong(), creativeIds);
            return StreamEx.of(videoAdditions)
                    .filter(CreativeResponse::getOk)
                    .map(CreativeResponse::getCreative)
                    .map(CreativeConverter::fromCanvasCreative)
                    .map(creative -> creative.withClientId(clientId.asLong())
                            .withModerateTryCount(0L)
                            .withStatusModerate(StatusModerate.YES))
                    .toList();
        } catch (CanvasClientException e) {
            return emptyList();
        }
    }

    public List<UcCreative> getUcCreatives(ClientId clientId,
                                           List<Long> creativeIds) {
        // фильтруем некорректные идентификаторы, чтобы не делать лишнее обращение в сторонний сервис
        List<Long> idsToRequest = filterList(creativeIds, Objects::nonNull);
        if (idsToRequest.isEmpty()) {
            return emptyList();
        }

        try {
            List<UcCreativeResponse> videoAdditions =
                    canvasClient.getUcCreativeData(clientId.asLong(), creativeIds);
            return StreamEx.of(videoAdditions)
                    .filter(UcCreativeResponse::getOk)
                    .map(UcCreativeResponse::getCreative)
                    .toList();
        } catch (CanvasClientException e) {
            logger.error("error getting uc creative data from canvas ", e);
            return emptyList();
        }
    }

    public static class VideoItemForUpload {
        private String url;
        private byte[] file;
        private String name;

        public VideoItemForUpload(String url, byte[] file, String name) {
            this.url = url;
            this.file = file;
            this.name = name;
        }

        public String getUrl() {
            return url;
        }

        public byte[] getFile() {
            return file;
        }

        public String getName() {
            return name;
        }
    }

    public static class VideoItem {
        private Long presetId;
        private String videoId;

        public VideoItem(Long presetId, String videoId) {
            this.presetId = presetId;
            this.videoId = videoId;
        }

        public VideoItem(String externalVideoId) {
            this.videoId = externalVideoId.substring(0, externalVideoId.length() - 3);
            this.presetId = Long.valueOf(externalVideoId.substring(externalVideoId.length() - 3));
        }

        public Long getPresetId() {
            return presetId;
        }

        public VideoItem withPresetId(Long presetId) {
            this.presetId = presetId;
            return this;
        }

        public String getVideoId() {
            return videoId;
        }

        public VideoItem withVideoId(String videoId) {
            this.videoId = videoId;
            return this;
        }

        public String getExternalVideoId() {
            return videoId + String.format("%03d", presetId);
        }
    }

    public static class VideoItemWithStatus extends VideoItem {
        private VideoUploadResponse.FileStatus status;

        public VideoItemWithStatus(Long presetId, String videoId, VideoUploadResponse.FileStatus status) {
            super(presetId, videoId);
            this.status = status;
        }

        public VideoUploadResponse.FileStatus getStatus() {
            return status;
        }

        public VideoItem withStatus(VideoUploadResponse.FileStatus status) {
            this.status = status;
            return this;
        }
    }

    public MassResult<VideoItem> createVideosForUpload(ClientId clientId,
                                                       List<VideoItemForUpload> videos) {
        ValidationResult<List<VideoItemForUpload>, Defect> validationResult =
                creativeValidationService.validateCreateVideosForUpload(videos);
        List<Result<VideoItem>> videoResults = new ArrayList<>(videos.size());

        int index = 0;
        for (var video : videos) {
            var validation = validationResult.getOrCreateSubValidationResult(index(index++), video);
            try {
                var videoUploadResponse = video.file == null ?
                        canvasClient.createVideoFromUrl(clientId.asLong(), video.url, null, null) :
                        canvasClient.createVideoFromFile(clientId.asLong(), video.file, video.name, null, null);

                var videoItem = new VideoItem(videoUploadResponse.getPresetId(), videoUploadResponse.getId());
                videoResults.add(Result.successful(videoItem, validation));
            } catch (CanvasClientException e) {
                logger.error("error uploading video by url to canvas ", e);

                videoResults.add(Result.broken(validation.addError(uploadVideoFailed())));
            }
        }

        boolean hasSuccessfulResults = videoResults.stream()
                .map(Result::getState)
                .anyMatch(ResultState.SUCCESSFUL::equals);
        return new MassResult<>(videoResults, validationResult,
                hasSuccessfulResults ? ResultState.SUCCESSFUL : ResultState.BROKEN);
    }

    public List<VideoItemWithStatus> getVideoStatuses(ClientId clientId,
                                                      List<VideoItem> videoItems) {
        List<VideoItemWithStatus> videos = new ArrayList<>(videoItems.size());

        for (var videoItem : videoItems) {
            String videoId = videoItem.getVideoId();
            Long presetId = videoItem.getPresetId();

            try {
                var status = getVideoStatus(clientId, presetId, videoId);
                videos.add(new VideoItemWithStatus(presetId, videoId, status));
            } catch (CanvasClientException e) {
                logger.error("error getting video by video id ", e);
            }
        }

        return videos;
    }

    private VideoUploadResponse.FileStatus getVideoStatus(ClientId clientId, Long presetId, String videoId) {
        if (!enableCreateEarlyCreativeProp.getOrDefault(false)) {
            return canvasClient.getVideo(clientId.asLong(), presetId, videoId).getStatus();
        }
        var video = canvasClient.getCreatedVideo(clientId.asLong(), presetId, videoId, null);
        var status = video.getStatus();
        if (VideoUploadResponse.FileStatus.CONVERTING.equals(status) &&
                Boolean.TRUE.equals(video.getCreateEarlyCreative())) {
            return VideoUploadResponse.FileStatus.SEMI_READY;
        }
        return status;
    }

    public MassResult<Long> createDefaultAdditionsFromVideoIds(ClientId clientId,
                                                               List<VideoItem> videoItems) {
        ValidationResult<List<VideoItem>, Defect> validationResult =
                creativeValidationService.validateCreateAdditionFromVideo(videoItems);
        List<Result<Long>> creativeResults = new ArrayList<>(videoItems.size());

        int index = 0;
        for (var videoItem : videoItems) {
            String videoId = videoItem.getVideoId();
            Long presetId = videoItem.getPresetId();
            var validation = validationResult.getOrCreateSubValidationResult(index(index++),
                    videoItem.getExternalVideoId());

            try {
                var additionResponse = canvasClient.createDefaultAddition(clientId.asLong(), presetId, videoId);

                creativeResults.add(Result.successful(additionResponse.getCreativeId(), validation));
            } catch (CanvasClientException e) {
                logger.error("error creating addition by video id ", e);

                creativeResults.add(Result.broken(validation.addError(createAdditionFromVideoFailed())));
            }
        }

        boolean hasSuccessfulResults = creativeResults.stream()
                .map(Result::getState)
                .anyMatch(ResultState.SUCCESSFUL::equals);
        return new MassResult<>(creativeResults, validationResult,
                hasSuccessfulResults ? ResultState.SUCCESSFUL : ResultState.BROKEN);
    }

    public Map<Long, List<Creative>> getCreativesByPerformanceAdGroups(ClientId clientId, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientId(clientId);
        return creativeRepository.getCreativesByPerformanceAdGroupIds(shard, clientId, null, adGroupIds);
    }

    public List<Creative> getCreativesWithBusinessType(ClientId clientId, BusinessType type,
                                                       @Nullable String idOrNameLike) {
        int shard = shardHelper.getShardByClientId(clientId);
        return creativeRepository.getPerfCreativesWithBusinessType(shard, clientId,
                creativeBusinessTypesFromBusinessType(type), idOrNameLike);
    }

    public MassResult<Long> createOrUpdate(List<Creative> creatives, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, Creative> knownCreatives =
                listToMap(creativeRepository.getCreatives(shard, clientId, mapList(creatives, Creative::getId)),
                        Creative::getId);

        ValidationResult<List<Creative>, Defect> validation =
                creativeValidationService.validateAddOrUpdate(creatives, clientId, knownCreatives);

        List<Creative> validCreatives = getValidItems(validation);
        List<Long> ids = mapList(creatives, Creative::getId);
        if (validCreatives.isEmpty()) {
            return MassResult.brokenMassAction(ids, validation);
        }
        clearAdditionalDataForNonSupportedTypes(validCreatives);

        List<Creative> creativesToAdd = new ArrayList<>();
        List<Creative> creativesToUpdate = new ArrayList<>();
        List<Long> creativeIdsToModerateAfterUpdate = new ArrayList<>();
        for (Creative validCreative : validCreatives) {
            if (!knownCreatives.containsKey(validCreative.getId())) {
                creativesToAdd.add(validCreative);
            } else {
                creativesToUpdate.add(validCreative);
                Creative existingCreative = knownCreatives.get(validCreative.getId());
                // Часть performance- и bannerstorage-креативов после редактирования
                // должна автоматически переотправляться в модерацию
                if (needModerateAfterEdit(existingCreative)) {
                    creativeIdsToModerateAfterUpdate.add(validCreative.getId());
                }
            }
        }

        dslContextProvider.ppcTransaction(shard, configuration -> {
            DSLContext txContext = configuration.dsl();

            creativeRepository.add(txContext, creativesToAdd);
            creativeRepository.update(txContext, appliedChangesCreatives(shard, clientId, creativesToUpdate));

            List<Long> performanceCreativeIds = creatives.stream()
                    .filter(c -> c.getType() == CreativeType.PERFORMANCE)
                    .map(Creative::getId)
                    .collect(toList());
            if (!performanceCreativeIds.isEmpty()) {
                List<Long> updatedPerformanceCreativeIds = creativesToUpdate.stream()
                        .filter(c -> c.getType() == CreativeType.PERFORMANCE)
                        .map(Creative::getId)
                        .collect(toList());
                // У отредактированных performance-креативов убираем причины отклонения
                if (!updatedPerformanceCreativeIds.isEmpty()) {
                    moderationReasonRepository.deleteFromModReasons(
                            txContext, updatedPerformanceCreativeIds, ModReasonsType.perf_creative);
                }
                // Отправляем отмеченные performance-креативы в модерацию
                if (!creativeIdsToModerateAfterUpdate.isEmpty()) {
                    creativeRepository.sendCreativesToModeration(txContext, creativeIdsToModerateAfterUpdate);
                }
                // Создаём таски на синхронизацию скриншотов из bannerstorage
                creativeRepository.addScreenshotsSyncTaskForPerformance(txContext, performanceCreativeIds);
            }
        });

        return MassResult.successfulMassAction(ids, validation);
    }

    private boolean needModerateAfterEdit(Creative existingCreative) {
        if (existingCreative.getType() == CreativeType.PERFORMANCE) {
            return existingCreative.getSumGeo() != null && existingCreative.getStatusModerate() != StatusModerate.NEW;
        }
        if (existingCreative.getType() == CreativeType.BANNERSTORAGE) {
            return existingCreative.getStatusModerate() != StatusModerate.NEW;
        }
        return false;
    }

    private void clearAdditionalDataForNonSupportedTypes(List<Creative> validCreatives) {
        validCreatives.forEach(creative -> {
            //DIRECT-164000 для кратных креативов в этом поле передаются оригинальные размеры картинки
            var additionalData = creative.getAdditionalData();
            var html5CreativeWithAdditionalData = creative.getType() == CreativeType.HTML5_CREATIVE
                    && additionalData != null
                    && additionalData.getOriginalHeight() != null
                    && additionalData.getOriginalWidth() != null;

            if (!ADDITIONAL_DATA_SUPPORTED_TYPES.contains(creative.getType()) && !html5CreativeWithAdditionalData) {
                creative.setAdditionalData(null);
            }
        });
    }

    private List<AppliedChanges<Creative>> appliedChangesCreatives(int shard, ClientId clientId,
                                                                   List<Creative> creativesToChange) {
        List<AppliedChanges<Creative>> appliedChanges = new ArrayList<>(creativesToChange.size());
        List<Creative> creativesFromDb =
                creativeRepository.getCreatives(shard, clientId, mapList(creativesToChange, Creative::getId));
        Map<Long, Creative> creativesFromDbMap = Maps.uniqueIndex(creativesFromDb, Creative::getId);
        for (Creative uploadData : creativesToChange) {
            ModelChanges<Creative> modelChanges = new ModelChanges<>(uploadData.getId(), Creative.class);
            Creative creative = creativesFromDbMap.get(uploadData.getId());
            if (uploadData.getName() != null) {
                modelChanges.process(uploadData.getName(), Creative.NAME);
            }
            if (uploadData.getType() == CreativeType.PERFORMANCE || uploadData.getType() == CreativeType.BANNERSTORAGE) {
                modelChanges.process(uploadData.getWidth(), Creative.WIDTH);
                modelChanges.process(uploadData.getHeight(), Creative.HEIGHT);
                modelChanges.process(uploadData.getVersion(), Creative.VERSION);
                modelChanges.process(uploadData.getIsBannerstoragePredeployed(), Creative.IS_BANNERSTORAGE_PREDEPLOYED);
            }
            if (CreativeConstants.VIDEO_TYPES.contains(uploadData.getType())) {
                modelChanges.process(uploadData.getWidth(), Creative.WIDTH);
                modelChanges.process(uploadData.getHeight(), Creative.HEIGHT);
            }
            if (uploadData.getType() == CreativeType.PERFORMANCE) {
                modelChanges.process(uploadData.getGroupName(), Creative.GROUP_NAME);
            } else if (uploadData.getType() == CreativeType.BANNERSTORAGE) {
                modelChanges.process(uploadData.getDuration(), Creative.DURATION);
            } else if (uploadData.getType() == CreativeType.HTML5_CREATIVE && uploadData.getLayoutId() != null) {
                modelChanges.process(uploadData.getLayoutId(), Creative.LAYOUT_ID);
            }

            if (uploadData.getPreviewUrl() != null) {
                modelChanges.process(uploadData.getPreviewUrl(), Creative.PREVIEW_URL);
            }
            if (uploadData.getArchiveUrl() != null) {
                modelChanges.process(uploadData.getArchiveUrl(), Creative.ARCHIVE_URL);
            }
            /* Хотя эти поля в БД содержат null представить себе ситуацию, когда их надо будет обнулить для
            существующего
            объекта в БД сложно. Если понадобится, то надо будет подумать как переписать этот код, потому что на
            текущем уровне понять было ли прислано null в значении поля или оно не было прислано вовсе уже нельзя
            Возможно стоит убрать проверки на null выше
             */
            if (uploadData.getYabsData() != null) {
                modelChanges.process(uploadData.getYabsData(), Creative.YABS_DATA);
            }
            // Для performance креативов null это законное значение для этого поля
            // В bannerstorage-креативах там хранится json со ссылкой на live preview
            if (uploadData.getModerationInfo() != null
                    || uploadData.getType() == CreativeType.PERFORMANCE
                    || uploadData.getType() == CreativeType.BANNERSTORAGE) {
                modelChanges.process(uploadData.getModerationInfo(), Creative.MODERATION_INFO);
            }
            modelChanges.process(uploadData.getLivePreviewUrl(), Creative.LIVE_PREVIEW_URL);
            if (uploadData.getAdditionalData() != null) {
                modelChanges.process(uploadData.getAdditionalData(), Creative.ADDITIONAL_DATA);
            }
            modelChanges.process(uploadData.getStatusModerate(), Creative.STATUS_MODERATE);
            appliedChanges.add(modelChanges.applyTo(creative));
        }
        return appliedChanges;
    }

    /**
     * Возвращает флаг - есть/нет html5-креативы у клиента (любые, включая не привязанные к баннерам или сайтлинкам)
     *
     * @param clientId идентификатор клиента
     */
    public boolean clientAlreadyHasHtml5Creatives(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return creativeRepository.clientAlreadyHasHtml5Creatives(shard, clientId);
    }

    /**
     * Получает из canvas видео дополнения, которые отсутствуют в ppc и сохраняет их.
     */
    public void synchronizeVideoAdditionCreatives(int shard, ClientId clientId, Set<Long> creativeIds) {
        Set<Long> existentIds = creativeRepository.getExistingClientCreativeIds(shard, clientId,
                creativeIds, singletonList(CreativeType.VIDEO_ADDITION_CREATIVE));
        List<Long> nonExistentIds = StreamEx.of(creativeIds)
                .remove(existentIds::contains)
                .toList();
        List<Creative> videoAdditions = getVideoAdditionFromCanvasHandlingException(clientId, nonExistentIds);

        creativeRepository.add(shard, videoAdditions);
    }

    public void setGeoForUnmoderatedCreatives(int shard, ClientId clientId, Collection<Long> creativeIds) {
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        Map<Long, String> geoByCreativeId = bannerCreativeRepository.getJoinedGeo(shard, geoTree, creativeIds);
        creativeRepository.updateCreativesGeo(shard, geoByCreativeId);
    }

    public void sendCreativesToModeration(int shard, Collection<Long> creativeIds) {
        creativeRepository.sendCreativesToModeration(shard, creativeIds);
    }

    public void sendRejectedCreativesToModeration(int shard, Collection<Long> creativeIds) {
        creativeRepository.sendRejectedCreativesToModeration(dslContextProvider.ppc(shard), creativeIds);
    }
}
