package ru.yandex.direct.core.entity.creative.service.add.validation;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoAspect;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoSound;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoText;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoVideo;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;
import ru.yandex.direct.core.entity.creative.repository.CreativeConstants;
import ru.yandex.direct.core.entity.creative.repository.CreativeMappings;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.creative.service.CreativeService.ADDITIONAL_DATA_SUPPORTED_TYPES;
import static ru.yandex.direct.core.entity.creative.service.add.validation.CreativeConstraints.sameCreativeType;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentStateAlreadyExists;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
public class CreativeValidationService {
    public static final String MULTI_PLAYLIST_FORMAT_TYPE = "application/vnd.apple.mpegurl";

    private final ShardSupport shardSupport;

    @Autowired
    public CreativeValidationService(ShardSupport shardSupport) {
        this.shardSupport = shardSupport;
    }

    @QueryWithForbiddenShardMapping("creative_id")
    public ValidationResult<List<Creative>, Defect> validateAddOrUpdate(List<Creative> creatives,
                                                                        ClientId clientId,
                                                                        Map<Long, Creative> existingCreatives) {
        List<Long> creativeIds = mapList(creatives, Creative::getId);
        Map<Long, Long> creativeToClient =
                shardSupport.getValuesMap(ShardKey.CREATIVE_ID, creativeIds, ShardKey.CLIENT_ID, Long.class);

        return generateValidation(creatives, clientId, creativeToClient, existingCreatives);
    }

    public ValidationResult<List<Creative>, Defect> generateValidation(
            List<Creative> creatives, ClientId clientId, Map<Long, Long> creativeToClient,
            Map<Long, Creative> existingCreatives) {
        return generateBaseCreativeValidation(creatives)
                .checkEachBy(cr -> validateOneCreative(cr, clientId, creativeToClient))
                .checkEach(unique(getVideoCreativeComparator()),
                        duplicatedObject(),
                        When.isValidAnd(When.valueIs(cr -> CreativeConstants.VIDEO_TYPES.contains(cr.getType()))))
                //Если видео-дополнения нет в базе (будет добавляться), то проверяем дубликаты clientId+stockCreativeId
                .checkEach(checkDuplicatesInCreatives(existingCreatives.values()),
                        When.isValidAnd(When.valueIs(cr -> CreativeConstants.VIDEO_TYPES.contains(cr.getType())
                                && !existingCreatives.containsKey(cr.getId()))))
                //Если креатив есть в базе (будет обновляться), то проверяем соответсвие типа креатива
                .checkEach(sameCreativeType(existingCreatives),
                        When.isValidAnd(When.valueIs(cr -> existingCreatives.containsKey(cr.getId()))))
                .getResult();
    }

    public ValidationResult<Creative, Defect> validateOneCreative(Creative creative,
                                                                  ClientId clientId, Map<Long, Long> creativeToClient) {
        ModelItemValidationBuilder<Creative> v = ModelItemValidationBuilder.of(creative);

        v.item(Creative.ID)
                .check(notNull())
                .check(CreativeConstraints.notInCreativeToClient(clientId.asLong(), creativeToClient))
                .check(greaterThan(0L));
        v.item(Creative.TYPE)
                .check(notNull());
        v.item(Creative.CLIENT_ID)
                .check(notNull());
        v.item(Creative.NAME)
                .check(notNull())
                // performance-креативы могут иметь пустое название
                .check(notEmpty(), When.isTrue(creative.getType() != CreativeType.PERFORMANCE))
                .check(CreativeConstraints.maxName());
        v.item(Creative.PREVIEW_URL)
                .check(notNull())
                .check(notEmpty())
                .check(CreativeConstraints.maxPreviewUrl());
        v.item(Creative.LIVE_PREVIEW_URL)
                .check(notNull())
                .check(notEmpty())
                .check(CreativeConstraints.maxPreviewUrl());
        v.item(Creative.LAYOUT_ID)
                .check(greaterThan(0L));

        ValidationResult<Creative, Defect> result = v.getResult();
        if (CreativeType.CANVAS.equals(creative.getType())) {
            result.merge(validateOneCanvasCreative(creative));
        } else if (CreativeConstants.VIDEO_TYPES.contains(creative.getType())) {
            result.merge(validateOneVideoCreative(creative));
        }
        return result;
    }

    private ListValidationBuilder<Creative, Defect> generateBaseCreativeValidation(
            List<Creative> creatives) {
        return ListValidationBuilder.<Creative, Defect>of(creatives)
                .check(notNull())
                .check(notEmptyCollection())
                .checkEach(unique(Creative::getId));
    }

    private ValidationResult<Creative, Defect> validateOneCanvasCreative(Creative creative) {
        ModelItemValidationBuilder<Creative> v = ModelItemValidationBuilder.of(creative);

        v.item(Creative.TYPE)
                .check(fromPredicate(CreativeType.CANVAS::equals, invalidValue()));
        v.item(Creative.WIDTH)
                .check(notNull())
                .check(notGreaterThan(Constants.UNSIGNED_SMALLINT_MAX_VALUE))
                .check(greaterThan(0L));
        v.item(Creative.HEIGHT)
                .check(notNull())
                .check(notGreaterThan(Constants.UNSIGNED_SMALLINT_MAX_VALUE))
                .check(greaterThan(0L));
        v.item(Creative.MODERATION_INFO)
                .checkBy(this::validateCanvasModerationInfo);

        return v.getResult();
    }

    private ValidationResult<Creative, Defect> validateOneVideoCreative(Creative creative) {
        ModelItemValidationBuilder<Creative> v = ModelItemValidationBuilder.of(creative);

        v.item(Creative.TYPE)
                .check(fromPredicate(CreativeConstants.VIDEO_TYPES::contains, invalidValue()));
        v.item(Creative.LAYOUT_ID)
                .check(fromPredicate(layoutId -> CreativeMappings.convertVideoType(layoutId) == creative.getType(),
                        invalidValue()));
        v.item(Creative.STOCK_CREATIVE_ID)
                .check(notNull())
                .check(greaterThan(0L));
        v.item(Creative.MODERATION_INFO)
                .checkBy(this::validateVideoAdditionModerationInfo);

        if (ADDITIONAL_DATA_SUPPORTED_TYPES.contains(creative.getType())) {
            v.item(Creative.ADDITIONAL_DATA)
                    .check(notNull())
                    .checkBy(this::validateAdditionalData, When.isValid());
        }

        return v.getResult();
    }

    private ValidationResult<ModerationInfo, Defect> validateCanvasModerationInfo(ModerationInfo moderationInfo) {
        ItemValidationBuilder<ModerationInfo, Defect> v = ItemValidationBuilder.of(moderationInfo);
        if (moderationInfo == null) {
            return v.getResult();
        }

        if (moderationInfo.getTexts() != null) {
            v.list(moderationInfo.getTexts(), "texts")
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(this::validateModerationInfoText);
        }
        return v.getResult();
    }

    private ValidationResult<ModerationInfo, Defect> validateVideoAdditionModerationInfo(
            ModerationInfo moderationInfo) {
        ItemValidationBuilder<ModerationInfo, Defect> v = ItemValidationBuilder.of(moderationInfo);
        if (moderationInfo == null) {
            return v.getResult();
        }
        if (moderationInfo.getSounds() != null) {
            v.list(moderationInfo.getSounds(), "sounds")
                    .checkEach(notNull())
                    .checkEachBy(this::validateModerationInfoSound);
        }
        if (moderationInfo.getVideos() != null) {
            v.list(moderationInfo.getVideos(), "videos")
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(this::validateModerationInfoVideo);
        }
        if (moderationInfo.getAspects() != null) {
            v.list(moderationInfo.getAspects(), "aspects")
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(this::validateModerationInfoAspect);
        }
        return v.getResult();
    }

    private ValidationResult<ModerationInfoText, Defect> validateModerationInfoText(ModerationInfoText text) {
        ItemValidationBuilder<ModerationInfoText, Defect> v = ItemValidationBuilder.of(text);
        if (text == null) {
            return v.getResult();
        }
        v.item(text.getText(), "text")
                .check(notNull())
                .check(notEmpty());
        return v.getResult();
    }

    private ValidationResult<ModerationInfoVideo, Defect> validateModerationInfoVideo(ModerationInfoVideo video) {
        ItemValidationBuilder<ModerationInfoVideo, Defect> v = ItemValidationBuilder.of(video);
        if (video == null) {
            return v.getResult();
        }
        v.item(video.getUrl(), "url")
                .check(notNull())
                .check(notEmpty());
        return v.getResult();
    }

    private ValidationResult<ModerationInfoAspect, Defect> validateModerationInfoAspect(
            ModerationInfoAspect aspect) {
        ItemValidationBuilder<ModerationInfoAspect, Defect> v = ItemValidationBuilder.of(aspect);
        if (aspect == null) {
            return v.getResult();
        }
        v.item(aspect.getWidth(), "width")
                .check(notNull())
                .check(notGreaterThan(Constants.MAX_ASPECT_VALUE))
                .check(greaterThan(0L));
        v.item(aspect.getHeight(), "height")
                .check(notNull())
                .check(notGreaterThan(Constants.MAX_ASPECT_VALUE))
                .check(greaterThan(0L));
        return v.getResult();
    }

    private ValidationResult<ModerationInfoSound, Defect> validateModerationInfoSound(ModerationInfoSound sound) {
        ItemValidationBuilder<ModerationInfoSound, Defect> v = ItemValidationBuilder.of(sound);
        if (sound == null) {
            return v.getResult();
        }
        v.item(sound.getUrl(), "url")
                .check(notNull())
                .check(notEmpty());
        return v.getResult();
    }

    private ValidationResult<AdditionalData, Defect> validateAdditionalData(
            AdditionalData additionalData) {
        ModelItemValidationBuilder<AdditionalData> vb = ModelItemValidationBuilder.of(additionalData);

        vb.item(AdditionalData.DURATION)
                .check(notNull());
        vb.list(AdditionalData.FORMATS)
                .check(notNull())
                .check(notEmptyCollection())
                .checkEachBy(this::validateVideoFormat);
        return vb.getResult();
    }

    private ValidationResult<VideoFormat, Defect> validateVideoFormat(
            VideoFormat videoFormat) {
        ModelItemValidationBuilder<VideoFormat> vb = ModelItemValidationBuilder.of(videoFormat);
        vb.item(VideoFormat.TYPE)
                .check(notNull())
                .check(notBlank());
        vb.item(VideoFormat.URL)
                .check(notNull())
                .check(notBlank());

        if (!MULTI_PLAYLIST_FORMAT_TYPE.equals(videoFormat.getType())) {
            vb.item(VideoFormat.WIDTH)
                    .check(notNull());
            vb.item(VideoFormat.HEIGHT)
                    .check(notNull());
        }
        return vb.getResult();
    }

    //todo: переделать
    /**
     * Проверяем, что среди добавляемых элементов нет дубликатов
     */
    private ListConstraint<Creative, Defect> checkDuplicatesInCreatives(
            Collection<Creative> existingCreatives) {
        return creatives -> {
            if (creatives.isEmpty()) {
                return Collections.emptyMap();
            }
            TreeSet<Creative> existing = new TreeSet<>(getVideoCreativeComparator());
            existing.addAll(existingCreatives);

            Map<Integer, Defect> integerDefectDefinitionMap = EntryStream.of(creatives)
                    .filterValues(existing::contains)
                    .mapValues(c -> (Defect) inconsistentStateAlreadyExists())
                    .toMap();
            return integerDefectDefinitionMap;
        };
    }

    private Comparator<Creative> getVideoCreativeComparator() {
        return Comparator.comparing(Creative::getClientId)
                .thenComparing(c -> nvl(c.getStockCreativeId(), c.getId()));
        //Т.к. в базе есть не заполненные stockCreativeId заменим его на creativeId.
        //Когда первичным ключем станет clientId+stockCreativeId нужно убрать замену nvl
    }

    public ValidationResult<List<CreativeService.VideoItemForUpload>, Defect> validateCreateVideosForUpload(
            List<CreativeService.VideoItemForUpload> videos) {
        return ListValidationBuilder.<CreativeService.VideoItemForUpload, Defect>of(videos)
                .check(notNull())
                .check(notEmptyCollection())
                .check(eachNotNull())
                .checkEachBy(validateCreateVideoForUpload(), When.notNull())
                .getResult();
    }

    public Validator<CreativeService.VideoItemForUpload, Defect> validateCreateVideoForUpload() {
        return video -> {
            ItemValidationBuilder<CreativeService.VideoItemForUpload, Defect> v = ItemValidationBuilder.of(video);
            if (video.getFile() == null) {
                v.item(video.getUrl(), "url")
                        .check(notNull());
                v.item(video.getName(), "name")
                        .check(isNull());
            } else {
                v.item(video.getUrl(), "url")
                        .check(isNull());
                v.item(video.getName(), "name")
                        .check(notNull());
            }
            return v.getResult();
        };
    }


    public ValidationResult<List<CreativeService.VideoItem>, Defect> validateCreateAdditionFromVideo(
            List<CreativeService.VideoItem> videoItems) {
        return ListValidationBuilder.<CreativeService.VideoItem, Defect>of(videoItems)
                .check(notNull())
                .check(notEmptyCollection())
                .check(eachNotNull())
                .getResult();
    }
}
