package ru.yandex.direct.core.entity.crypta.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.crypta.model.CryptaSegmentValue;
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.repository.InterestTypeMappings;
import ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.audio_genres;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.behaviors;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.brandsafety;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.content_category;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.content_genre;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.family;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.interests;
import static ru.yandex.direct.dbschema.ppcdict.enums.CryptaGoalsCryptaGoalType.social_demo;
import static ru.yandex.direct.dbschema.ppcdict.tables.CryptaGoals.CRYPTA_GOALS;
import static ru.yandex.direct.dbutil.SqlUtils.isEmptyString;
import static ru.yandex.direct.dbutil.SqlUtils.setField;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Получение сегментов крипты
 */

@Repository
public class CryptaSegmentRepository {

    public final JooqMapperWithSupplier<Goal> mapper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public CryptaSegmentRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }

    /**
     * Получить сегменты крипты, доступные "обычным" пользователям
     *
     * @return {@link Map} список сегментов крипты по идентификатору
     */
    public Map<Long, Goal> getAll() {
        return getAll(CryptaGoalScope.COMMON);
    }

    /**
     * Получить сегменты крипты
     *
     * @param scope видимость сегментов, например CryptaGoalScope.INTERNAL_AD -- сегменты, видимые
     *              для внутренней рекламы, при этом они могут быть недоступны для "обычных" пользователей.
     * @return {@link Map} список сегментов крипты по идентификатору
     */
    public Map<Long, Goal> getAll(CryptaGoalScope scope) {
        return getAll(List.of(scope));
    }

    public Map<Long, Goal> getAll(Collection<CryptaGoalScope> scopes) {
        if (CollectionUtils.isEmpty(scopes)) return Map.of();
        var allKnownGoalTypes = getAllKnownGoalTypes();

        var scopeField = setField(CRYPTA_GOALS.SCOPE);
        var scopeCondition = scopes
                .stream()
                .map( value -> scopeField.contains(value.getTypedValue()))
                .reduce(Condition::or)
                .orElseThrow(() -> new IllegalStateException("Unable to create scope condition in select"));

        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(CRYPTA_GOALS)
                .where(scopeCondition.and(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(allKnownGoalTypes)))
                .fetchMap(CRYPTA_GOALS.GOAL_ID, mapper::fromDb);
    }

    public int add(Collection<Goal> addRecords) {
        return new InsertHelper<>(dslContextProvider.ppcdict(), CRYPTA_GOALS)
                .addAll(mapper, addRecords)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    public Map<Long, Goal> getWithoutBrandSafety(CryptaGoalScope scope) {
        var goalTypes = getAllKnownGoalTypes().stream().filter(t -> !brandsafety.equals(t)).collect(toList());
        return getByCondition(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(goalTypes), scope);
    }

    /**
     * Получить типы социал-демо
     *
     * @return {@link Map} список сегментов крипты по идентификатору
     */
    public Map<Long, Goal> getSocialDemoTypes() {
        return getByCondition(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(social_demo, family, behaviors)
                .and(CRYPTA_GOALS.PARENT_GOAL_ID.eq(0L)), CryptaGoalScope.COMMON);
    }

    /**
     * Получить социал-демо сегменты
     *
     * @return {@link Map} список сегментов крипты по идентификатору
     */
    public Map<Long, Goal> getSocialDemo() {
        return getByCondition(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(social_demo, family, behaviors)
                .and(CRYPTA_GOALS.PARENT_GOAL_ID.ne(0L)), CryptaGoalScope.COMMON);
    }

    /**
     * Получить сегменты интересов
     *
     * @return {@link Map} список сегментов крипты по идентификатору
     */
    public Map<Long, Goal> getInterests() {
        return getByType(interests);
    }

    public Map<Long, Goal> getBrandSafety() {
        return getByType(brandsafety);
    }

    public Map<Long, Goal> getContentCategories() {
        return getByTypes(content_category);
    }

    public Map<Long, Goal> getContentSegments() {
        return getByTypes(content_category, content_genre);
    }

    public Map<Long, Goal> getBrandSafetyAndContentSegments() {
        return getByTypes(brandsafety, content_category, content_genre);
    }

    public Goal getById(Long id) {
        return getByCondition(CRYPTA_GOALS.GOAL_ID.in(singleton(id))).get(id);
    }

    /**
     * Получение списка целей по id без учета scope
     */
    public Map<Long, Goal> getByIdsForCa(Collection<Long> ids) {
        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(CRYPTA_GOALS)
                .where(CRYPTA_GOALS.GOAL_ID.in(ids))
                .fetchMap(CRYPTA_GOALS.GOAL_ID, mapper::fromDb);
    }

    public Map<Long, Goal> getByIds(Collection<Long> ids) {
        return getByCondition(CRYPTA_GOALS.GOAL_ID.in(ids));
    }

    /**
     * Получить аудио жанры
     *
     * @return {@link Map} список жанров по идентификатору
     */
    public Map<Long, Goal> getAudioGenres() {
        return getByType(audio_genres);
    }

    public Map<Long, Goal> getBehaviorsByIds(Collection<Long> ids) {
        return getByTypeAndIds(behaviors, ids);
    }

    public Map<Long, Goal> getInterestsByIds(Collection<Long> ids) {
        return getByTypeAndIds(interests, ids);
    }

    /**
     * Найти цель в таблице crypta_goals по segment_id и keyword_id из BigB:
     * - bb_keyword - keyword_id долгосрочной цели
     * - bb_keyword_value - segment_id долгосрочной цели
     * - bb_keyword_short - keyword_id краткосрочной цели
     * - bb_keyword_value_short - segment_id краткосрочной цели
     */
    public Optional<Goal> findByKeywordIdSegmentId(CryptaSegmentValue firstSegment,
                                                   @Nullable CryptaSegmentValue secondSegment) {
        Condition condition;
        if (secondSegment != null) {
            condition = CRYPTA_GOALS.BB_KEYWORD.in(firstSegment.getKeywordId(), secondSegment.getKeywordId())
                    .and(CRYPTA_GOALS.BB_KEYWORD_VALUE.in(firstSegment.getSegmentId(), secondSegment.getSegmentId()))
                    .and(CRYPTA_GOALS.BB_KEYWORD_SHORT.in(firstSegment.getKeywordId(), secondSegment.getKeywordId()))
                    .and(CRYPTA_GOALS.BB_KEYWORD_VALUE_SHORT.in(firstSegment.getSegmentId(), secondSegment.getSegmentId()));
        } else {
            condition = CRYPTA_GOALS.BB_KEYWORD.eq(firstSegment.getKeywordId())
                    .and(CRYPTA_GOALS.BB_KEYWORD_VALUE.eq(firstSegment.getSegmentId()))
                    .and(isEmptyString(CRYPTA_GOALS.BB_KEYWORD_SHORT))
                    .and(isEmptyString(CRYPTA_GOALS.BB_KEYWORD_VALUE_SHORT));
        }

        var goalTypes =  filterList(getAllKnownGoalTypes(), type -> !brandsafety.equals(type));
        condition = condition.and(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(goalTypes));

        var fieldsToSelect = List.of(
                CRYPTA_GOALS.GOAL_ID,
                CRYPTA_GOALS.BB_KEYWORD, CRYPTA_GOALS.BB_KEYWORD_VALUE,
                CRYPTA_GOALS.BB_KEYWORD_SHORT, CRYPTA_GOALS.BB_KEYWORD_VALUE_SHORT
        );

        return dslContextProvider.ppcdict()
                .select(fieldsToSelect)
                .from(CRYPTA_GOALS)
                .where(condition)
                .orderBy(CRYPTA_GOALS.SCOPE)
                .limit(1)
                .fetchOptional(mapper::fromDb);
    }

    /**
     * Собирает все типы сегментов из базы, для которых уже есть маппинг в коде
     */
    private List<CryptaGoalsCryptaGoalType> getAllKnownGoalTypes() {
        return StreamEx.of(GoalType.values())
                .filter(GoalType::isCrypta)
                .map(t -> toDbGoalType(t.name().toLowerCase()))
                .nonNull()
                .collect(toList());
    }

    private CryptaGoalsCryptaGoalType toDbGoalType(String typeName) {
        return StreamEx.of(CryptaGoalsCryptaGoalType.values())
                .findFirst(type -> type.name().equals(typeName))
                .orElse(null);
    }

    private Map<Long, Goal> getByType(CryptaGoalsCryptaGoalType type) {
        return getByCondition(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(type));
    }

    private Map<Long, Goal> getByTypeAndIds(CryptaGoalsCryptaGoalType type, Collection<Long> ids) {
        return getByCondition(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(type)
                .and(CRYPTA_GOALS.GOAL_ID.in(ids)));
    }

    private Map<Long, Goal> getByTypes(CryptaGoalsCryptaGoalType... types) {
        return getByCondition(CRYPTA_GOALS.CRYPTA_GOAL_TYPE.in(types));
    }

    private Map<Long, Goal> getByCondition(Condition whereCondition) {
        return getByCondition(whereCondition, CryptaGoalScope.COMMON);
    }

    private Map<Long, Goal> getByCondition(Condition whereCondition, CryptaGoalScope scope) {
        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(CRYPTA_GOALS)
                .where(whereCondition)
                .and(setField(CRYPTA_GOALS.SCOPE).contains(scope.getTypedValue()))
                .fetchMap(CRYPTA_GOALS.GOAL_ID, mapper::fromDb);
    }

    private JooqMapperWithSupplier<Goal> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(Goal::new)
                .map(property(Goal.ID, CRYPTA_GOALS.GOAL_ID))
                .map(property(Goal.PARENT_ID, CRYPTA_GOALS.PARENT_GOAL_ID))
                .map(convertibleProperty(Goal.TYPE, CRYPTA_GOALS.CRYPTA_GOAL_TYPE,
                        t -> GoalType.valueOf(t.name().toUpperCase()),
                        t -> CryptaGoalsCryptaGoalType.valueOf(t.name().toLowerCase())))
                .map(convertibleProperty(Goal.INTEREST_TYPE, CRYPTA_GOALS.INTEREST_TYPE,
                        InterestTypeMappings::interestTypeFromDb,
                        InterestTypeMappings::interestTypeToDb))
                .map(convertibleEnumSet(Goal.CRYPTA_SCOPE, CRYPTA_GOALS.SCOPE,
                        CryptaGoalScope::fromTypedValue,
                        CryptaGoalScope::getTypedValue))
                .map(property(Goal.NAME, CRYPTA_GOALS.NAME))
                .map(property(Goal.TANKER_NAME_KEY, CRYPTA_GOALS.TANKER_NAME_KEY))
                .map(property(Goal.TANKER_DESCRIPTION_KEY, CRYPTA_GOALS.TANKER_DESCRIPTION_KEY))
                .map(property(Goal.KEYWORD, CRYPTA_GOALS.BB_KEYWORD))
                .map(property(Goal.KEYWORD_VALUE, CRYPTA_GOALS.BB_KEYWORD_VALUE))
                .map(property(Goal.KEYWORD_SHORT, CRYPTA_GOALS.BB_KEYWORD_SHORT))
                .map(property(Goal.KEYWORD_VALUE_SHORT, CRYPTA_GOALS.BB_KEYWORD_VALUE_SHORT))
                .map(property(Goal.TANKER_AUDIENCE_TYPE_KEY, CRYPTA_GOALS.TANKER_AUDIENCE_TYPE_KEY))
                .build();
    }
}
