package ru.yandex.direct.core.entity.crypta.service

import com.google.protobuf.util.JsonFormat
import ru.yandex.crypta.siberia.bin.custom_audience.common.proto.TCaRule
import ru.yandex.crypta.siberia.bin.custom_audience.suggester.grpc.TExport
import ru.yandex.crypta.siberia.bin.custom_audience.suggester.grpc.TItem
import ru.yandex.direct.core.entity.retargeting.Constants.AUDIENCE_TIME_VALUE
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestItem
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestSegment
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestType
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.utils.FunctionalUtils

object CryptaSuggestConverter {

    private val jsonPrinter = JsonFormat.printer()
        .omittingInsignificantWhitespace()
        .preservingProtoFieldNames()

    fun convertToCryptaSuggest(cryptaRule: TItem): CryptaGoalsSuggestItem = CryptaGoalsSuggestItem()
        .withText(cryptaRule.text)
        .withType(this.convertTypeFromString(cryptaRule.type.uppercase()))
        .withDescription(cryptaRule.description)
        .withSegments(FunctionalUtils.mapList(cryptaRule.exportsList, this::convertToSegment))

    fun convertToGoal(item: CryptaGoalsSuggestItem) = Goal()
        .withTime(AUDIENCE_TIME_VALUE)
        .withCaText(item.text)
        .withCryptaParentRule(jsonPrinter.print(convertToCaRule(item))) as Goal

    fun convertHostToGoal(cryptaRule: TItem) = Goal()
        .withId(cryptaRule.hostId)
        .withName(cryptaRule.text) as Goal

    fun convertToSegment(export: TExport): CryptaGoalsSuggestSegment =
        createSuggestSegment(export.keywordId, export.segmentId)

    /**
     * Конвертирует предсавление цели Директа в ответ от саджестера Крипты
     * GoalType.HOST не поддерживается
     */
    fun convertGoalToSegments(goal: Goal): List<CryptaGoalsSuggestSegment> {
        requireNotNull(goal.keyword) { "Keyword cannot be null" }
        requireNotNull(goal.keywordValue) { "KeywordValue cannot be null" }
        val longTermSegment = createSuggestSegment(goal.keyword.toLong(), goal.keywordValue.toLong())
        val shortTermSegment = goal.keywordShort?.let {
            requireNotNull(goal.keywordValueShort) { "KeywordValueShort cannot be null if keywordShort is present" }
            createSuggestSegment(it.toLong(), goal.keywordValueShort.toLong())
        }

        return when(shortTermSegment) {
            null -> listOf(longTermSegment)
            else -> listOf(longTermSegment, shortTermSegment)
        }
    }

    fun convertTypeFromString(type: String) = CryptaGoalsSuggestType.values().find { it.name == type }

    private fun convertToCaRule(item: CryptaGoalsSuggestItem) = TCaRule.newBuilder().addHosts(item.text).build()

    private fun createSuggestSegment(keywordId: Long, segmentId: Long) = CryptaGoalsSuggestSegment()
        .withKeywordId(keywordId)
        .withSegmentId(segmentId)
}
