package ru.yandex.direct.core.entity.crypta.service

import one.util.streamex.EntryStream
import org.slf4j.LoggerFactory
import org.springframework.context.i18n.LocaleContextHolder
import org.springframework.stereotype.Service
import org.springframework.util.CollectionUtils
import ru.yandex.crypta.siberia.bin.custom_audience.suggester.grpc.TExport
import ru.yandex.crypta.siberia.bin.custom_audience.suggester.grpc.TItem
import ru.yandex.direct.core.entity.crypta.exception.InvalidCryptaSegmentException
import ru.yandex.direct.core.entity.crypta.model.CryptaSegmentValue
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository
import ru.yandex.direct.core.entity.crypta.service.CryptaSuggestConverter.convertHostToGoal
import ru.yandex.direct.core.entity.lal.CaLalSegmentRepository
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestItem
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestSegment
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestType
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.Goal.computeType
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.crypta.client.CryptaClient
import ru.yandex.direct.crypta.client.model.CryptaSuggestCampaignType

/**
 * Сервис для взаимодействия с саджестером крипты.
 */
@Service
class CryptaSuggestService(
    private val cryptaClient: CryptaClient,
    private val cryptaSegmentRepository: CryptaSegmentRepository,
    private val caLalSegmentRepository: CaLalSegmentRepository
) {

    companion object {
        private val logger = LoggerFactory.getLogger(CryptaSuggestService::class.java)

        private const val INVALID_SEGMENT_COUNT_MESSAGE = "Should be at least one keywordId and segmentId"
        private const val CRYPTA_GOAL_NOT_FOUND_MESSAGE = "Unable to find crypta goal"
    }

    /**
     * Получить рекомендации аудиторий от Крипты.
     *
     * @return список целей в Директе + хосты
     */
    fun getRetargetingGoalsSuggest(text: String): List<Goal> {
        //пока поддерживаем только перфоманс кампании
        val cryptaSuggest = cryptaClient.getSuggestRules(
            text.trim(),
            CryptaSuggestCampaignType.PERFORMANCE,
            LocaleContextHolder.getLocale())
        val goals = cryptaSegmentRepository.getAll(CryptaGoalScope.PERFORMANCE).flatMap { entry ->
            CryptaSuggestConverter.convertGoalToSegments(entry.value).map { it to entry.value }
        }.toMap()

        return cryptaSuggest.mapNotNull { convertCryptaSuggestToGoal(it, goals) }
    }

    private fun convertCryptaSuggestToGoal(cryptaItem: TItem, goals: Map<CryptaGoalsSuggestSegment, Goal>): Goal? =
        when (CryptaSuggestConverter.convertTypeFromString(cryptaItem.type.uppercase())) {
            CryptaGoalsSuggestType.HOST -> convertHostToGoal(cryptaItem)
            else -> findGoal(goals, cryptaItem.exportsList)
        }

    private fun findGoal(goals: Map<CryptaGoalsSuggestSegment, Goal>, exportsList: List<TExport>?): Goal? =
        when {
            exportsList.isNullOrEmpty() -> null
            else -> exportsList.map(CryptaSuggestConverter::convertToSegment).firstOrNull()?.let { goals[it] }
        }

    /**
     * Получить рекомендации по целям от Крипты
     */
    @Deprecated("start using getRetargetingGoalsSuggest")
    fun getRetargetingSuggests(text: String): List<CryptaGoalsSuggestItem> = cryptaClient.getSuggestRules(text.trim())
        .map(CryptaSuggestConverter::convertToCryptaSuggest)
        .filter { it.type != null }

    /**
     * Преобразовать полученные рекомендации от Крипты в цели для таргетинга.
     * В зависимости от типа рекомендации:
     * - SEGMENT матчится с данными из crypta_goals
     * - HOST проверяется с ca_lal_segments (если сегмента нет, то он будет создан)
     *
     * @throws InvalidCryptaSegmentException - если интерес имеет неверный формат или не существует
     */
    fun getRetargetingGoals(suggests: List<CryptaGoalsSuggestItem>?): List<Goal> {
        if (CollectionUtils.isEmpty(suggests)) return listOf()

        val suggestsByType = suggests!!.groupBy { it.type }
        val interestsFromDB = getCryptaGoalsFromDb(
            suggestsByType.getOrDefault(CryptaGoalsSuggestType.SEGMENT, listOf())
        ).associateBy {
            CryptaGoalsSuggestSegment()
                .withKeywordId(it.keyword.toLongOrNull())
                .withSegmentId(it.keywordValue.toLongOrNull())
        }
        val lalSegmentsByHosts = getOrCreateCaLalSegments(suggestsByType[CryptaGoalsSuggestType.HOST])
        return suggests.mapNotNull { calculateRetargetingGoal(it, interestsFromDB, lalSegmentsByHosts) }
    }

    /**
     * Преобразовать список наших целей в формат рекомендаций от крипты
     */
    fun getCryptaSuggestFromGoals(goals: List<Goal>): List<CryptaGoalsSuggestItem> {
        val filteredGoals = goals.filter { isSupportedGoal(it) }
        //для обратной совместимости, нам надо наполнить цели данными
        val cryptaGoalIdsWithoutKeywords = filteredGoals.filter { it.type.isCrypta && it.keyword == null }
            .map { it.id }
        val cryptaGoalsById = cryptaSegmentRepository.getByIdsForCa(cryptaGoalIdsWithoutKeywords)

        return filteredGoals.mapNotNull { castToCryptaItem(cryptaGoalsById[it.id] ?: it) }
    }

    private fun isSupportedGoal(goal: Goal) = goal.type?.isCrypta ?: false
            || (goal.type == GoalType.LAL_SEGMENT && goal.caText != null)

    private fun getCryptaGoalsFromDb(suggests: List<CryptaGoalsSuggestItem>): List<Goal> =
        suggests.map(this::getCryptaGoalFromDb)

    private fun getCryptaGoalFromDb(suggest: CryptaGoalsSuggestItem): Goal {
        val segments = extractSegmentValues(suggest.segments)

        val optionalGoal = if (segments.size > 1) {
            cryptaSegmentRepository.findByKeywordIdSegmentId(segments[0], segments[1])
        } else {
            cryptaSegmentRepository.findByKeywordIdSegmentId(segments[0], null)
        }

        return optionalGoal.map {
            it.caText = suggest.text
            it
        }.orElseThrow {
            logger.error("Unable to get goals for retargeting from crypta suggester: $suggest")
            InvalidCryptaSegmentException("$CRYPTA_GOAL_NOT_FOUND_MESSAGE: $suggest")
        }
    }

    private fun getOrCreateCaLalSegments(suggests: List<CryptaGoalsSuggestItem>?): Map<String, Goal> {
        if (CollectionUtils.isEmpty(suggests)) return mapOf()

        val existingHosts = caLalSegmentRepository.findAllByHosts(suggests!!.map { it.text })
        val restHosts = suggests.filter { !existingHosts.containsKey(it.text) }
            .map { CryptaSuggestConverter.convertToGoal(it) }
        caLalSegmentRepository.createHostSegments(restHosts)

        return EntryStream.of(restHosts.associateBy { it.caText })
            .append(existingHosts)
            .toMap()
    }

    private fun extractSegmentValues(segments: List<CryptaGoalsSuggestSegment>): List<CryptaSegmentValue> {
        if (CollectionUtils.isEmpty(segments)) throw InvalidCryptaSegmentException(INVALID_SEGMENT_COUNT_MESSAGE)

        return segments.map {
            CryptaSegmentValue(
                it.segmentId.toString(), it.keywordId.toString()
            )
        }
    }

    /**
     * Вычислить цель по типу рекомендации от Крипты:
     * - если это хост, то берем как есть из наших lal сегментов
     * - если это сегмент, то пытаемся найти из доступных целей Директа
     *   значение по одной из связок keywordId+segmentId от Крипты:
     *   таких связок может быть до двух - краткосрочная цель и долгосрочная.
     *   Т.к. Крипта не передает информацию о том, какая из них какой является,
     *   то мы ищем поочередную каждую из доступных
     *
     * @param goalsSuggestItem - рекомендация от крипты
     * @param interestsFromDB - мапа доступных целей в Директе на рекомендацию интересов из Крипты
     * @param lalSegmentsByHosts - мапа доступных целей в Директе на рекомендацию хостов из Крипты
     */
    private fun calculateRetargetingGoal(
        goalsSuggestItem: CryptaGoalsSuggestItem,
        interestsFromDB: Map<CryptaGoalsSuggestSegment, Goal>,
        lalSegmentsByHosts: Map<String, Goal>
    ): Goal? = when (goalsSuggestItem.type) {
        CryptaGoalsSuggestType.HOST -> lalSegmentsByHosts[goalsSuggestItem.text]
        CryptaGoalsSuggestType.SEGMENT -> when (goalsSuggestItem.segments.size) {
            1 -> interestsFromDB[goalsSuggestItem.segments[0]]
            2 -> interestsFromDB[goalsSuggestItem.segments[0]] ?: interestsFromDB[goalsSuggestItem.segments[1]]
            else -> null
        }
        else -> null
    }

    /**
     * cast to CryptaGoalsSuggestItem.
     * supports "old" variant:
     * - if goal.type is LAL and doesn't have caText then it is old format and we should skip it
     * - if goal.type is not crypta type then return null
     */
    private fun castToCryptaItem(goal: Goal): CryptaGoalsSuggestItem? = when {
        goal.type == GoalType.LAL_SEGMENT -> CryptaGoalsSuggestItem()
            .withText(goal.caText)
            .withType(CryptaGoalsSuggestType.HOST)
            .withSegments(listOf())
        goal.type.isCrypta -> CryptaGoalsSuggestItem()
            .withText(goal.caText ?: goal.name)
            .withType(CryptaGoalsSuggestType.SEGMENT)
            .withSegments(extractSegments(goal))
        else -> null
    }

    private fun extractSegments(goal: Goal): List<CryptaGoalsSuggestSegment> {
        val segments = mutableListOf(
            CryptaGoalsSuggestSegment()
                .withKeywordId(goal.keyword.toLongOrNull())
                .withSegmentId(goal.keywordValue.toLongOrNull())
        )
        if (!goal.keywordShort.isNullOrEmpty() && !goal.keywordValueShort.isNullOrEmpty()) {
            segments.add(
                CryptaGoalsSuggestSegment()
                    .withKeywordId(goal.keywordShort.toLongOrNull())
                    .withSegmentId(goal.keywordValueShort.toLongOrNull())
            )
        }
        return segments
    }

    /**
     * Получаем мапу из типа цели в множество целей с таким типом, и для всех айдишников целей
     * с типом HOST обращается за целями, имена в которых соответствует актуальным названиям хостов в крипте,
     * для остальных айдишников получает им соответствующие цели из репозитория с сегментами крипты.
     */
    fun getGoalsByIds(goalIds: List<Long>): Map<Long, Goal> {
        val hostGoalsIds = goalIds.filter { computeType(it) == GoalType.HOST }
        val hostGoals = getGoalsByHostIds(hostGoalsIds)
        val otherGoalsIds = goalIds
            .filter{ computeType(it) != GoalType.HOST }
            .toSet()
        val otherGoals = cryptaSegmentRepository.getByIdsForCa(otherGoalsIds)
        return otherGoals + hostGoals
    }

    /**
     * Получаем мапу из айдшников хостов в цели, которые отражают инфу, полученные от крипты
     */
    fun getGoalsByHostIds(ids: List<Long>) : Map<Long, Goal> =
        cryptaClient.getHostsByIds(ids).associateBy(TItem::getHostId) { convertHostToGoal(it) }
}
