package ru.yandex.direct.core.entity.crypta.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.validation.constraints.NotNull;

import ru.yandex.direct.feature.FeatureName;

import static java.util.stream.Collectors.toList;

@ParametersAreNonnullByDefault
public class CryptaSegmentBrandSafetyUtils {
    private static final long BRAND_SAFETY_ADULT_ID = 4294967297L;
    private static final long BRAND_SAFETY_TERRORISM_ID = 4294967298L;
    private static final long BRAND_SAFETY_NEGATIVE_ID = 4294967299L;
    private static final long BRAND_SAFETY_WEAPONS_ID = 4294967300L;
    private static final long BRAND_SAFETY_OBSCENE_ID = 4294967301L;
    private static final long BRAND_SAFETY_GAMBLING_ID = 4294967304L;
    private static final long BRAND_SAFETY_TOBACCO_ID = 4294967312L;

    private static final long BRAND_SAFETY_POLITICS_ID = 4294967302L;
    private static final long BRAND_SAFETY_CHILD_ID = 4294967303L;
    private static final long BRAND_SAFETY_NEWS_ID = 4294967305L;
    private static final long BRAND_SAFETY_RELIGION_ID = 4294967307L;
    private static final long BRAND_SAFETY_GAMES_ID = 4294967308L;
    private static final long BRAND_SAFETY_DATING_ID = 4294967309L;
    private static final long BRAND_SAFETY_MEDICINE_ID = 4294967310L;
    private static final long BRAND_SAFETY_ALCO_ID = 4294967311L;
    private static final long BRAND_SAFETY_PARA_ID = 4294967313L;
    private static final long BRAND_SAFETY_UKRAINE_ID = 4294967314L;

    private static final Set<Long> BRAND_SAFETY_BASE_CATEGORIES = Set.of(
            BRAND_SAFETY_ADULT_ID,
            BRAND_SAFETY_TERRORISM_ID,
            BRAND_SAFETY_NEGATIVE_ID,
            BRAND_SAFETY_WEAPONS_ID,
            BRAND_SAFETY_OBSCENE_ID,
            BRAND_SAFETY_GAMBLING_ID,
            BRAND_SAFETY_TOBACCO_ID);

    private static final Map<Long, FeatureName> BRAND_SAFETY_ADDITIONAL_CATEGORIES_TO_FEATURE = Map.of(
            BRAND_SAFETY_POLITICS_ID, FeatureName.BRANDSAFETY_POLITICS,
            BRAND_SAFETY_CHILD_ID, FeatureName.BRANDSAFETY_CHILD,
            BRAND_SAFETY_NEWS_ID, FeatureName.BRANDSAFETY_NEWS,
            BRAND_SAFETY_RELIGION_ID, FeatureName.BRANDSAFETY_RELIGION,
            BRAND_SAFETY_GAMES_ID, FeatureName.BRANDSAFETY_GAMES,
            BRAND_SAFETY_DATING_ID, FeatureName.BRANDSAFETY_DATING,
            BRAND_SAFETY_MEDICINE_ID, FeatureName.BRANDSAFETY_MEDICINE,
            BRAND_SAFETY_ALCO_ID, FeatureName.BRANDSAFETY_ALCO,
            BRAND_SAFETY_PARA_ID, FeatureName.BRANDSAFETY_PARA,
            BRAND_SAFETY_UKRAINE_ID, FeatureName.BRANDSAFETY_UKRAINE);

    private static final Set<Long> BRAND_SAFETY_ADDITIONAL_CATEGORIES =
            BRAND_SAFETY_ADDITIONAL_CATEGORIES_TO_FEATURE.keySet();

    private CryptaSegmentBrandSafetyUtils() {
    }

    public static boolean extractBrandSafetyIsEnabled(@Nullable List<Long> categories) {
        return categories != null && categories.stream()
                .anyMatch(BRAND_SAFETY_BASE_CATEGORIES::contains);
    }

    public static Set<Long> extractBrandSafetyAdditionalCategories(@Nullable List<Long> categories) {
        if (categories == null) {
            return Collections.emptySet();
        }
        return categories.stream()
                .filter(BRAND_SAFETY_ADDITIONAL_CATEGORIES::contains)
                .collect(Collectors.toSet());
    }

    public static List<Long> makeBrandSafetyCategories(boolean isEnabled, @NotNull Collection<Long> additionalCategories) {
        if (!isEnabled) {
            return Collections.emptyList();
        }
        Stream<Long> filteredAdditionalCategories = additionalCategories.stream()
                .filter(BRAND_SAFETY_ADDITIONAL_CATEGORIES::contains);
        return Stream.concat(BRAND_SAFETY_BASE_CATEGORIES.stream(), filteredAdditionalCategories)
                .collect(Collectors.toList());
    }

    public static List<Long> getBrandSafetyAdditionalCategoriesForClient(Set<String> clientFeatures) {
        return BRAND_SAFETY_ADDITIONAL_CATEGORIES_TO_FEATURE.entrySet()
                .stream()
                .filter(entry -> clientFeatures.contains(entry.getValue().getName()))
                .map(Map.Entry::getKey)
                .collect(toList());
    }

    /**
     * Клиент может использовать категорию Brand Safety, если она является базовой, либо если ему выдана фича для нее
     */
    public static boolean checkIfBrandSafetyCategoryIsAvailableForClient(Long categoryId, Set<String> clientFeatures) {
        return BRAND_SAFETY_BASE_CATEGORIES.contains(categoryId) ||
                (BRAND_SAFETY_ADDITIONAL_CATEGORIES.contains(categoryId) &&
                        clientFeatures.contains(BRAND_SAFETY_ADDITIONAL_CATEGORIES_TO_FEATURE.get(categoryId).getName()));
    }
}
