package ru.yandex.direct.core.entity.currency.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;

import ru.yandex.direct.currency.CurrencyCode;

/**
 * Допустимые валюты для стран
 */
@ParametersAreNonnullByDefault
public final class CountryCurrencies {
    /**
     * Допустимые валюты по странам для агентств (region_id => список валют)
     */
    private final Multimap<Long, CurrencyCode> agenciesCurrencies;
    /**
     * Допустимые валюты по странам для обычных клиентов (region_id => список валют)
     */
    private final Multimap<Long, CurrencyCode> clientCurrencies;

    private CountryCurrencies(
            Multimap<Long, CurrencyCode> agenciesCurrencies,
            Multimap<Long, CurrencyCode> clientCurrencies) {
        Objects.requireNonNull(agenciesCurrencies, "agenciesCurrencies");
        Objects.requireNonNull(clientCurrencies, "clientCurrencies");

        this.agenciesCurrencies = ImmutableMultimap.copyOf(agenciesCurrencies);
        this.clientCurrencies = ImmutableMultimap.copyOf(clientCurrencies);
    }

    /**
     * Сконструировать допустимые валюты для стран из map-ы
     *
     * @param map Отображение true => данные по агентствам, false => данные по обычным клиентам
     */
    @Nonnull
    public static CountryCurrencies fromMap(Map<Boolean, Multimap<Long, CurrencyCode>> map) {
        Objects.requireNonNull(map, "map");
        return new CountryCurrencies(
                // Агентства
                Optional.ofNullable(map.get(Boolean.TRUE))
                        .orElseGet(HashMultimap::create),
                // Обычные клиенты
                Optional.ofNullable(map.get(Boolean.FALSE))
                        .orElseGet(HashMultimap::create));
    }

    @Nonnull
    public Multimap<Long, CurrencyCode> getAgenciesCurrencies() {
        return agenciesCurrencies;
    }

    @Nonnull
    public Multimap<Long, CurrencyCode> getClientCurrencies() {
        return clientCurrencies;
    }
}
