package ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BinaryOperator;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.utils.CommonUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Ограничения на ставку для ретаргетингов данной группы на главной
 * Зависят от регионов на группе и типа показа соответствующей кампании на главной
 * Есть ограничения на минимальную и максимальную ставку, которые нельзя нарушать вообще (выдаётся ошибка),
 * и те, которые запрещают показы в определённых регионах (выдаётся ворнинг)
 */
public class CpmYndxFrontpageAdGroupPriceRestrictions {
    //ограничение по минимальной цене, нарушение которого приводит к ошибке
    private BigDecimal cpmYndxFrontpageMinPrice;
    //ограничение по максимальной цене, нарушение которого приводит к ошибке
    private BigDecimal cpmYndxFrontpageMaxPrice;
    //ограничения по минимальной цене, полученное по тем регионам, по которым может быть ворнинг
    //считаем отдельно для каждого типа площадок кампании на главной
    private Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> minPriceByRegion;
    //ограничения по максимальной цене, полученное по тем регионам, по которым может быть ворнинг
    //считаем отдельно для каждого типа площадок кампании на главной
    private Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> maxPriceByRegion;
    //мапа регионов по их идентификаторам
    private Map<Long, Region> regionsById;
    //валюта клиента
    private Currency clientCurrency;

    public CpmYndxFrontpageAdGroupPriceRestrictions(Currency clientCurrency) {
        this.clientCurrency = clientCurrency;
    }

    public CpmYndxFrontpageAdGroupPriceRestrictions(BigDecimal cpmYndxFrontpageMinPrice,
                                                    BigDecimal cpmYndxFrontpageMaxPrice) {
        this.cpmYndxFrontpageMinPrice = cpmYndxFrontpageMinPrice;
        this.cpmYndxFrontpageMaxPrice = cpmYndxFrontpageMaxPrice;
    }

    public BigDecimal getCpmYndxFrontpageMinPrice() {
        return nvl(cpmYndxFrontpageMinPrice, clientCurrency.getMinCpmPrice());
    }

    public BigDecimal getCpmYndxFrontpageMaxPrice() {
        return nvl(cpmYndxFrontpageMaxPrice, clientCurrency.getMaxCpmPrice());
    }

    public Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> getMinPriceByRegion() {
        return nvl(minPriceByRegion, Collections.emptyMap());
    }

    public CpmYndxFrontpageAdGroupPriceRestrictions withMinPriceByRegion(
            Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> minPriceByRegion) {
        this.minPriceByRegion = minPriceByRegion;
        return this;
    }

    public Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> getMaxPriceByRegion() {
        return nvl(maxPriceByRegion, Collections.emptyMap());
    }

    public CpmYndxFrontpageAdGroupPriceRestrictions withMaxPriceByRegion(
            Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> maxPriceByRegion) {
        this.maxPriceByRegion = maxPriceByRegion;
        return this;
    }

    public Map<Long, Region> getRegionsById() {
        return nvl(regionsById, Collections.emptyMap());
    }

    public CpmYndxFrontpageAdGroupPriceRestrictions withRegionsById(Map<Long, Region> regionsById) {
        this.regionsById = regionsById;
        return this;
    }

    public CpmYndxFrontpageAdGroupPriceRestrictions withClientCurrency(Currency clientCurrency) {
        this.clientCurrency = clientCurrency;
        return this;
    }

    public static CpmYndxFrontpageAdGroupPriceRestrictions mergeLessStrict(
            CpmYndxFrontpageAdGroupPriceRestrictions first, CpmYndxFrontpageAdGroupPriceRestrictions second) {
        return mergeLessStrict(List.of(first, second));
    }

    private static CpmYndxFrontpageAdGroupPriceRestrictions mergeLessStrict(
            List<CpmYndxFrontpageAdGroupPriceRestrictions> restrictions) {
        if (CollectionUtils.isEmpty(restrictions)) {
            return null;
        }
        Currency clientCurrency = restrictions.get(0).clientCurrency;
        checkState(restrictions.stream().allMatch(r -> clientCurrency == null && r.clientCurrency == null ||
                clientCurrency != null && r.clientCurrency.getCode() == clientCurrency.getCode()),
                "Restrictions have different client currencies");

        Optional<BigDecimal> cpmYndxFrontpageMinPrice = StreamEx.of(restrictions)
                .map(r -> r.cpmYndxFrontpageMinPrice).nonNull()
                .min(BigDecimal::compareTo);
        Optional<BigDecimal> cpmYndxFrontpageMaxPrice = StreamEx.of(restrictions)
                .map(r -> r.cpmYndxFrontpageMaxPrice).nonNull()
                .max(BigDecimal::compareTo);

        Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> minPriceByRegion =
                mergePriceByRegion(restrictions, r -> r.getMinPriceByRegion(),
                        // по идее ограничение по региону должно быть одинаковым, но на всякий случай берем слабее
                        CommonUtils::min);
        Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> maxPriceByRegion =
                mergePriceByRegion(restrictions, r -> r.getMaxPriceByRegion(),
                        // по идее ограничение по региону должно быть одинаковым, но на всякий случай берем слабее
                        CommonUtils::max);

        Map<Long, Region> regionsById = StreamEx.of(restrictions)
                .flatMapToEntry(r -> r.getRegionsById()).nonNull()
                // произвольно берем первый регион, считаем что эти справочные данные не отличаются
                .toMap((r1, r2) -> r1);

        return new CpmYndxFrontpageAdGroupPriceRestrictions(cpmYndxFrontpageMinPrice.orElse(null),
                cpmYndxFrontpageMaxPrice.orElse(null))
                .withMinPriceByRegion(minPriceByRegion)
                .withMaxPriceByRegion(maxPriceByRegion)
                .withClientCurrency(clientCurrency)
                .withRegionsById(regionsById);
    }

    private static Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> mergePriceByRegion(
            List<CpmYndxFrontpageAdGroupPriceRestrictions> restrictions,
            Function<CpmYndxFrontpageAdGroupPriceRestrictions,
                    Map<FrontpageCampaignShowType, Map<Long, BigDecimal>>> getter,
            BinaryOperator<BigDecimal> mergeFunction) {
        return StreamEx.of(restrictions)
                .flatMapToEntry(getter).nonNull()
                .flatMapValues(EntryStream::of)
                .grouping(toMap(Map.Entry::getKey, Map.Entry::getValue, mergeFunction));
    }
}
