package ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;

import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;

import static ru.yandex.direct.utils.CommonUtils.max;
import static ru.yandex.direct.utils.CommonUtils.min;

/**
 * Информация об ограничениях на минимальную и максимальную ставку для группы на главной для данного региона
 * при некотором типе показа кампаний на главной
 * Хранится минимальная и максимальная ставка для каждой валюты
 */
public class CpmYndxFrontpageRegionPriceRestrictions {
    private Map<CurrencyCode, BigDecimal> cpmYndxFrontpageMinPrices;
    private Map<CurrencyCode, BigDecimal> cpmYndxFrontpageMaxPrices;

    public CpmYndxFrontpageRegionPriceRestrictions(
            Map<CurrencyCode, BigDecimal> cpmYndxFrontpageMinPrices,
            Map<CurrencyCode, BigDecimal> cpmYndxFrontpageMaxPrices) {
        this.cpmYndxFrontpageMinPrices = cpmYndxFrontpageMinPrices;
        this.cpmYndxFrontpageMaxPrices = cpmYndxFrontpageMaxPrices;
    }

    public BigDecimal getMinFrontpagePrice(Currency currency) {
        return cpmYndxFrontpageMinPrices.get(currency.getCode());
    }

    public BigDecimal getMaxFrontpagePrice(Currency currency) {
        return cpmYndxFrontpageMaxPrices.get(currency.getCode());
    }

    /**
     * Слияние двух различных инстансов CpmYndxFrontpageRegionPriceRestrictions для разных регионов
     * Оставляем для каждой валюты самые строгие ограничение по цене и снизу, и сверху
     */
    public static CpmYndxFrontpageRegionPriceRestrictions mergeCpmYndxFrontpageRegionPriceRestrictionsMoreStrict(
            CpmYndxFrontpageRegionPriceRestrictions firstConstraint,
            CpmYndxFrontpageRegionPriceRestrictions secondConstraint) {
        return mergeCpmYndxFrontpageRegionPriceRestrictionsMoreStrict(
                ImmutableList.of(firstConstraint, secondConstraint));
    }

    /**
     * Слияние нескольких различных инстансов CpmYndxFrontpageRegionPriceRestrictions для разных регионов
     * Оставляем для каждой валюты самые строгие ограничение по цене и снизу, и сверху
     */
    public static CpmYndxFrontpageRegionPriceRestrictions mergeCpmYndxFrontpageRegionPriceRestrictionsMoreStrict(
            List<CpmYndxFrontpageRegionPriceRestrictions> cpmYndxFrontpageCurrencyInfoList) {
        Map<CurrencyCode, BigDecimal> minPricesResultMap = StreamEx.of(cpmYndxFrontpageCurrencyInfoList)
                .map(t -> t.cpmYndxFrontpageMinPrices)
                .flatCollection(t -> t.entrySet())
                .toMap(t -> t.getKey(), t -> t.getValue(), (p1, p2) -> max(p1, p2));
        Map<CurrencyCode, BigDecimal> maxPricesResultMap = StreamEx.of(cpmYndxFrontpageCurrencyInfoList)
                .map(t -> t.cpmYndxFrontpageMaxPrices)
                .flatCollection(t -> t.entrySet())
                .toMap(t -> t.getKey(), t -> t.getValue(), (p1, p2) -> min(p1, p2));
        return new CpmYndxFrontpageRegionPriceRestrictions(minPricesResultMap, maxPricesResultMap);
    }

    /**
     * Слияние нескольких различных инстансов CpmYndxFrontpageRegionPriceRestrictions для разных регионов
     * Оставляем для каждой валюты наименее строгие ограничение по цене и снизу, и сверху
     */
    public static CpmYndxFrontpageRegionPriceRestrictions mergeCpmYndxFrontpageRegionPriceRestrictionsLessStrict(
            List<CpmYndxFrontpageRegionPriceRestrictions> cpmYndxFrontpageCurrencyInfoList) {
        Map<CurrencyCode, BigDecimal> minPricesResultMap = StreamEx.of(cpmYndxFrontpageCurrencyInfoList)
                .map(t -> t.cpmYndxFrontpageMinPrices)
                .flatCollection(t -> t.entrySet())
                .toMap(t -> t.getKey(), t -> t.getValue(), (p1, p2) -> min(p1, p2));
        Map<CurrencyCode, BigDecimal> maxPricesResultMap = StreamEx.of(cpmYndxFrontpageCurrencyInfoList)
                .map(t -> t.cpmYndxFrontpageMaxPrices)
                .flatCollection(t -> t.entrySet())
                .toMap(t -> t.getKey(), t -> t.getValue(), (p1, p2) -> max(p1, p2));
        return new CpmYndxFrontpageRegionPriceRestrictions(minPricesResultMap, maxPricesResultMap);
    }

    public Map<CurrencyCode, BigDecimal> getCpmYndxFrontpageMinPrices() {
        return cpmYndxFrontpageMinPrices;
    }
}
