package ru.yandex.direct.core.entity.currency.repository;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageMinBid;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageRegionPriceRestrictions;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageBidCurrency;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppcdict.enums.CpmYndxFrontpageMinBidsCurrency;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.dbschema.ppcdict.tables.CpmYndxFrontpageMinBids.CPM_YNDX_FRONTPAGE_MIN_BIDS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Репозиторий для работы с таблицей ppcdict.cpm_yndx_frontpage_min_bids
 * В таблице хранятся ограничения на минимальные ставки для показа на главной
 */
@Repository
@ParametersAreNonnullByDefault
public class CpmYndxFrontpageMinBidsRepository {
    private final DslContextProvider databaseWrapperProvider;

    public final JooqMapperWithSupplier<CpmYndxFrontpageMinBid> frontpageMinBidReadWriteMapper
            = createCpmYndxFrontpageMinBidMapper();

    private static final Set<CpmYndxFrontpageMinBidsCurrency> supportedCurrencies =
            Arrays.stream(CpmYndxFrontpageMinBidsCurrency.values())
                    .filter(t -> Currencies.currencyExists(t.getLiteral()))
                    .collect(Collectors.toSet());

    @Autowired
    public CpmYndxFrontpageMinBidsRepository(DslContextProvider databaseWrapperProvider) {
        this.databaseWrapperProvider = databaseWrapperProvider;
    }

    /**
     * Получение всех ограничений на ставки в виде мапы типов показа кампании для главной
     * в мапу идентификаторов регионов в ограничения минимальных ставок по ним из базы
     */
    public Map<FrontpageCampaignShowType, Map<Long, CpmYndxFrontpageRegionPriceRestrictions>> getAllPriceRestrictions() {
        return StreamEx.of(fetchAllRows())
                .toMap(t -> t.getFrontpageCampaignShowType(),
                        t -> fromCpmYndxFrontpageMinBid(t),
                        (p1, p2) -> mergeRegionIdsToPriceRestrictionsMaps(p1, p2));
    }

    /**
     * Считывание всех строк из таблицы ppcdict.cpm_yndx_frontpage_min_bids
     */
    private List<CpmYndxFrontpageMinBid> fetchAllRows() {
        return databaseWrapperProvider.ppcdict()
                .select(CPM_YNDX_FRONTPAGE_MIN_BIDS.FRONTPAGE_TYPE, CPM_YNDX_FRONTPAGE_MIN_BIDS.REGION_ID,
                        CPM_YNDX_FRONTPAGE_MIN_BIDS.CURRENCY, CPM_YNDX_FRONTPAGE_MIN_BIDS.MIN_BID)
                .from(CPM_YNDX_FRONTPAGE_MIN_BIDS)
                .where(CPM_YNDX_FRONTPAGE_MIN_BIDS.CURRENCY.in(supportedCurrencies))
                .fetch()
                .map(frontpageMinBidReadWriteMapper::fromDb);
    }

    private static JooqMapperWithSupplier<CpmYndxFrontpageMinBid> createCpmYndxFrontpageMinBidMapper() {
        return JooqMapperWithSupplierBuilder.builder(CpmYndxFrontpageMinBid::new)
                .map(convertibleProperty(CpmYndxFrontpageMinBid.FRONTPAGE_CAMPAIGN_SHOW_TYPE,
                        CPM_YNDX_FRONTPAGE_MIN_BIDS.FRONTPAGE_TYPE,
                        FrontpageCampaignShowType::fromSource, FrontpageCampaignShowType::toSource))
                .map(property(CpmYndxFrontpageMinBid.REGION_ID, CPM_YNDX_FRONTPAGE_MIN_BIDS.REGION_ID))
                .map(convertibleProperty(CpmYndxFrontpageMinBid.FRONTPAGE_BID_CURRENCY,
                        CPM_YNDX_FRONTPAGE_MIN_BIDS.CURRENCY,
                        FrontpageBidCurrency::fromSource, FrontpageBidCurrency::toSource))
                .map(property(CpmYndxFrontpageMinBid.MIN_BID, CPM_YNDX_FRONTPAGE_MIN_BIDS.MIN_BID))
                .build();
    }

    private static CurrencyCode fromFrontpageBidCurrency(FrontpageBidCurrency frontpageBidCurrency) {
        if (frontpageBidCurrency == null) {
            return null;
        }
        return CurrencyCode.valueOf(frontpageBidCurrency.toString());
    }

    /**
     * Преобразование одной считанной строчки из таблицы ppcdict.cpm_yndx_frontpage_min_bids в мапу единичного размера
     * Используется в методе getAllPriceRestrictions: сначала все считанные строки переводятся в мапу требуемого вида
     * единичного размера, а затем мапы мёрджатся между собой методом mergeRegionIdsToPriceRestrictionsMaps
     */
    private static Map<Long, CpmYndxFrontpageRegionPriceRestrictions> fromCpmYndxFrontpageMinBid(
            CpmYndxFrontpageMinBid cpmYndxFrontpageMinBid) {
        return ImmutableMap
                .of(cpmYndxFrontpageMinBid.getRegionId(), new CpmYndxFrontpageRegionPriceRestrictions(ImmutableMap
                        .of(fromFrontpageBidCurrency(cpmYndxFrontpageMinBid.getFrontpageBidCurrency()),
                                cpmYndxFrontpageMinBid.getMinBid()),
                        emptyMap()));
    }

    private static Map<Long, CpmYndxFrontpageRegionPriceRestrictions> mergeRegionIdsToPriceRestrictionsMaps(
            Map<Long, CpmYndxFrontpageRegionPriceRestrictions> first,
            Map<Long, CpmYndxFrontpageRegionPriceRestrictions> second) {
        Map<Long, CpmYndxFrontpageRegionPriceRestrictions> result = new HashMap<>(first);
        second.forEach((key, value) ->
                result.merge(key, value, (info1, info2) -> CpmYndxFrontpageRegionPriceRestrictions
                        .mergeCpmYndxFrontpageRegionPriceRestrictionsMoreStrict(info1, info2)));
        return result;

    }
}
