package ru.yandex.direct.core.entity.currency.repository;

import java.time.Duration;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.currency.model.CurrencyConversionState;
import ru.yandex.direct.dbschema.ppc.enums.CurrencyConvertQueueState;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.CURRENCY_CONVERT_QUEUE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Работа с заявками на переход клиентов в реальную валюту. Использутеся для блокировки всех интерфейсов незадолго до начала конвертации.
 */
@Repository
@ParametersAreNonnullByDefault
public class CurrencyConvertQueueRepository {
    protected final DslContextProvider dslContextProvider;

    @Autowired
    public CurrencyConvertQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }


    /**
     * помечает клиента, для которого завершена конвертация валюты в Балансе
     * (поле {@code CURRENCY_CONVERT_QUEUE.BALANCE_CONVERT_FINISHED})
     */
    public void setBalanceSideConvertFinished(int shard, ClientId clientId) {
        dslContextProvider.ppc(shard)
                .update(CURRENCY_CONVERT_QUEUE)
                .set(CURRENCY_CONVERT_QUEUE.BALANCE_CONVERT_FINISHED, RepositoryUtils.TRUE)
                .where(CURRENCY_CONVERT_QUEUE.CLIENT_ID.eq(clientId.asLong())
                        .and(CURRENCY_CONVERT_QUEUE.BALANCE_CONVERT_FINISHED.eq(RepositoryUtils.FALSE)))
                .execute();
    }

    /**
     * @return true, если конвертация валюты клиента на стороне Баланса завершена.
     */
    public boolean isBalanceSideConvertFinished(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(DSL.val(1))
                .from(CURRENCY_CONVERT_QUEUE)
                .where(CURRENCY_CONVERT_QUEUE.CLIENT_ID.eq(clientId.asLong())
                        .and(CURRENCY_CONVERT_QUEUE.BALANCE_CONVERT_FINISHED.eq(RepositoryUtils.TRUE)))
                .limit(1)
                .fetchAny() != null;
    }

    /**
     * Возвращает коллекцию id клиентов (из {@code clientIds}) для которых запущен, либо скоро будет запущен, процесс конвертации валюты.
     * Насколько скоро - определяется параметром {@code stopOperationMinutesBeforeConvert}.
     * Клиенты со статусом CurrencyConversionState.DONE либо одним из {@code excludeStates} в итоговый список не попадут.
     */
    public Collection<ClientId> fetchConvertingClients(int shard, Collection<ClientId> clientIds,
                                                       Duration stopOperationMinutesBeforeConvert,
                                                       Collection<CurrencyConversionState> excludeStates) {
        Set<CurrencyConversionState> statesFilter = EnumSet.of(CurrencyConversionState.DONE);
        statesFilter.addAll(excludeStates);
        List<CurrencyConvertQueueState> jooqExcludeStates =
                mapList(statesFilter, CurrencyConversionState::jooqModelValue);
        return dslContextProvider.ppc(shard)
                .select(CURRENCY_CONVERT_QUEUE.CLIENT_ID)
                .from(CURRENCY_CONVERT_QUEUE)
                .where(CURRENCY_CONVERT_QUEUE.CLIENT_ID.in(mapList(clientIds, ClientId::asLong))
                        .and(CURRENCY_CONVERT_QUEUE.START_CONVERT_AT.lessThan(
                                SqlUtils.localDateTimeAdd(DSL.currentLocalDateTime(),
                                        stopOperationMinutesBeforeConvert))
                        )
                        .and(CURRENCY_CONVERT_QUEUE.STATE.notIn(jooqExcludeStates))
                )
                .fetch(r -> ClientId.fromLong(r.get(CURRENCY_CONVERT_QUEUE.CLIENT_ID)));
    }

}
