package ru.yandex.direct.core.entity.currency.repository;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record2;
import org.jooq.Record3;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.GuavaCollectors;
import ru.yandex.direct.core.entity.currency.model.AnyCountryCurrencyCount;
import ru.yandex.direct.core.entity.currency.model.CountryCurrencies;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.partitioningBy;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.dbschema.ppcdict.tables.CountryCurrencies.COUNTRY_CURRENCIES;

/**
 * Справочники по допустимым валютам для разных стран
 */
@Repository
@ParametersAreNonnullByDefault
public class CurrencyDictRepository {
    private final DslContextProvider databaseWrapperProvider;

    @Autowired
    public CurrencyDictRepository(DslContextProvider databaseWrapperProvider) {
        this.databaseWrapperProvider = Objects.requireNonNull(databaseWrapperProvider, "databaseWrapperProvider");
    }

    /**
     * Вернуть список допустимых валют по странам для агентств и обычных клиентов
     * <p>
     * См. Currencies.pm::get_currencies_by_country_hash_not_for_agency
     */
    public CountryCurrencies getCountryCurrencies() {
        return CountryCurrencies.fromMap(
                databaseWrapperProvider.ppcdict()
                        .select(COUNTRY_CURRENCIES.IS_AGENCY, COUNTRY_CURRENCIES.REGION_ID, COUNTRY_CURRENCIES.CURRENCY)
                        .from(COUNTRY_CURRENCIES)
                        // Чтобы добавление новых валют в базу не ломало обновлённый код
                        .where(COUNTRY_CURRENCIES.CURRENCY.in(Currencies.getCurrencies().keySet()))
                        .fetch()
                        .stream()
                        .collect(
                                partitioningBy(
                                        r -> r.value1().intValue() != 0, // Это агентство?
                                        GuavaCollectors.toMultimap(
                                                Record3::value2,
                                                r -> CurrencyCode.valueOf(r.value3())))));
    }

    /**
     * Вернуть общее кол-во допустимых валют по всем странам для агентств и обычных клиентов
     * <p>
     * См. Currencies.pm::get_any_country_currency_cnt_for_agency
     */
    public AnyCountryCurrencyCount getAnyCountryCurrencyCount() {
        return AnyCountryCurrencyCount.fromMap(
                databaseWrapperProvider.ppcdict()
                        .select(COUNTRY_CURRENCIES.IS_AGENCY, DSL.count())
                        .from(COUNTRY_CURRENCIES)
                        // Чтобы добавление новых валют в базу не ломало обновлённый код
                        .where(COUNTRY_CURRENCIES.CURRENCY.in(Currencies.getCurrencies().keySet()))
                        .groupBy(COUNTRY_CURRENCIES.IS_AGENCY)
                        .fetch()
                        .stream()
                        .collect(
                                toMap(
                                        r -> r.value1().intValue() != 0, // Это агентство?,
                                        Record2::value2)));
    }
}
