package ru.yandex.direct.core.entity.currency.repository;

import java.time.LocalDate;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.currency.model.CurrencyRate;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppcdict.tables.CurrencyRates.CURRENCY_RATES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class CurrencyRateRepository {
    private static final Logger logger = LoggerFactory.getLogger(CurrencyRateRepository.class);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<CurrencyRate> rateJooqMapper;

    @Autowired
    public CurrencyRateRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.rateJooqMapper = createCurrencyRateMapper();
    }

    @VisibleForTesting
    public static JooqMapperWithSupplier<CurrencyRate> createCurrencyRateMapper() {
        return JooqMapperWithSupplierBuilder.builder(CurrencyRate::new)
                .map(convertibleProperty(CurrencyRate.CURRENCY_CODE, CURRENCY_RATES.CURRENCY,
                        CurrencyCode::valueOf, CurrencyCode::name))
                .map(property(CurrencyRate.DATE, CURRENCY_RATES.DATE))
                .map(property(CurrencyRate.RATE, CURRENCY_RATES.RATE))
                .build();
    }

    @Nullable
    public CurrencyRate getCurrencyRate(CurrencyCode currencyCode, LocalDate date) {
        Record record = dslContextProvider.ppcdict()
                .select(rateJooqMapper.getFieldsToRead())
                .from(CURRENCY_RATES)
                .where(CURRENCY_RATES.CURRENCY.eq(currencyCode.name()))
                .and(CURRENCY_RATES.DATE.eq(date))
                .fetchOne();
        if (record == null) {
            logger.warn("Can't find currency rate by (code='{}', date='{}')", currencyCode, date);
            return null;
        }
        return record.map(rateJooqMapper::fromDb);
    }

    /**
     * @return Список самых свежих курсов валют, но не более свежих чем `maxDate`
     */
    public List<CurrencyRate> getLastCurrencyRates(LocalDate maxDate) {
        return getLastCurrencyRates(dslContextProvider.ppcdict(), maxDate);
    }

    List<CurrencyRate> getLastCurrencyRates(DSLContext dslContext, @Nullable LocalDate maxDate) {
        var datesFilterCondition = maxDate != null ? CURRENCY_RATES.DATE.le(maxDate) : DSL.trueCondition();
        // подзапрос с получением максимальных дат для валют
        Table<?> lastCurrencyDates = DSL
                .select(CURRENCY_RATES.CURRENCY, DSL.max(CURRENCY_RATES.DATE).as(CURRENCY_RATES.DATE))
                .from(CURRENCY_RATES)
                .where(datesFilterCondition)
                .and(CURRENCY_RATES.CURRENCY.in(Currencies.getCurrencies().keySet()))
                .groupBy(CURRENCY_RATES.CURRENCY).asTable();
        return dslContext.select(rateJooqMapper.getFieldsToRead())
                .from(lastCurrencyDates)
                .join(CURRENCY_RATES).on(
                        CURRENCY_RATES.CURRENCY.eq(lastCurrencyDates.field(CURRENCY_RATES.CURRENCY))
                                .and(CURRENCY_RATES.DATE.eq(lastCurrencyDates.field(CURRENCY_RATES.DATE)))
                ).fetch(rateJooqMapper::fromDb);
    }
}
