package ru.yandex.direct.core.entity.currency.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmYndxFrontpageCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageRegionPriceRestrictions;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.currency.repository.CpmYndxFrontpageMinBidsRepository;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageRegionPriceRestrictions.mergeCpmYndxFrontpageRegionPriceRestrictionsMoreStrict;
import static ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageGeoUtils.getAdGroupRegionsIncludedIntoRestrictedRegions;
import static ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageGeoUtils.getAdGroupRegionsInclusiveRestrictedRegions;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Получение CpmYndxFrontpageAdGroupPriceRestrictions для каждой из групп переданного списка
 */
@Service
@ParametersAreNonnullByDefault
public class CpmYndxFrontpageCurrencyService {

    private final CampaignRepository campaignRepository;
    private final GeoTreeFactory geoTreeFactory;
    private final ClientService clientService;
    private final CpmYndxFrontpageMinBidsRepository cpmYndxFrontpageMinBidsRepository;
    private final AdGroupRepository adGroupRepository;

    private Map<FrontpageCampaignShowType, Map<Long, CpmYndxFrontpageRegionPriceRestrictions>>
            frontpageRegionPriceRestrictions;

    @Autowired
    public CpmYndxFrontpageCurrencyService(CampaignRepository campaignRepository, GeoTreeFactory geoTreeFactory,
                                           ClientService clientService, CpmYndxFrontpageMinBidsRepository cpmYndxFrontpageMinBidsRepository,
                                           AdGroupRepository adGroupRepository) {
        this.campaignRepository = campaignRepository;
        this.geoTreeFactory = geoTreeFactory;
        this.clientService = clientService;
        this.cpmYndxFrontpageMinBidsRepository = cpmYndxFrontpageMinBidsRepository;
        this.adGroupRepository = adGroupRepository;
        this.frontpageRegionPriceRestrictions = emptyMap();
    }

    /**
     * Получение по списку групп мапу их идентификаторов в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups список групп
     * @param shard    номер шарда
     * @param clientId идентификатор клиента
     */
    public <A extends AdGroupSimple> Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIdsToPriceDataMapByAdGroups(
            Collection<A> adGroups, int shard, ClientId clientId) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        return getAdGroupIdsToPriceDataMapByAdGroups(adGroups, shard, clientCurrency);
    }

    /**
     * Получение по списку групп мапу их идентификаторов в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups список групп
     * @param shard    номер шарда
     * @param currency валюта клиента
     */
    public <A extends AdGroupSimple> Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIdsToPriceDataMapByAdGroups(
            Collection<A> adGroups, int shard, Currency currency) {

        return getAdGroupIdsToPriceDataMapByAdGroups(adGroups, shard, currency, emptyMap());
    }

    /**
     * Получение по списку групп мапу их идентификаторов в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups список групп
     * @param shard    номер шарда
     * @param currency валюта клиента
     * @param campaignShowTypeOverrideMap переопределение мапы типа кампаний на главной в список их площадок показа, чтобы брать не из базы
     */
    private <A extends AdGroupSimple> Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIdsToPriceDataMapByAdGroups(
            Collection<A> adGroups, int shard, Currency currency,
            Map<Long, Set<FrontpageCampaignShowType>> campaignShowTypeOverrideMap) {

        Map<Long, A> adGroupsById = StreamEx.of(adGroups).toMap(AdGroupSimple::getId, identity(), (a1, a2) -> a1);
        return getPriceRestrictionsByAdGroupsWithIdentifier(adGroupsById, shard, currency, campaignShowTypeOverrideMap);
    }

    /**
     * Получение по списку групп мапу их индексов в списке в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups список групп
     * @param clientId идентификатор клиента
     * @param shard    номер шарда
     */
    public <A extends AdGroupSimple> Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIndexesToPriceDataMapByAdGroups(
            List<A> adGroups, int shard, ClientId clientId) {
        return getAdGroupIndexesToPriceDataMapByAdGroups(adGroups, shard, clientId, emptyMap());
    }

    /**
     * Получение по списку групп мапу их индексов в списке в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups                    список групп
     * @param clientId                    идентификатор клиента
     * @param shard                       номер шарда
     * @param campaignShowTypeOverrideMap переопределение мапы типа кампаний на главной в список их площадок показа, чтобы брать не из базы
     */
    public <A extends AdGroupSimple> Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIndexesToPriceDataMapByAdGroups(
            List<A> adGroups, int shard, ClientId clientId,
            Map<Long, Set<FrontpageCampaignShowType>> campaignShowTypeOverrideMap) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        return getAdGroupIndexesToPriceDataMapByAdGroups(adGroups, shard, clientCurrency, campaignShowTypeOverrideMap);
    }

    /**
     * Получение по списку групп мапу их индексов в списке в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups список групп
     * @param currency валюта клиента
     * @param shard    номер шарда
     */
    public <A extends AdGroupSimple> Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIndexesToPriceDataMapByAdGroups(
            List<A> adGroups, int shard, Currency currency) {
        return getAdGroupIndexesToPriceDataMapByAdGroups(adGroups, shard, currency, emptyMap());
    }

    /**
     * Получение по списку групп мапу их индексов в списке в CpmYndxFrontpageAdGroupPriceRestrictions по группе
     *
     * @param adGroups                    список групп
     * @param currency                    валюта клиента
     * @param shard                       номер шарда
     * @param campaignShowTypeOverrideMap переопределение мапы типа кампаний на главной в список их площадок показа, чтобы брать не из базы
     */
    public <A extends AdGroupSimple> Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> getAdGroupIndexesToPriceDataMapByAdGroups(
            List<A> adGroups, int shard, Currency currency,
            Map<Long, Set<FrontpageCampaignShowType>> campaignShowTypeOverrideMap) {
        Map<Integer, A> adGroupsByIndex = EntryStream.of(adGroups).toMap();
        return getPriceRestrictionsByAdGroupsWithIdentifier(adGroupsByIndex, shard, currency,
                campaignShowTypeOverrideMap);
    }

    /**
     * Получение аггрегированного CpmYndxFrontpageAdGroupPriceRestrictions по кампаниям
     *
     * @param campaigns список кампаний
     * @param shard     номер шарда
     * @param clientId  идентификатор клиента
     */
    public Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> getPriceDataByCampaigns(
            List<? extends BaseCampaign> campaigns, int shard, ClientId clientId) {
        Map<Long, CpmYndxFrontpageCampaign> frontpageCampaignMap = StreamEx.of(campaigns)
                .select(CpmYndxFrontpageCampaign.class)
                .toMap(BaseCampaign::getId, Function.identity());
        if (frontpageCampaignMap.isEmpty()) {
            return Map.of();
        }
        Map<Long, List<AdGroupSimple>> adGroupsByFrontpageCampaignIds =
                adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, frontpageCampaignMap.keySet());
        if (adGroupsByFrontpageCampaignIds.isEmpty()) {
            return Map.of();
        }

        List<AdGroupSimple> allAdGroups = StreamEx.of(adGroupsByFrontpageCampaignIds.values()).toFlatList(identity());
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        Map<Long, Set<FrontpageCampaignShowType>> frontpageShowTypeByCampaigns =
                StreamEx.of(frontpageCampaignMap.values())
                        .toMap(BaseCampaign::getId, CpmYndxFrontpageCampaign::getAllowedFrontpageType);
        Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> restrictionsByAdGroupId =
                getAdGroupIdsToPriceDataMapByAdGroups(allAdGroups, shard, clientCurrency, frontpageShowTypeByCampaigns);
        return EntryStream.of(adGroupsByFrontpageCampaignIds)
                .flatMapValues(StreamEx::of)
                .mapValues(adGroup -> restrictionsByAdGroupId.get(adGroup.getId()))
                .nonNullValues()
                .toMap(CpmYndxFrontpageAdGroupPriceRestrictions::mergeLessStrict);
    }

    /**
     * Получение по мапе идентификаторов групп в сами группы экземпляров CpmYndxFrontpageAdGroupPriceRestrictions
     *
     * @param adGroups                    мапа идентификаторов в группы
     * @param shard                       номер шарда
     * @param currency                    валюта клиента
     * @param campaignShowTypeOverrideMap переопределение мапы типа кампаний на главной в список их площадок показа, чтобы брать не из базы
     */
    private <Identifier, A extends AdGroupSimple> Map<Identifier, CpmYndxFrontpageAdGroupPriceRestrictions> getPriceRestrictionsByAdGroupsWithIdentifier(
            Map<Identifier, A> adGroups, int shard, Currency currency,
            Map<Long, Set<FrontpageCampaignShowType>> campaignShowTypeOverrideMap) {
        adGroups = filterYndxFrontpageAdGroups(adGroups);
        if (adGroups.isEmpty()) {
            return emptyMap();
        }

        if (frontpageRegionPriceRestrictions.isEmpty()) {
            frontpageRegionPriceRestrictions = cpmYndxFrontpageMinBidsRepository.getAllPriceRestrictions();
            checkState(!frontpageRegionPriceRestrictions.isEmpty(),
                    "cpm_yndx_frontpage price restrictions must be provided in database");
        }
        checkState(campaignShowTypeOverrideMap != null, "CpmYndxFrontpage ShowTypeOverrideMap must not be null");

        fillCampaignIdValues(adGroups, shard);
        Map<Identifier, A> applicableAdGroups = filterApplicableAdGroups(adGroups, shard);
        Map<Long, Set<FrontpageCampaignShowType>> frontpageShowTypeByCampaigns = campaignShowTypeOverrideMap.isEmpty() ?
                getFrontpageShowTypeByCampaigns(applicableAdGroups, shard, !campaignShowTypeOverrideMap.isEmpty()) :
                campaignShowTypeOverrideMap;

        return EntryStream.of(adGroups)
                .mapToValue((identifier, adGroup) -> {
                    if (applicableAdGroups.containsKey(identifier)) {
                        return getAdGroupRestrictions(adGroup.getGeo(), currency,
                                getShowTypesToRegionPriceRestrictionsMap(
                                        frontpageShowTypeByCampaigns.get(adGroup.getCampaignId())));
                    } else {
                        // для групп к которым не применимы ограничения по цене - будем проверять только ограничения валюты
                        return new CpmYndxFrontpageAdGroupPriceRestrictions(currency);
                    }
                })
                .toMap();
    }

    private <Identifier, A extends AdGroupSimple> Map<Identifier, A> filterYndxFrontpageAdGroups(Map<Identifier, A> adGroups) {
        if (adGroups.isEmpty()) {
            return emptyMap();
        }

        return EntryStream.of(adGroups)
                .filterValues(adGroup -> adGroup.getType() == AdGroupType.CPM_YNDX_FRONTPAGE)
                .toMap();
    }

    private <Identifier, A extends AdGroupSimple> Map<Identifier, A> filterApplicableAdGroups(Map<Identifier, A> adGroups, int shard) {
        if (adGroups.isEmpty()) {
            return emptyMap();
        }

        var campaignsId = mapList(adGroups.values(), AdGroupSimple::getCampaignId);
        var campaignsType = campaignRepository.getCampaignsTypeMap(shard, campaignsId);
        return EntryStream.of(adGroups)
                .filterValues(adGroup -> campaignsType.get(adGroup.getCampaignId()) != CampaignType.CPM_PRICE)
                .toMap();
    }

    /**
     * В комплексной операции обновления теоретически могут возникнуть группы с id, но без id кампаний
     * Для таких групп типа AdGroupType.CPM_YNDX_FRONTPAGE заполним id кампаний из базы
     */
    private <Identifier, A extends AdGroupSimple> void fillCampaignIdValues(Map<Identifier, A> adGroups, int shard) {
        List<Long> adGroupIdsWithoutCampaignIds = adGroups.values()
                .stream()
                .filter(adGroup -> adGroup.getCampaignId() == null && adGroup.getId() != null)
                .map(AdGroupSimple::getId)
                .collect(Collectors.toList());
        if (adGroupIdsWithoutCampaignIds.isEmpty()) {
            return;
        }
        Map<Long, Long> cidsByPids = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIdsWithoutCampaignIds);
        adGroups.forEach((index, adGroup) -> {
            if (adGroup.getId() != null) {
                adGroup.withCampaignId(
                        adGroup.getCampaignId() != null ? adGroup.getCampaignId() : cidsByPids.get(adGroup.getId()));
            }
            checkState(adGroup.getCampaignId() != null,
                    "У каждой группы на главной должен по итогу быть заполнен идентификатор кампании");
        });
    }

    /**
     * Получаем из базы типы разрешённых площадок показа для кампаний по главной
     * Проверяем, что для каждого типа в базе указан тип, причём валидный
     * Если кампания новая, не делаем проверок
     */
    private <Identifier, A extends AdGroupSimple> Map<Long, Set<FrontpageCampaignShowType>> getFrontpageShowTypeByCampaigns(
            Map<Identifier, A> adGroups, int shard, boolean suppressCheckState) {
        Map<Long, Set<FrontpageCampaignShowType>> frontpageShowTypeByCampaigns = campaignRepository
                .getFrontpageTypesForCampaigns(shard,
                        mapList(adGroups.values(), AdGroupSimple::getCampaignId));
        if (!suppressCheckState) {
            Set<Long> uniqueFrontpageCampaignIds = adGroups.values()
                    .stream()
                    .map(AdGroupSimple::getCampaignId)
                    .collect(Collectors.toSet());
            checkState(uniqueFrontpageCampaignIds.size() == frontpageShowTypeByCampaigns.size(),
                    "Для каждой кампании на главной должна быть указана площадка");
            checkState(frontpageShowTypeByCampaigns.values().stream().noneMatch(t -> t.isEmpty()),
                    "Набор типов площадок для кампании на главной в базе должен быть не пустым");
        }
        return frontpageShowTypeByCampaigns;
    }

    /**
     * Получаем по набору разрешённых площадок показа кампании на главной список актуальных для неё ограничений по цене
     */
    private Map<FrontpageCampaignShowType, Map<Long, CpmYndxFrontpageRegionPriceRestrictions>> getShowTypesToRegionPriceRestrictionsMap(
            Set<FrontpageCampaignShowType> campaignShowTypes) {
        return EntryStream.of(frontpageRegionPriceRestrictions)
                .filterKeys(showType -> campaignShowTypes.contains(showType))
                .toMap();
    }

    /**
     * Получение экземпляра CpmYndxFrontpageAdGroupRestrictions для одной группы
     *
     * @param geo                                       географический таргетинг данной группы
     * @param currency                                  валюта клиента
     * @param regionPriceRestrictionsByCampaignShowType по каждому типу площадок данной кампании храним мапу
     *                                                  идентификаторов регионов в ограничения на ставку по ним, полученные из базы
     */
    private CpmYndxFrontpageAdGroupPriceRestrictions getAdGroupRestrictions(List<Long> geo, Currency currency,
                                                                            Map<FrontpageCampaignShowType, Map<Long, CpmYndxFrontpageRegionPriceRestrictions>> regionPriceRestrictionsByCampaignShowType) {
        Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> minPriceByRegion = new HashMap<>();
        Map<FrontpageCampaignShowType, Map<Long, BigDecimal>> maxPriceByRegion = new HashMap<>();
        GeoTree geoTree = getGeoTree();
        for (FrontpageCampaignShowType campaignShowType : regionPriceRestrictionsByCampaignShowType.keySet()) {
            Map<Long, CpmYndxFrontpageRegionPriceRestrictions> regionPriceRestrictions =
                    regionPriceRestrictionsByCampaignShowType.get(campaignShowType);
            //получаем список всех регионов, по которым могут быть ворнинги,
            //далее получаем минимальную и максимальную допустимые цены по ним
            //регионы записываем в мапы вида регион, который выдадим в ворнинге->регион с ограничениями из базы, ему соответствующий

            //сначала находим регионы geo, строго содержащие в себе регионы regionPriceRestrictions
            Map<Long, Long> regionsWithWarningsInclusive =
                    getAdGroupRegionsInclusiveRestrictedRegions(geoTree, geo, regionPriceRestrictions.keySet())
                            .stream()
                            .collect(Collectors.toMap(t -> t, t -> t));
            //потом находим регионы geo, которые содержатся в регионах regionPriceRestrictions
            Map<Long, Long> regionsWithWarningsIncluded =
                    getAdGroupRegionsIncludedIntoRestrictedRegions(geoTree, geo, regionPriceRestrictions.keySet());

            //теперь для найденных регионов находим минимальные и максимальные допустимые ставки
            //и кладём в minPriceByRegion и maxPriceByRegion
            EntryStream.of(regionsWithWarningsInclusive)
                    .append(regionsWithWarningsIncluded)
                    .mapValues(regionPriceRestrictions::get)
                    .filterValues(Objects::nonNull)
                    .forKeyValue((regionId, priceRestriction) -> {
                        BigDecimal minPrice = priceRestriction.getMinFrontpagePrice(currency);
                        BigDecimal maxPrice = priceRestriction.getMaxFrontpagePrice(currency);
                        if (minPrice != null) {
                            //Кладём максимальную из вновь пришедшей и имеющейся ставки в minPriceByRegion
                            minPriceByRegion.computeIfAbsent(campaignShowType, t -> new HashMap<>())
                                    .compute(regionId, (id, oldPrice) -> minPrice.max(nvl(oldPrice, minPrice)));
                        }
                        if (maxPrice != null) {
                            //Кладём минимальную из вновь пришедшей и имеющейся ставки в maxPriceByRegion
                            maxPriceByRegion.computeIfAbsent(campaignShowType, t -> new HashMap<>())
                                    .compute(regionId, (id, oldPrice) -> maxPrice.min(nvl(oldPrice, maxPrice)));
                        }
                    });
        }

        Set<Long> regionsWithAnyWarnings = new HashSet<>();
        minPriceByRegion.values().forEach(t -> regionsWithAnyWarnings.addAll(t.keySet()));
        maxPriceByRegion.values().forEach(t -> regionsWithAnyWarnings.addAll(t.keySet()));

        return calcMinMaxAdGroupPriceRestrictionsFromGeo(geo, regionPriceRestrictionsByCampaignShowType, currency)
                .withMinPriceByRegion(minPriceByRegion)
                .withMaxPriceByRegion(maxPriceByRegion)
                .withRegionsById(StreamEx.of(regionsWithAnyWarnings).toMap(geoTree::getRegion))
                .withClientCurrency(currency);
    }

    private GeoTree getGeoTree() {
        //Берётся для однозначной обработки Крыма DIRECT-92239
        return geoTreeFactory.getRussianGeoTree();
    }

    /**
     * Заполнение полей CpmYndxFrontpageAdGroupRestrictions, соответствующих
     * таким значениям минимальной и максимальной цены, чтобы не было ошибки валидации
     */
    private CpmYndxFrontpageAdGroupPriceRestrictions calcMinMaxAdGroupPriceRestrictionsFromGeo(List<Long> geo,
                                                                                               Map<FrontpageCampaignShowType, Map<Long, CpmYndxFrontpageRegionPriceRestrictions>> regionPriceRestrictionsByCampaignShowType,
                                                                                               Currency currency) {
        List<CpmYndxFrontpageRegionPriceRestrictions> priceRestrictions = new ArrayList<>();
        for (Map<Long, CpmYndxFrontpageRegionPriceRestrictions> regionPriceRestrictions :
                regionPriceRestrictionsByCampaignShowType.values()) {
            Set<Long> regionsWithRestrictions = regionPriceRestrictions.keySet();
            Set<Long> regionsWithErrors = new HashSet<>(
                    getAdGroupRegionsIncludedIntoRestrictedRegions(getGeoTree(), geo, regionsWithRestrictions).values());
            priceRestrictions.addAll(mapList(regionsWithErrors, regionPriceRestrictions::get));
        }

        CpmYndxFrontpageRegionPriceRestrictions result = CpmYndxFrontpageRegionPriceRestrictions
                .mergeCpmYndxFrontpageRegionPriceRestrictionsLessStrict(priceRestrictions);
        return new CpmYndxFrontpageAdGroupPriceRestrictions(result.getMinFrontpagePrice(currency),
                result.getMaxFrontpagePrice(currency));
    }

    /**
     * хэш валюты на рекомендованную ставку для медийки на главной
     */
    public Map<CurrencyCode, BigDecimal>  getDefaultCpmFrontpagePrice() {
        if (frontpageRegionPriceRestrictions.isEmpty()) {
            frontpageRegionPriceRestrictions = cpmYndxFrontpageMinBidsRepository.getAllPriceRestrictions();
        }
        var restriction = mergeCpmYndxFrontpageRegionPriceRestrictionsMoreStrict(
                StreamEx.of(frontpageRegionPriceRestrictions.values())
                .flatMap(it -> it.values().stream())
                .toList());
        return restriction.getCpmYndxFrontpageMinPrices();
    }
}
