package ru.yandex.direct.core.entity.currency.service;

import java.math.BigDecimal;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.MoneyUtils;

/**
 * Хелпер для частной конвертации из одной валюты в другую, поддерживает опциональные верхнюю и нижнюю границы
 */
@ParametersAreNonnullByDefault
public class CurrencyConverter {
    private final CurrencyRateService currencyRateService;
    private final CurrencyCode currencyFrom;
    private final CurrencyCode currencyTo;

    public CurrencyConverter(CurrencyRateService currencyRateService, CurrencyCode currencyFrom,
                             CurrencyCode currencyTo) {
        this.currencyRateService = currencyRateService;
        this.currencyFrom = currencyFrom;
        this.currencyTo = currencyTo;
    }

    public CurrencyCode getCurrencyFrom() {
        return currencyFrom;
    }

    public CurrencyCode getCurrencyTo() {
        return currencyTo;
    }

    /**
     * Конвертирует сумму из одной валюты в другую, с учетом лимитов - не меньше минимума (если он задан), не больше
     * максимума (если он задан). Валюты заданы в самом конвертере.
     *
     * @param sum       конвертируемая сумма
     * @param lowLimit  ограничение по сумме снизу - конвертация вернет не меньше этой суммы
     * @param highLimit ограничение по сумме сверху - конвертация вернет не больше этой суммы
     * @return сконевертированная сумма
     */
    public BigDecimal convert(@Nullable BigDecimal sum, @Nullable BigDecimal lowLimit, @Nullable BigDecimal highLimit) {
        if (sum == null) {
            return null;
        }

        Money converted = currencyRateService.convertMoney(Money.valueOf(sum, currencyFrom), currencyTo);
        if (lowLimit != null) {
            converted = MoneyUtils.max(converted, Money.valueOf(lowLimit, currencyTo));
        }
        if (highLimit != null) {
            converted = MoneyUtils.min(converted, Money.valueOf(highLimit, currencyTo));
        }

        return converted.bigDecimalValue();
    }
}
