package ru.yandex.direct.core.entity.currency.service;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.currency.model.AnyCountryCurrencyCount;
import ru.yandex.direct.core.entity.currency.model.CountryCurrencies;
import ru.yandex.direct.core.entity.currency.repository.CurrencyDictRepository;

/**
 * Кеш справочников допустимых валют для разных стран (См. Currencies.pm)
 */
@Service
@ParametersAreNonnullByDefault
public class CurrencyDictCache {
    private static final int COUNTRY_CURRENCIES_RELOAD_INTERVAL_IN_HOURS = 24;

    /**
     * Кеш допустимых валют по разным странам для агентств и обычных клиентов
     */
    private final Supplier<CountryCurrencies> countryCurrenciesCache;

    /**
     * Кеш максимально возможных количеств валют для агентств и обычных клиентов
     */
    private final Supplier<AnyCountryCurrencyCount> anyCountryCurrencyCountCache;

    @Autowired
    public CurrencyDictCache(CurrencyDictRepository currencyDictRepository) {
        Objects.requireNonNull(currencyDictRepository, "currencyRepository");
        this.countryCurrenciesCache = Suppliers.memoizeWithExpiration(
                currencyDictRepository::getCountryCurrencies,
                COUNTRY_CURRENCIES_RELOAD_INTERVAL_IN_HOURS,
                TimeUnit.HOURS);
        this.anyCountryCurrencyCountCache = Suppliers.memoizeWithExpiration(
                currencyDictRepository::getAnyCountryCurrencyCount,
                COUNTRY_CURRENCIES_RELOAD_INTERVAL_IN_HOURS,
                TimeUnit.HOURS);
    }

    /**
     * @see CurrencyDictRepository#getCountryCurrencies()
     */
    public CountryCurrencies getCountryCurrencies() {
        return countryCurrenciesCache.get();
    }

    /**
     * @see CurrencyDictRepository#getAnyCountryCurrencyCount()
     */
    public AnyCountryCurrencyCount getAnyCountryCurrencyCount() {
        return anyCountryCurrencyCountCache.get();
    }
}
