package ru.yandex.direct.core.entity.currency.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.currency.model.CurrencyRate;
import ru.yandex.direct.core.entity.currency.repository.CurrencyRateRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Сервис для конвертации валют по курсу на заданную дату.
 * <p>
 * Референсная реализация в Perl: {@code protected/Currency/Rate.pm}.
 */
@Service
@ParametersAreNonnullByDefault
public class CurrencyRateService {

    private static final BigDecimal YND_FIXED_RATE = BigDecimal.valueOf(30);

    private final CurrencyRateRepository currencyRateRepository;

    private Cache<CurrencyWithDate, CurrencyRate> currencyRateCache;

    @Autowired
    public CurrencyRateService(CurrencyRateRepository currencyRateRepository) {
        this.currencyRateRepository = currencyRateRepository;

        // Кэшировать навечно не хотим, поскольку курс в БД может плавать
        currencyRateCache = CacheBuilder.newBuilder()
                .maximumSize(100)
                .expireAfterWrite(1, TimeUnit.HOURS)
                .build();
    }

    /**
     * Конвертирует валюту по курсу на сегодняшний день (в часовой зоне MSK)
     */
    @SuppressWarnings("WeakerAccess")
    public Money convertMoney(Money money, CurrencyCode targetCurrency) {
        return convertMoney(money, targetCurrency, null);
    }

    @SuppressWarnings("SameParameterValue")
    private Money convertMoney(Money money, CurrencyCode targetCurrency, @Nullable LocalDate date) {
        CurrencyCode originalCurrency = money.getCurrencyCode();
        if (originalCurrency == targetCurrency) {
            return money;
        }
        LocalDate targetDate = date != null ? date : LocalDate.now(MSK);

        BigDecimal amount = money.bigDecimalValue();
        BigDecimal amountInTargetCurrency = convertCurrency(amount, originalCurrency, targetCurrency, targetDate);

        return Money.valueOf(amountInTargetCurrency, targetCurrency);
    }

    private BigDecimal convertCurrency(BigDecimal amount, CurrencyCode from, CurrencyCode to, LocalDate date) {
        if (to == from) {
            return amount;
        }
        CurrencyRate fromRate = getCurrencyRate(from, date);
        CurrencyRate toRate = getCurrencyRate(to, date);

        BigDecimal rate = toRate.getRate().divide(fromRate.getRate(), Money.MONEY_MATH_CONTEXT);
        return amount.divide(rate, Money.MONEY_MATH_CONTEXT);
    }

    @Nonnull
    private CurrencyRate getCurrencyRate(CurrencyCode currencyCode, LocalDate date) {
        // Для рублей и YND_FIXED курс фиксирован, поэтому из базы их не получить
        if (currencyCode == CurrencyCode.RUB) {
            return new CurrencyRate().withCurrencyCode(CurrencyCode.RUB).withDate(date).withRate(BigDecimal.ONE);
        } else if (currencyCode == CurrencyCode.YND_FIXED) {
            return new CurrencyRate().withCurrencyCode(CurrencyCode.YND_FIXED).withDate(date).withRate(YND_FIXED_RATE);
        }

        LocalDate today = LocalDate.now(MSK);
        checkArgument(date.compareTo(today) <= 0, "You can't get currencyRate for future date");

        CurrencyWithDate cacheKey = new CurrencyWithDate(currencyCode, today);
        CurrencyRate currencyRate = currencyRateCache.getIfPresent(cacheKey);
        if (currencyRate == null) {
            // обрабатываем явно, чтобы не обрабатывать ExecutionException метода get(key, loader)
            currencyRate = currencyRateRepository.getCurrencyRate(currencyCode, date);
            if (currencyRate == null) {
                throw new IllegalStateException("Can't find currency rate by '" + currencyCode + "' for date " + date);
            }
            currencyRateCache.put(cacheKey, currencyRate);
        }

        return currencyRate;
    }

    /**
     * Экземпляр используется как ключ в кэше
     */
    private static class CurrencyWithDate {
        private final CurrencyCode currencyCode;
        private final LocalDate date;

        private CurrencyWithDate(CurrencyCode currencyCode, LocalDate date) {
            this.currencyCode = currencyCode;
            this.date = date;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            CurrencyWithDate that = (CurrencyWithDate) o;
            return currencyCode == that.currencyCode &&
                    Objects.equals(date, that.date);
        }

        @Override
        public int hashCode() {
            return Objects.hash(currencyCode, date);
        }
    }
}
