package ru.yandex.direct.core.entity.currency.service;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.agency.service.AgencyService;
import ru.yandex.direct.core.entity.currency.model.CountryCurrencies;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.geo.model.GeoRegion;
import ru.yandex.direct.core.entity.geo.repository.IGeoRegionRepository;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toCollection;
import static ru.yandex.direct.common.db.PpcPropertyNames.GBP_CLIENT_UID;
import static ru.yandex.direct.common.db.PpcPropertyNames.GBP_CLIENT_UID_ALLOWED_VALUE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Сервис для работы с валютами
 */
@Service
@Lazy
@ParametersAreNonnullByDefault
public class CurrencyService {
    private final CurrencyDictCache currencyDictCache;
    private final AgencyService agencyService;
    private final BalanceService balanceService;
    private final FeatureService featureService;
    private final PpcProperty<Set<Long>> gbpClientUidProperty;
    private final IGeoRegionRepository geoRegionRepository;

    @Autowired
    public CurrencyService(
            CurrencyDictCache currencyDictCache,
            AgencyService agencyService,
            BalanceService balanceService,
            FeatureService featureService,
            PpcPropertiesSupport ppcPropertiesSupport,
            IGeoRegionRepository geoRegionRepository) {
        this.currencyDictCache = Objects.requireNonNull(currencyDictCache, "currencyService");
        this.agencyService = Objects.requireNonNull(agencyService, "agencyService");
        this.balanceService = Objects.requireNonNull(balanceService, "balanceService");
        this.featureService = Objects.requireNonNull(featureService, "featureService");
        this.gbpClientUidProperty = Objects.requireNonNull(ppcPropertiesSupport, "ppcPropertiesSupport")
                .get(GBP_CLIENT_UID, Duration.ofMinutes(1));
        this.geoRegionRepository = geoRegionRepository;
    }

    /**
     * Вернуть множество допустимых валют для клиентов агенства
     * <p>
     * Кусок из реализации из Common.pm::get_client_allowed_country_currency (См. if ($agency_client_id))
     *
     * @param clientId       Идентификатор клиента
     * @param agencyClientId Идентификатор агентства
     */
    @Nonnull
    public Set<CurrencyCode> getAllowedCurrenciesForAgencyClient(
            @Nullable ClientId clientId, ClientId agencyClientId) {
        Objects.requireNonNull(agencyClientId, "agencyClientId");

        Set<CurrencyCode> agencyCurrencies = agencyService.getAllowedCurrencies(agencyClientId);

        return getAllowedCountryCurrenciesForAgency(clientId, agencyClientId)
                .values()
                .stream()
                .flatMap(Collection::stream)
                .filter(agencyCurrencies::contains)
                .collect(toCollection(() -> EnumSet.noneOf(CurrencyCode.class)));
    }

    @Nonnull
    private Map<Long, Set<CurrencyCode>> filterByAvailableGlobalRegions(Set<Long> availableRegions,
                                                                        Multimap<Long, CurrencyCode> multimap) {
        return multimap.entries()
                .stream()
                .filter(e -> availableRegions.contains(e.getKey()))
                .collect(
                        groupingBy(Map.Entry::getKey,
                                mapping(
                                        Map.Entry::getValue,
                                        toCollection(() -> EnumSet.noneOf(CurrencyCode.class)))));
    }

    @Nonnull
    private Map<Long, Set<CurrencyCode>> getAllowedCountryCurrencies(
            Multimap<Long, CurrencyCode> allCountryCurrencies,
            @Nullable ClientId clientId,
            @Nullable ClientId agencyClientId,
            BiPredicate<Long, CurrencyCode> skipCountryCurrency) {
        Multimap<Long, CurrencyCode> balanceCountryCurrencies = null;
        // У существующих клиентов (есть ClientID) даём выбирать только доступные по
        // мнению Баланса сочетания валюта-страна, т.к. у него уже могут быть плательщики
        // и оплаты в других странах/валютах
        if (clientId != null) {
            balanceCountryCurrencies = balanceService.getClientCountryCurrencies(clientId, agencyClientId);
        }
        return getAllowedCountryCurrencies(allCountryCurrencies, balanceCountryCurrencies, skipCountryCurrency);
    }

    @Nonnull
    private Map<Long, Set<CurrencyCode>> getAllowedCountryCurrencies(
            Multimap<Long, CurrencyCode> allCountryCurrencies,
            @Nullable Multimap<Long, CurrencyCode> balanceCountryCurrencies,
            BiPredicate<Long, CurrencyCode> skipCountryCurrency) {
        /* Ходим в таблицу за всеми возможными регионами, т.к. GeoTreeFactory#getGlobalGeoTree возвращает
        дерево, которое не содержит нужных регионов. */
        final var allRegionIds = new HashSet<>(allCountryCurrencies.keySet());
        ifNotNull(balanceCountryCurrencies, bcc -> allRegionIds.addAll(bcc.keySet()));

        final var availableRegions = geoRegionRepository.getGeoRegionsByIds(allRegionIds)
                .stream()
                .map(GeoRegion::getId)
                .collect(Collectors.toSet());

        Map<Long, Set<CurrencyCode>> result = filterByAvailableGlobalRegions(availableRegions, allCountryCurrencies);

        // Пересекаем с доступными странами и валютами из баланса
        Map<Long, Set<CurrencyCode>> balanceCountryCurrenciesMap = ifNotNull(
                balanceCountryCurrencies, bcc -> filterByAvailableGlobalRegions(availableRegions, bcc)
        );

        for (Iterator<Map.Entry<Long, Set<CurrencyCode>>> it = result.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<Long, Set<CurrencyCode>> entry = it.next();
            if (balanceCountryCurrenciesMap != null && !balanceCountryCurrenciesMap.containsKey(entry.getKey())) {
                it.remove();
            } else if (balanceCountryCurrenciesMap != null) {
                entry.getValue().retainAll(balanceCountryCurrenciesMap.get(entry.getKey()));
            }
            entry.getValue().removeIf(currencyCode -> skipCountryCurrency.test(entry.getKey(), currencyCode));
        }

        //Удаляем страны с пустым список валют
        result.entrySet().removeIf(entry -> entry.getValue().isEmpty());

        return result;
    }

    /**
     * Вернуть допустимые валюты по странам для клиента агентства/агентства (См. Common
     * .pm::get_client_allowed_country_currency)
     *
     * @param clientId       Идентификатор клиента
     * @param agencyClientId Идентификатор агенства
     * @return Возвращает отображение идентификатора региона в множество допустимых для него валют
     * для заданного клиента агентства/агентства
     */
    @Nonnull
    public Map<Long, Set<CurrencyCode>> getAllowedCountryCurrenciesForAgency(
            @Nullable ClientId clientId, ClientId agencyClientId) {
        Objects.requireNonNull(agencyClientId, "agencyClientId");

        return getAllowedCountryCurrencies(
                currencyDictCache.getCountryCurrencies()
                        .getAgenciesCurrencies(),
                clientId,
                agencyClientId,
                (country, currency) ->
                        isCountryCurrencyInterconnectionForbiddenForAgency(agencyClientId, country, currency));
    }

    /**
     * Вернуть допустимые валюты по странам для обычных клиентов (См. Common.pm::get_client_allowed_country_currency)
     *
     * @param clientId Идентификатор клиента. Если null, не валидируем балансовые данные
     * @return Возвращает отображение идентификатор региона в множество допустимых для него валют
     * для заданного клиента
     */
    @Nonnull
    public Map<Long, Set<CurrencyCode>> getAllowedCountryCurrenciesForClient(ClientId clientId, @Nullable Long uid) {
        return getAllowedCountryCurrencies(
                currencyDictCache.getCountryCurrencies()
                        .getClientCurrencies(),
                clientId,
                null,
                (country, currency) ->
                        isCountryCurrencyInterconnectionForbiddenForNonAgencyClient(uid, country, currency));
    }

    /**
     * Вернуть допустимые валюты по странам для клиента агентства/агентства (См. Common
     * .pm::get_client_allowed_country_currency)
     *
     * @return Возвращает отображение идентификатора региона в множество допустимых для него валют
     * для заданного клиента агентства/агентства
     */
    @Nonnull
    public Map<Long, Set<CurrencyCode>> getAllowedCountryCurrencies(
            @Nullable Multimap<Long, CurrencyCode> balanceCountryCurrencies,
            @Nullable Long uid, @Nullable ClientId agencyClientId) {
        CountryCurrencies countryCurrencies = currencyDictCache.getCountryCurrencies();
        BiPredicate<Long, CurrencyCode> skipCountryCurrency = (agencyClientId != null) ?
                (country, currency) ->
                        isCountryCurrencyInterconnectionForbiddenForAgency(agencyClientId, country, currency) :
                (country, currency) ->
                        isCountryCurrencyInterconnectionForbiddenForNonAgencyClient(uid, country, currency);
        return getAllowedCountryCurrencies(
                (agencyClientId != null) ? countryCurrencies.getAgenciesCurrencies() :
                        countryCurrencies.getClientCurrencies(),
                balanceCountryCurrencies,
                skipCountryCurrency);
    }

    private boolean isCountryCurrencyInterconnectionForbiddenForAgency(
            @Nonnull ClientId agencyClientId, Long countryRegionId, CurrencyCode currencyCode) {
        return CurrencyCode.NOT_FOR_CLIENT_CREATION.contains(currencyCode) ||
                (currencyCode == CurrencyCode.GBP && !isGbpAllowedForAgencySubclients(agencyClientId));
    }

    private boolean isCountryCurrencyInterconnectionForbiddenForNonAgencyClient(
            @Nullable Long uid, Long countryRegionId, CurrencyCode currencyCode) {
        return CurrencyCode.NOT_FOR_CLIENT_CREATION.contains(currencyCode) ||
                (currencyCode == CurrencyCode.GBP && (uid == null || !isGbpAllowedForClient(uid)));
    }

    public boolean isGbpAllowedForClient(long uid) {
        return Objects.equals(gbpClientUidProperty.get(), GBP_CLIENT_UID_ALLOWED_VALUE) ||
                gbpClientUidProperty.getOrDefault(Collections.emptySet()).contains(uid);
    }

    public boolean isGbpAllowedForAgencySubclients(ClientId agencyClientId) {
        return featureService.isEnabledForClientId(agencyClientId, FeatureName.GBP_ALLOWED_FOR_AGENCY_SUBCLIENTS);
    }
}
