package ru.yandex.direct.core.entity.deal.repository;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.dbschema.ppc.Tables.DEAL_NOTIFICATIONS;

/**
 * Репозиторий для сохранения в базу записей о том, что Директ отправлял уведомления
 * о заключённых частных сделках. Записи пишутся в лог-таблицу deal_notifications на шардах ppc.
 * <p>
 * В записях в логе хранится информация (serializedMdsKey и mdsUrl) о том, где в MDS лежит
 * подписанный PDF-файл, который отправился клиенту.
 */
@ParametersAreNonnullByDefault
@Repository
public class DealNotificationRepository {
    private DslContextProvider dslContextProvider;
    private ShardHelper shardHelper;

    public DealNotificationRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    /**
     * Записать информацию об отправленном уведомлении
     *
     * @param notificationId   ID уведомления, который увидит клиент
     * @param serializedMdsKey ключ в MDS в формате, который выдаёт и распознаёт ru.yandex.inside.mds.MdsFileKey
     * @param mdsUrl           URL, по которому можно получить файл из MDS
     * @return ID новой записи
     */
    public Long logNotification(int shard, ClientId clientId, String notificationId, Long dealId,
                                String serializedMdsKey, String mdsUrl) {
        Long logId = shardHelper.generateDealNotificationIds(1).get(0);

        checkArgument(serializedMdsKey.length() <= DEAL_NOTIFICATIONS.PDF_MDS_KEY.getDataType().length());

        dslContextProvider.ppc(shard).insertInto(DEAL_NOTIFICATIONS)
                .set(DEAL_NOTIFICATIONS.DEAL_NOTIFICATION_ID, logId)
                .set(DEAL_NOTIFICATIONS.CLIENT_NOTIFICATION_ID, notificationId)
                .set(DEAL_NOTIFICATIONS.DEAL_ID, dealId)
                .set(DEAL_NOTIFICATIONS.CLIENT_ID, clientId.asLong())
                .set(DEAL_NOTIFICATIONS.CREATION_TIME, DSL.currentLocalDateTime())
                .set(DEAL_NOTIFICATIONS.PDF_MDS_KEY, serializedMdsKey)
                .set(DEAL_NOTIFICATIONS.PDF_MDS_URL, mdsUrl)
                .execute();

        return logId;
    }

}
