package ru.yandex.direct.core.entity.deal.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Record2;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.deal.container.CampaignDeal;
import ru.yandex.direct.core.entity.deal.model.CompleteReason;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealAdfox;
import ru.yandex.direct.core.entity.deal.model.DealDirect;
import ru.yandex.direct.core.entity.deal.model.DealType;
import ru.yandex.direct.core.entity.deal.model.StatusAdfox;
import ru.yandex.direct.core.entity.deal.model.StatusAdfoxSync;
import ru.yandex.direct.core.entity.deal.model.StatusDirect;
import ru.yandex.direct.dbschema.ppc.enums.DealsStatusAdfoxSync;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsDealsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.DealsRecord;
import ru.yandex.direct.dbschema.ppcdict.tables.records.DealsAdfoxRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.percentProperty;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsDeals.CAMPAIGNS_DEALS;
import static ru.yandex.direct.dbschema.ppc.tables.Deals.DEALS;
import static ru.yandex.direct.dbschema.ppcdict.tables.DealsAdfox.DEALS_ADFOX;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class DealRepository {
    private static final Logger logger = LoggerFactory.getLogger(DealRepository.class);

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<Deal> ppcDealJooqMapper;
    private final JooqMapperWithSupplier<Deal> ppcDictDealJooqMapper;
    private final JooqMapperWithSupplier<CampaignDeal> campaignDealJooqMapper;

    @Autowired
    public DealRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        ppcDealJooqMapper = createPpcDealJooqMapper();
        ppcDictDealJooqMapper = createPpcDictDealJooqMapper();
        campaignDealJooqMapper = createCampaignsDealsJooqMapper();
    }

    /**
     * Добавление сделок в ppcdict
     */
    public List<Long> addDealsPpcDict(List<Deal> deals) {
        if (deals.isEmpty()) {
            return emptyList();
        }
        InsertHelper<DealsAdfoxRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), DEALS_ADFOX);
        insertHelper.addAll(ppcDictDealJooqMapper, deals);
        int inserted = insertHelper.execute();
        if (inserted != deals.size()) {
            // не падаем, а логгируем, поскольку мы уже закоммитили изменение в БД
            logger.warn("Not all deals inserted into ppcdict. Deals to add: {}", deals);
        }

        return mapList(deals, Deal::getId);
    }

    /**
     * Добавление сделок в ppc
     */
    public List<Long> addDealsPpc(int shard, List<Deal> deals) {
        if (deals.isEmpty()) {
            return emptyList();
        }
        InsertHelper<DealsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), DEALS);
        insertHelper.addAll(ppcDealJooqMapper, deals);
        int inserted = insertHelper.execute();
        if (inserted != deals.size()) {
            // не падаем, а логгируем, поскольку мы уже закоммитили изменение в БД
            logger.warn("Not all deals inserted into ppcdict. Deals to add: {}", deals);
        }

        return mapList(deals, Deal::getId);
    }

    /**
     * Получение списка сделок по клиентам из одного шарда. Читает данные из deals и adfox_deals
     *
     * @param shard     шард, в котором находятся clientIds
     * @param clientIds список клиентов, к которым могут быть привязаны искомые сделки в deals
     */
    public List<Deal> getDealsSeveralClientsFromSingleShard(int shard, Collection<Long> clientIds,
                                                            @Nullable Collection<Long> dealIds) {
        if (dealIds != null && dealIds.isEmpty()) {
            return emptyList();
        }
        DSLContext dslContext = dslContextProvider.ppc(shard);
        DSLContext ppcDictDslContext = dslContextProvider.ppcdict();

        List<Deal> ppcDeals = dslContext.select(ppcDealJooqMapper.getFieldsToRead())
                .from(DEALS)
                .where(DEALS.CLIENT_ID.in(clientIds))
                .and(dealIds != null ? DEALS.DEAL_ID.in(dealIds) : DSL.trueCondition())
                .fetch()
                .map(ppcDealJooqMapper::fromDb);

        if (ppcDeals.isEmpty()) {
            return emptyList();
        }

        Map<Long, Deal> ppcDictDeals = StreamEx.of(ppcDictDslContext.select(ppcDictDealJooqMapper.getFieldsToRead())
                .from(DEALS_ADFOX)
                .where(DEALS_ADFOX.DEAL_ID.in(mapList(ppcDeals, ModelWithId::getId)))
                .fetch()
                .map(ppcDictDealJooqMapper::fromDb))
                .toMap(Deal::getId, Function.identity());

        for (Deal deal : ppcDeals) {
            Deal ppcDictDeal = ppcDictDeals.get(deal.getId());
            if (ppcDictDeal == null) {
                throw new IllegalStateException(
                        "Deal with deal_id=" + deal.getId() + " found in ppc.deals, but missed in ppcdict.deals_adfox");
            }
            fillPpcDictDealParameters(deal, ppcDictDeal);
        }
        return ppcDeals;
    }

    /**
     * Получение списка сделок. Читает данные из deals и adfox_deals
     *
     * @param shard    шард, в котором находится clientId
     * @param clientId клиент, к которому привязана сделка в deals
     */
    public List<Deal> getDeals(int shard, ClientId clientId, @Nullable Collection<Long> dealIds) {
        return getDealsSeveralClientsFromSingleShard(shard, singletonList(clientId.asLong()), dealIds);
    }

    /**
     * Получение информации о сделке, пришедшей из Adfox.
     * Читает данные из {@code ppcdict.adfox_deals}.
     * Без дополнительных полей, заполняемых Директом.
     */
    public List<DealAdfox> getDealsAdfox(Collection<Long> dealIds) {
        if (dealIds.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppcdict()
                .select(ppcDictDealJooqMapper.getFieldsToRead())
                .from(DEALS_ADFOX)
                .where(DEALS_ADFOX.DEAL_ID.in(dealIds))
                .fetch(ppcDictDealJooqMapper::fromDb);
    }

    /**
     * Получение ClientID владельцев сделок по списку сделок
     */
    public Map<Long, Long> getClientIdsByDealIdsFromDealsAdfox(Collection<Long> dealIds) {
        if (dealIds.isEmpty()) {
            return emptyMap();
        }

        List<DealAdfox> deals = dslContextProvider.ppcdict()
                .select(DEALS_ADFOX.DEAL_ID, DEALS_ADFOX.CLIENT_ID)
                .from(DEALS_ADFOX)
                .where(DEALS_ADFOX.DEAL_ID.in(dealIds))
                .fetch(ppcDictDealJooqMapper::fromDb);
        return StreamEx.of(deals)
                .toMap(DealAdfox::getId, DealAdfox::getClientId);
    }

    /**
     * Получение статусов для сделок
     *
     * @param shard шард, в котором лежат сделки
     */
    public Map<Long, StatusDirect> getDealsStatuses(int shard, List<Long> dealIds) {
        if (dealIds.isEmpty()) {
            return emptyMap();
        }
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return getDealsStatuses(dslContext, dealIds);
    }

    public Map<Long, StatusDirect> getDealsStatuses(DSLContext dslContext, List<Long> dealIds) {
        List<Deal> deals =
                dslContext.select(DEALS.DEAL_ID, DEALS.DIRECT_DEAL_STATUS)
                        .from(DEALS)
                        .where(DEALS.DEAL_ID.in(dealIds))
                        .fetch()
                        .map(ppcDealJooqMapper::fromDb);

        return StreamEx.of(deals)
                .toMap(Deal::getId, Deal::getDirectStatus);
    }

    /**
     * Обновляет поля в deals. deals_adfox — только для чтения.
     */
    public int updateDeals(int shard, Collection<AppliedChanges<Deal>> appliedChanges) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        return updateDeals(dslContext, appliedChanges);
    }

    public int updateDeals(DSLContext dslContext, Collection<AppliedChanges<Deal>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return 0;
        }

        JooqUpdateBuilder<DealsRecord, Deal> ub = new JooqUpdateBuilder<>(DEALS.DEAL_ID, appliedChanges);
        ub.processProperty(Deal.DIRECT_STATUS, DEALS.DIRECT_DEAL_STATUS, StatusDirect::toSource);
        ub.processProperty(Deal.COMPLETE_REASON, DEALS.COMPLETE_REASON, CompleteReason::toSource);
        ub.processProperty(Deal.NAME, DEALS.DIRECT_NAME);
        ub.processProperty(Deal.DESCRIPTION, DEALS.DIRECT_DESCRIPTION);
        ub.processProperty(Deal.STATUS_ADFOX_SYNC, DEALS.STATUS_ADFOX_SYNC, StatusAdfoxSync::toSource);

        return dslContext
                .update(DEALS)
                .set(ub.getValues())
                .where(DEALS.DEAL_ID.in(ub.getChangedIds()))
                .execute();
    }

    /**
     * Обновляет ту часть сделки, которая управляется Adfox'ом.
     *
     * <b>Важно:</b> этот метод должен вызываться только из Транспорта из Adfox.
     * Директ самостоятельно эти данные не изменяет.
     */
    public int updateDealsAdfox(Collection<AppliedChanges<DealAdfox>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return 0;
        }
        JooqUpdateBuilder<DealsAdfoxRecord, DealAdfox> ub =
                new JooqUpdateBuilder<>(DEALS_ADFOX.DEAL_ID, appliedChanges);
        // пока что обновляем только статус
        ub.processProperty(DealAdfox.ADFOX_STATUS, DEALS_ADFOX.ADFOX_DEAL_STATUS, StatusAdfox::toSource);

        return dslContextProvider.ppcdict()
                .update(DEALS_ADFOX)
                .set(ub.getValues())
                .where(DEALS_ADFOX.DEAL_ID.in(ub.getChangedIds()))
                .execute();
    }

    public List<ModelWithId> getClientDealIds(int shard, Long clientId) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        List<Deal> ppcDeals =
                dslContext.select(DEALS.DEAL_ID)
                        .from(DEALS)
                        .where(DEALS.CLIENT_ID.eq(clientId))
                        .fetch()
                        .map(ppcDealJooqMapper::fromDb);
        return mapList(ppcDeals, ModelWithId.class::cast);
    }

    /**
     * Получение списка сделок с полями {@link Deal}
     *
     * @param shard    шард, в котором находится clientId
     * @param clientId идентификатор клиента, по которому происходит поиск
     */
    public List<Deal> getDealsBriefByClientId(int shard, ClientId clientId) {
        return getDeals(shard, clientId, null);
    }

    /**
     * @return Сделки в соответствующем статусе
     */
    public List<DealDirect> getDirectDealsByAdfoxSyncStatus(int shard, Collection<StatusAdfoxSync> statuses,
                                                            LimitOffset limitOffset) {
        List<DealsStatusAdfoxSync> dbStatuses = mapList(statuses, StatusAdfoxSync::toSource);
        return dslContextProvider.ppc(shard)
                .select(ppcDealJooqMapper.getFieldsToRead())
                .from(DEALS)
                .where(DEALS.STATUS_ADFOX_SYNC.in(dbStatuses))
                .orderBy(DEALS.DEAL_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(ppcDealJooqMapper::fromDb);
    }

    /**
     * Обновляет значение колонки {@code DEALS.STATUS_ADFOX_SYNC} с {@code from} на {@code to}
     *
     * @return количество изменённых строк
     */
    public int updateAdfoxSyncStatus(int shard, Collection<Long> dealIds, StatusAdfoxSync from, StatusAdfoxSync to) {
        return dslContextProvider.ppc(shard)
                .update(DEALS)
                .set(DEALS.STATUS_ADFOX_SYNC, StatusAdfoxSync.toSource(to))
                .where(DEALS.DEAL_ID.in(dealIds))
                .and(DEALS.STATUS_ADFOX_SYNC.eq(StatusAdfoxSync.toSource(from)))
                .execute();
    }

    /**
     * Получение привязанных кампаний для списка сделок
     */
    public Map<Long, List<Long>> getLinkedCampaigns(int shard, List<Long> dealIds) {
        return getCampaignsDealsByDealIds(shard, dealIds, false);
    }

    /**
     * Получение всех записей из campaigns_deals по списку сделок
     */
    public Map<Long, List<Long>> getAllCampaignsDealsByDealIds(int shard, List<Long> dealIds) {
        return getCampaignsDealsByDealIds(shard, dealIds, true);
    }

    /**
     * Получение записей из campaigns_deals по списку сделок
     */
    private Map<Long, List<Long>> getCampaignsDealsByDealIds(int shard, List<Long> dealIds, boolean includeDeleted) {
        if (dealIds.isEmpty()) {
            return emptyMap();
        }
        DSLContext dslContext = dslContextProvider.ppc(shard);
        SelectConditionStep<Record2<Long, Long>> step = dslContext.select(CAMPAIGNS_DEALS.DEAL_ID, CAMPAIGNS_DEALS.CID)
                .from(CAMPAIGNS_DEALS)
                .where(CAMPAIGNS_DEALS.DEAL_ID.in(dealIds));
        if (!includeDeleted) {
            step = step.and(CAMPAIGNS_DEALS.IS_DELETED.eq(0L));
        }
        List<Record2<Long, Long>> dealCampaigns = step.fetch();
        return StreamEx.of(dealCampaigns)
                .mapToEntry(t -> t.field1().getValue(t), t -> t.field2().get(t))
                .grouping();
    }


    /**
     * Получение привязанных сделок для списка кампаний
     */
    public Map<Long, List<Long>> getLinkedDeals(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        DSLContext dslContext = dslContextProvider.ppc(shard);
        List<Record2<Long, Long>> dealCampaigns = dslContext.select(CAMPAIGNS_DEALS.CID, CAMPAIGNS_DEALS.DEAL_ID)
                .from(CAMPAIGNS_DEALS)
                .where(CAMPAIGNS_DEALS.CID.in(campaignIds))
                .and(CAMPAIGNS_DEALS.IS_DELETED.eq(0L))
                .fetch();
        return StreamEx.of(dealCampaigns)
                .mapToEntry(t -> t.field1().getValue(t), t -> t.field2().get(t))
                .grouping();
    }

    /**
     * Добавляет связку в campaigns_deal
     *
     * @param shard шард кампаний
     */
    public int linkCampaigns(int shard, List<CampaignDeal> dealsToCampaigns) {
        return linkCampaigns(dslContextProvider.ppc(shard).configuration(), dealsToCampaigns);
    }

    /**
     * Добавляет связку в campaigns_deal
     */
    public int linkCampaigns(Configuration conf, List<CampaignDeal> dealsToCampaigns) {
        if (dealsToCampaigns.isEmpty()) {
            return 0;
        }
        DSLContext dslContext = conf.dsl();
        InsertHelper<CampaignsDealsRecord> insertHelper = new InsertHelper<>(dslContext, CAMPAIGNS_DEALS);
        insertHelper.addAll(campaignDealJooqMapper, dealsToCampaigns);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate().set(CAMPAIGNS_DEALS.IS_DELETED, 0L);
        }
        return insertHelper.executeIfRecordsAdded();
    }

    /**
     * Удаляет связку в campaigns_deal
     *
     * @param shard шард кампаний
     */
    public int unlinkCampaigns(int shard, List<CampaignDeal> dealsToCampaigns) {
        return unlinkCampaigns(dslContextProvider.ppc(shard).configuration(), dealsToCampaigns);
    }

    /**
     * Удаляет связку в campaigns_deal
     */
    public int unlinkCampaigns(Configuration conf, List<CampaignDeal> dealsToCampaigns) {
        if (dealsToCampaigns.isEmpty()) {
            return 0;
        }
        Condition updateCondition = StreamEx.of(dealsToCampaigns)
                .map(t -> CAMPAIGNS_DEALS.DEAL_ID.eq(t.getDealId())
                        .and(CAMPAIGNS_DEALS.CID.eq(t.getCampaignId())))
                .reduce(Condition::or)
                .orElse(DSL.condition(false));
        DSLContext dslContext = conf.dsl();
        return dslContext.update(CAMPAIGNS_DEALS)
                .set(CAMPAIGNS_DEALS.IS_DELETED, 1L)
                .where(updateCondition)
                .execute();
    }

    /**
     * Дозаполняет поля сделки, которые лежат в adfox_deals
     */
    private void fillPpcDictDealParameters(Deal ppcDeal, Deal ppcDictDeal) {
        ppcDeal
                .withDealType(ppcDictDeal.getDealType())
                .withCpm(ppcDictDeal.getCpm())
                .withMarginRatio(ppcDictDeal.getMarginRatio())
                .withPublisherName(ppcDictDeal.getPublisherName())
                .withCurrencyCode(ppcDictDeal.getCurrencyCode())
                .withAdfoxName(ppcDictDeal.getAdfoxName())
                .withAdfoxDescription(ppcDictDeal.getAdfoxDescription())
                .withAdfoxStatus(ppcDictDeal.getAdfoxStatus())
                .withContacts(ppcDictDeal.getContacts())
                .withDateStart(ppcDictDeal.getDateStart())
                .withDateEnd(ppcDictDeal.getDateEnd())
                .withDateCreated(ppcDictDeal.getDateCreated())
                .withExpectedImpressionsPerWeek(ppcDictDeal.getExpectedImpressionsPerWeek())
                .withExpectedMoneyPerWeek(ppcDictDeal.getExpectedMoneyPerWeek())
                .withAgencyFeePercent(ppcDictDeal.getAgencyFeePercent())
                .withAgencyFeeType(ppcDictDeal.getAgencyFeeType())
                .withTargetingsText(ppcDictDeal.getTargetingsText())
                .withDealJson(ppcDictDeal.getDealJson())
                .withPlacements(ppcDictDeal.getPlacements());

    }

    private JooqMapperWithSupplier<Deal> createPpcDealJooqMapper() {
        return JooqMapperWithSupplierBuilder.builder(Deal::new)
                .map(property(Deal.ID, DEALS.DEAL_ID))
                .map(property(Deal.NAME, DEALS.DIRECT_NAME))
                .map(property(Deal.DESCRIPTION, DEALS.DIRECT_DESCRIPTION))
                .map(property(Deal.CLIENT_ID, DEALS.CLIENT_ID))
                .map(convertibleProperty(Deal.DIRECT_STATUS, DEALS.DIRECT_DEAL_STATUS,
                        StatusDirect::fromSource, StatusDirect::toSource))
                .map(convertibleProperty(Deal.COMPLETE_REASON, DEALS.COMPLETE_REASON,
                        CompleteReason::fromSource, CompleteReason::toSource))
                .map(convertibleProperty(Deal.STATUS_ADFOX_SYNC, DEALS.STATUS_ADFOX_SYNC,
                        StatusAdfoxSync::fromSource, StatusAdfoxSync::toSource))
                .build();
    }

    private JooqMapperWithSupplier<Deal> createPpcDictDealJooqMapper() {
        return JooqMapperWithSupplierBuilder.builder(Deal::new)
                .map(property(Deal.ID, DEALS_ADFOX.DEAL_ID))
                .map(property(Deal.CPM, DEALS_ADFOX.CPM))
                .map(percentProperty(Deal.MARGIN_RATIO, DEALS_ADFOX.MARGIN_RATIO))
                .map(property(Deal.CLIENT_ID, DEALS_ADFOX.CLIENT_ID))
                .map(property(Deal.PUBLISHER_NAME, DEALS_ADFOX.PUBLISHER_NAME))
                .map(property(Deal.ADFOX_NAME, DEALS_ADFOX.ADFOX_NAME))
                .map(property(Deal.ADFOX_DESCRIPTION, DEALS_ADFOX.ADFOX_DESCRIPTION))
                .map(property(Deal.CONTACTS, DEALS_ADFOX.CONTACTS))
                .map(property(Deal.TARGETINGS_TEXT, DEALS_ADFOX.TARGETINGS))
                .map(property(Deal.EXPECTED_IMPRESSIONS_PER_WEEK, DEALS_ADFOX.EXPECTED_IMPRESSIONS_PER_WEEK))
                .map(property(Deal.EXPECTED_MONEY_PER_WEEK, DEALS_ADFOX.EXPECTED_MONEY_PER_WEEK))
                .map(property(Deal.DEAL_JSON, DEALS_ADFOX.MISC_ATTRS_JSON))
                .map(percentProperty(Deal.AGENCY_FEE_PERCENT, DEALS_ADFOX.AGENCY_REVENUE_RATIO))
                .map(property(Deal.AGENCY_FEE_TYPE, DEALS_ADFOX.AGENCY_REVENUE_TYPE))
                .map(property(Deal.DATE_CREATED, DEALS_ADFOX.CREATE_TIME_UTC))
                .map(property(Deal.DATE_START, DEALS_ADFOX.DATE_START_UTC))
                .map(property(Deal.DATE_END, DEALS_ADFOX.DATE_END_UTC))
                .map(convertibleProperty(Deal.CURRENCY_CODE, DEALS_ADFOX.CURRENCY_ISO_CODE,
                        RepositoryUtils::currencyFromDb, RepositoryUtils::currencyToDb))
                .map(convertibleProperty(Deal.ADFOX_STATUS, DEALS_ADFOX.ADFOX_DEAL_STATUS,
                        StatusAdfox::fromSource, StatusAdfox::toSource))
                .map(convertibleProperty(Deal.DEAL_TYPE, DEALS_ADFOX.DEAL_TYPE,
                        DealType::fromSource, DealType::toSource))
                .map(convertibleProperty(Deal.PLACEMENTS, DEALS_ADFOX.PLACEMENTS,
                        DealMappings::placementsFromJson, DealMappings::placementsToJson))
                .build();
    }

    private JooqMapperWithSupplier<CampaignDeal> createCampaignsDealsJooqMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampaignDeal::new)
                .map(property(CampaignDeal.DEAL_ID, CAMPAIGNS_DEALS.DEAL_ID))
                .map(property(CampaignDeal.CAMPAIGN_ID, CAMPAIGNS_DEALS.CID))
                .build();
    }
}
