package ru.yandex.direct.core.entity.deal.service;

import java.time.Duration;
import java.time.LocalDate;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.balance.client.model.response.PartnerContractCollateralInfo;
import ru.yandex.direct.balance.client.model.response.PartnerContractContractInfo;
import ru.yandex.direct.balance.client.model.response.PartnerContractInfo;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.deal.model.BalanceCollateralProblem;
import ru.yandex.direct.core.entity.deal.model.BalanceContractProblem;
import ru.yandex.direct.core.entity.deal.model.BalancePrivateDealInfo;
import ru.yandex.direct.core.entity.deal.model.BalancePrivateDealSearchResult;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;

/**
 * Сервис с логикой про то, как запросить из Баланса и правильно отфильтровать в его ответе информацию
 * о договоре и дополнительном соглашении, нужную для частной сделки.
 */
@ParametersAreNonnullByDefault
@Service
public class BalancePrivateDealInfoService {
    private static final Logger LOGGER = LoggerFactory.getLogger(BalancePrivateDealInfoService.class);

    private final BalanceClient balanceClient;
    private final long directServiceId;
    private final long directPrivateDealCollateralTypeId;
    private final long rubCurrencyCode;
    private final Duration requestTimeout;

    @Autowired
    public BalancePrivateDealInfoService(BalanceClient balanceClient,
                                         @Value("${balance.directServiceId}") long directServiceId,
                                         @Value("${balance.directPrivateDealCollateralTypeId}") long directPrivateDealCollateralTypeId,
                                         @Value("${balance.rubCurrencyCode}") long rubCurrencyCode,
                                         DirectConfig directConfig) {
        this.balanceClient = balanceClient;
        this.directServiceId = directServiceId;
        this.directPrivateDealCollateralTypeId = directPrivateDealCollateralTypeId;
        this.rubCurrencyCode = rubCurrencyCode;
        this.requestTimeout = directConfig.getDuration("balance.get_partner_contracts_request_timeout");
    }

    /**
     * Найти "правильные" договор и допсоглашение в Балансе и вернуть объект с информацией, нужной
     * для уведомления о заключённой сделке.
     *
     * @see BalancePrivateDealInfo
     */
    @Nonnull
    @SuppressWarnings("squid:S2629")
    public BalancePrivateDealInfo getBalancePrivateDealInfo(Long clientId) {
        BalancePrivateDealSearchResult searchResult = searchForContractAndCollateral(clientId);
        if (!searchResult.foundContractAndCollateral() || searchResult.getBalancePrivateDealInfo() == null) {
            LOGGER.error("found no suitable contract/collateral pair, searchResult = {}",
                    JsonUtils.toJson(searchResult));
            throw new IllegalStateException("balance has not returned a valid contract and collateral");
        }

        return searchResult.getBalancePrivateDealInfo();
    }

    /**
     * Найти "правильные" договор и допсоглашение в Балансе и вернуть объект, в котором записано,
     * что вернул Баланс (все договоры и допсоглашения) и по шагам информация, как мы их искали.
     *
     * @see BalancePrivateDealSearchResult
     */
    public BalancePrivateDealSearchResult searchForContractAndCollateral(Long clientId) {
        BalancePrivateDealSearchResult result = new BalancePrivateDealSearchResult();

        List<PartnerContractInfo> contractInfos = balanceClient.getPartnerContracts(clientId, null, requestTimeout);
        result.setBalanceResponse(contractInfos);

        List<Set<BalanceContractProblem>> contractProblems = contractInfos.stream()
                .map(this::findContractProblems)
                .collect(toList());
        result.setContractProblems(contractProblems);

        List<PartnerContractInfo> suitableContractInfos = IntStream.range(0, contractInfos.size()).boxed()
                .filter(idx -> contractProblems.get(idx).isEmpty())
                .map(contractInfos::get)
                .collect(toList());

        int suitableContractCount = suitableContractInfos.size();
        result.setSuitableContractCount(suitableContractCount);

        if (suitableContractCount != 1) {
            return result;
        }

        PartnerContractInfo relevantContractInfo = suitableContractInfos.get(0);
        result.setRelevantContractInfo(relevantContractInfo);

        if (relevantContractInfo.getContract().getCurrency() == null ||
                !relevantContractInfo.getContract().getCurrency().equals(rubCurrencyCode)) {
            result.setRelevantContractHasCorrectCurrency(false);
            return result;
        }

        result.setRelevantContractHasCorrectCurrency(true);

        List<PartnerContractCollateralInfo> collaterals = relevantContractInfo.getCollaterals();
        List<Set<BalanceCollateralProblem>> collateralProblems = collaterals.stream()
                .map(this::findCollateralProblems)
                .collect(toList());
        result.setCollateralProblems(collateralProblems);

        List<PartnerContractCollateralInfo> suitableCollaterals = IntStream.range(0, collaterals.size()).boxed()
                .filter(idx -> collateralProblems.get(idx).isEmpty())
                .map(collaterals::get)
                .collect(toList());

        int suitableCollateralCount = suitableCollaterals.size();
        result.setSuitableCollateralCount(suitableCollateralCount);

        if (suitableCollateralCount != 1) {
            return result;
        }

        PartnerContractCollateralInfo relevantCollateral = suitableCollaterals.get(0);

        result.setRelevantCollateral(relevantCollateral);
        result.setFoundContractAndCollateral(true);

        BalancePrivateDealInfo balancePrivateDealInfo = new BalancePrivateDealInfo(
                relevantContractInfo.getContract().getExternalContractId(),
                relevantContractInfo.getContract().getStartDate(),
                relevantCollateral.getExternalCollateralId(),
                relevantCollateral.getPersonalDealBasePercent());
        result.setBalancePrivateDealInfo(balancePrivateDealInfo);

        return result;
    }

    Set<BalanceContractProblem> findContractProblems(PartnerContractInfo contractInfo) {
        PartnerContractContractInfo contract = contractInfo.getContract();

        Set<BalanceContractProblem> result = EnumSet.noneOf(BalanceContractProblem.class);

        if (!contract.getServices().contains(directServiceId)) {
            result.add(BalanceContractProblem.NOT_RELATED_TO_DIRECT);
        }

        if (contract.getStartDate().isAfter(LocalDate.now())) {
            result.add(BalanceContractProblem.NOT_STARTED_YET);
        }

        LocalDate finishDate = contractInfo.getCollaterals().stream()
                .filter(collateral -> !collateral.getStartDate().isAfter(LocalDate.now()) &&
                        isSigned(collateral) &&
                        collateral.getDateOfCancellation() == null)
                .sorted(comparing(PartnerContractCollateralInfo::getStartDate).reversed())
                .map(PartnerContractCollateralInfo::getFinishDate)
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(contract.getFinishDate());

        if (finishDate != null && !finishDate.isAfter(LocalDate.now())) {
            result.add(BalanceContractProblem.EXPIRED);
        }

        if (contract.getDateOfSigning() == null && contract.getDateOfSigningByFax() == null) {
            result.add(BalanceContractProblem.NOT_SIGNED);
        }

        if (contract.getDateOfCancellation() != null) {
            result.add(BalanceContractProblem.CANCELLED);
        }

        if (contract.getDateOfSuspension() != null) {
            result.add(BalanceContractProblem.SUSPENDED);
        }

        return result;
    }

    Set<BalanceCollateralProblem> findCollateralProblems(PartnerContractCollateralInfo balanceCollateral) {
        Set<BalanceCollateralProblem> result = EnumSet.noneOf(BalanceCollateralProblem.class);

        if (balanceCollateral.getCollateralTypeId() != directPrivateDealCollateralTypeId) {
            result.add(BalanceCollateralProblem.WRONG_TYPE);
        }

        if (balanceCollateral.getStartDate().isAfter(LocalDate.now())) {
            result.add(BalanceCollateralProblem.NOT_STARTED_YET);
        }

        if (!isSigned(balanceCollateral)) {
            result.add(BalanceCollateralProblem.NOT_SIGNED);
        }

        if (balanceCollateral.getDateOfCancellation() != null) {
            result.add(BalanceCollateralProblem.CANCELLED);
        }

        return result;
    }

    private static boolean isSigned(PartnerContractCollateralInfo collateral) {
        return collateral.getDateOfSigning() != null || collateral.getDateOfSigningByFax() != null;
    }
}
