package ru.yandex.direct.core.entity.deal.service;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.logmail.service.LogMailService;
import ru.yandex.direct.mail.EmailAddress;
import ru.yandex.direct.mail.MailMessage;
import ru.yandex.direct.mail.MailSender;

/**
 * Сервис отправки email-уведомлений о заключённой частной сделке, который занимается
 * собственно формированием заголовка/текста и отправкой письма
 */
@ParametersAreNonnullByDefault
@Service
public class DealNotificationMailSenderService {
    public static final String EMAIL_FROM = "deal-notifications@direct.yandex.ru";

    private static final String EMAIL_CC = "com-vip@yandex-team.ru";
    private static final DateTimeFormatter CONTRACT_DATE_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");
    private static final EmailAddress EMAIL_ADDRESS_FROM = new EmailAddress(EMAIL_FROM, "Yandex.Direct");
    private static final EmailAddress EMAIL_ADDRESS_CC = new EmailAddress(EMAIL_CC, null);

    private MailSender mailSender;
    private LogMailService logMailService;

    @Autowired
    public DealNotificationMailSenderService(MailSender mailSender,
                                             LogMailService logMailService) {
        this.mailSender = mailSender;
        this.logMailService = logMailService;
    }

    /**
     * Отправляет оповещение о сделке в виде электронного письма.
     *
     * @param parameters параметры электронного письма
     * @param mdsUrl     Ссылка на pdf в mds
     */
    public void sendDealNotificationEmail(DealNotificationEmailParameters parameters, String mdsUrl) {
        MailMessage emailContent = createEmail(parameters);

        // Сохраняем адрес получателя письма, чтобы использовать при логгировании письма. Делается для того,
        // чтобы на девтесте в логи вставлялся правильный адрес, а не direct-dev-letters@.
        String emailTo = emailContent.getTo().getEmail();

        mailSender.send(emailContent);
        String content = emailContent.getMessageBody() + "\n" + mdsUrl;
        logMailService.addDealNotificationLogMail(emailTo, emailContent.getSubject(), content);
    }

    private MailMessage createEmail(DealNotificationEmailParameters parameters) {
        String subject = "Яндекс.Директ: Уведомление по Сделке № " + parameters.getNotificationId();

        String messageBody = "Добрый день!\n"
                + "В соответствии с п.3.3 ДС № " + parameters.getContractAdditionalAgreementId() + " к Договору № "
                + parameters.getContractId() + " от " + formatContractDate(parameters.getContractDate())
                + " направляем Вам Условия Сделки № " + parameters.getDeal().getId() + " (приложенный файл). "
                + "Напоминаем Вам, что настоящее Уведомление имеет силу дополнительного "
                + "соглашения к Договору и определяет его условия в отношении соответствующего "
                + "размещения Материалов и начисления Специальной премии - 2, предусмотренной "
                + "разделом 6 ДС № " + parameters.getContractAdditionalAgreementId() + ". "
                + "Такое дополнительное соглашение не требует подписания с Вашей стороны "
                + "и считается действительным вне зависимости от наличия у Вас технических "
                + "средств проверки электронной подписи и средств криптографической защиты. "
                + "Датой заключения и вступления в силу такого дополнительного соглашения "
                + "считается дата получения Вами настоящего Уведомления. Датой получения "
                + "Уведомления считается дата его направления Яндексом.\n"
                + "\n"
                + "Файл во вложении к письму.\n"
                + "\n"
                + "С уважением,\n"
                + "команда Яндекс.Директа\n"
                + "\n"
                + "Это автоматическое письмо. Пожалуйста, не отвечайте на него.\n"
                + "Если у Вас появились вопросы, Вы можете задать их Вашему менеджеру в Яндексе.\n"
                + "Помощь Директа: https://yandex.ru/support/direct/";

        if (Strings.nullToEmpty(parameters.getEmailTo()).isEmpty()) {
            throw new IllegalStateException("Email address for notification is empty");
        }
        MailMessage result = new MailMessage(
                EMAIL_ADDRESS_FROM,
                new EmailAddress(parameters.getEmailTo(), null),
                subject, messageBody);

        result.addAttachment("application/pdf", parameters.getPdfFileName(), parameters.getSignedPdfContent());
        result.withCc(EMAIL_ADDRESS_CC);

        return result;
    }

    private static String formatContractDate(Calendar contractDate) {
        LocalDate localDate = contractDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        return CONTRACT_DATE_FORMATTER.format(localDate);
    }

}
