package ru.yandex.direct.core.entity.deal.service;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.dssclient.DssClient;
import ru.yandex.direct.dssclient.http.certificates.Certificate;
import ru.yandex.direct.dssclient.pdfstamp.Box;
import ru.yandex.direct.dssclient.pdfstamp.Colors;
import ru.yandex.direct.dssclient.pdfstamp.Font;
import ru.yandex.direct.dssclient.pdfstamp.Icon;
import ru.yandex.direct.dssclient.pdfstamp.Image;
import ru.yandex.direct.dssclient.pdfstamp.Stamp;
import ru.yandex.direct.dssclient.pdfstamp.Text;
import ru.yandex.direct.utils.io.RuntimeIoException;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.DSS_CERTIFICATE_SERIAL_NUMBER;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Сервис подписывания PDF-файлов с помощью яндексового DSS
 */
@ParametersAreNonnullByDefault
@Service
public class DealNotificationPdfSignerService {
    private static final Logger LOGGER = LoggerFactory.getLogger(DealNotificationPdfSignerService.class);

    private static final DateTimeFormatter STAMP_DATE_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");

    private static final Box PDF_STAMP_BOX = new Box.Builder()
            .setBackgroundColor(null)
            .setMargin(Box.DEFAULT_MARGIN)
            .setWidth(320)
            .setHeight(76)
            .setHorizontalPosition(57)
            .setVerticalPosition(57)
            .setBorderColor(Colors.NAVY)
            .setBorderRadius(0)
            .setBorderWidth(2)
            .build();

    private static final Font PDF_STAMP_FONT = new Font(8, emptySet(), Font.Family.ARIAL, Colors.NAVY);
    private static final Font PDF_STAMP_LARGER_FONT = new Font(11, emptySet(), Font.Family.ARIAL, Colors.NAVY);

    private static final Icon PDF_STAMP_ICON;

    static {
        try (InputStream logoInputStream = ClassLoader.getSystemResourceAsStream("deal-notifications/logo.png")) {
            byte[] logoImageContent = IOUtils.toByteArray(logoInputStream);
            PDF_STAMP_ICON = new Icon(7, 20, new Image(logoImageContent), 10);
        } catch (IOException e) {
            throw new RuntimeIoException(e);
        }
    }

    private final DssClient dssClient;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public DealNotificationPdfSignerService(DssClient dssClient,
                                            PpcPropertiesSupport ppcPropertiesSupport) {
        this.dssClient = dssClient;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    byte[] signPdf(String pdfFileName, byte[] pdfContent) {
        List<Certificate> certificates = dssClient.getCertificateList();
        LOGGER.info("certificates = {}", certificates);

        List<Certificate> activeCertificates = filterList(certificates, Certificate::isActive);
        LOGGER.info("activeCertificates = {}", activeCertificates);

        if (activeCertificates.isEmpty()) {
            throw new IllegalStateException("no active certificates");
        }

        @Nullable
        String requiredCertificateSerialNumberSerialized =
                ppcPropertiesSupport.get(DSS_CERTIFICATE_SERIAL_NUMBER.getName());

        final Certificate relevantCertificate;
        if (requiredCertificateSerialNumberSerialized != null &&
                !StringUtils.isBlank(requiredCertificateSerialNumberSerialized)) {
            BigInteger requiredCertificateSerialNumber = new BigInteger(requiredCertificateSerialNumberSerialized, 16);
            List<Certificate> allRelevantCertificates = filterList(activeCertificates,
                    certificate -> certificate.getCertificateSerialNumber().equals(requiredCertificateSerialNumber));

            if (allRelevantCertificates.isEmpty()) {
                throw new IllegalStateException("no active certificates with serial = " +
                        requiredCertificateSerialNumberSerialized);
            }

            if (allRelevantCertificates.size() > 1) {
                throw new IllegalStateException("serial = " +
                        requiredCertificateSerialNumberSerialized + " is ambiguous");
            }

            relevantCertificate = allRelevantCertificates.get(0);
        } else {
            relevantCertificate = activeCertificates.get(0);
        }

        LOGGER.info("relevantCertificate = {}", relevantCertificate);

        String certificateValidityPeriod = String.format("Срок действия: с %s по %s",
                formatStampDate(relevantCertificate.getStartDate()),
                formatStampDate(relevantCertificate.getEndDate()));

        int leftMargin = 80;
        List<Text> textContent = asList(
                new Text(PDF_STAMP_LARGER_FONT, "Документ подписан электронной подписью", leftMargin),
                new Text(PDF_STAMP_FONT, "Сертификат: " + hexSerialNumber(relevantCertificate), leftMargin),
                new Text(PDF_STAMP_FONT, certificateValidityPeriod, leftMargin),
                new Text(PDF_STAMP_FONT, "Владелец: ООО «ЯНДЕКС»", leftMargin));

        Stamp stamp = Stamp.textWithIconStamp(PDF_STAMP_BOX, textContent, PDF_STAMP_ICON);

        return dssClient.signPdf(
                pdfContent, pdfFileName, stamp,
                relevantCertificate.getId(), relevantCertificate.getPinCode());

    }

    private static String formatStampDate(Date date) {
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        return STAMP_DATE_FORMATTER.format(localDate);
    }

    private static String hexSerialNumber(Certificate certificate) {
        return certificate.getCertificateSerialNumber().toString(16).toUpperCase();
    }

}
