package ru.yandex.direct.core.entity.deal.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.core.entity.agency.service.AgencyService;
import ru.yandex.direct.core.entity.deal.model.BalancePrivateDealInfo;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.repository.DealNotificationRepository;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.pdfgen.PdfBuilder;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.inside.mds.MdsUrlUtils;
import ru.yandex.misc.io.ByteArrayInputStreamSource;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для отправки уведомлений о частных сделках
 * <p>
 * Уведомление отправляется электронной почтой. В уведомлении текст с информацией
 * и вложение с подписанным PDF-файлом, который имеет силу дополнительного соглашения.
 */
@ParametersAreNonnullByDefault
@Service
public class DealNotificationService {
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("HHmmss");

    private static final Logger LOGGER = LoggerFactory.getLogger(DealNotificationService.class);
    private static final String PDF_TEMPLATE_FILENAME = "deal-notifications/deal-notification.fo.thyme";

    private final MdsHolder mdsHolder;
    private final PdfBuilder pdfBuilder;
    private final AgencyService agencyService;
    private final DealNotificationRepository dealNotificationRepository;
    private final DealNotificationPdfSignerService pdfSignerService;
    private final DealNotificationMailSenderService mailSenderService;
    private final BalancePrivateDealInfoService balancePrivateDealInfoService;
    private final ShardHelper shardHelper;

    @Autowired
    public DealNotificationService(MdsHolder mdsHolder,
                                   PdfBuilder pdfBuilder,
                                   AgencyService agencyService,
                                   DealNotificationRepository dealNotificationRepository,
                                   DealNotificationPdfSignerService pdfSignerService,
                                   DealNotificationMailSenderService mailSenderService,
                                   BalancePrivateDealInfoService balancePrivateDealInfoService,
                                   ShardHelper shardHelper) {
        this.mdsHolder = mdsHolder;
        this.pdfBuilder = pdfBuilder;
        this.dealNotificationRepository = dealNotificationRepository;
        this.pdfSignerService = pdfSignerService;
        this.mailSenderService = mailSenderService;
        this.balancePrivateDealInfoService = balancePrivateDealInfoService;
        this.shardHelper = shardHelper;
        this.agencyService = agencyService;
    }

    /**
     * Отправить уведомление о заключении частной сделки
     *
     * @param deal про какую частную сделку уведомление
     */
    public void sendDealNotification(Deal deal) {
        ClientId clientId = ClientId.fromLong(deal.getClientId());

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        String notificationId = DATE_FORMATTER.format(LocalDate.now()) + "-" + deal.getId();
        LOGGER.info("notificationId = {}", notificationId);

        String pdfFileName = String.format("yandex-direct-deal-confirmation-%s.pdf", notificationId);

        BalancePrivateDealInfo balancePrivateDealInfo =
                balancePrivateDealInfoService.getBalancePrivateDealInfo(clientId.asLong());

        String contractId = balancePrivateDealInfo.getExternalContractId();
        Calendar contractDate = GregorianCalendar.from(
                balancePrivateDealInfo.getContractStartDate().atStartOfDay(ZoneId.systemDefault()));
        String contractAdditionalAgreementId = balancePrivateDealInfo.getExternalCollateralId();

        List<Long> pageIds = mapList(deal.getPlacements(), DealPlacement::getPageId);

        String displayedSpecialPremiumPercentage = Optional.ofNullable(deal.getAgencyFeePercent())
                .map(Percent::asPercent)
                .map(BigDecimal::stripTrailingZeros)
                .map(percentage -> percentage.scale() < 0
                        ? percentage.setScale(0, BigDecimal.ROUND_UNNECESSARY)
                        : percentage)
                .map(BigDecimal::toString)
                .orElse("0");

        byte[] pdfContent = createPdf(deal, notificationId,
                contractId, contractDate,
                pageIds, displayedSpecialPremiumPercentage);

        byte[] signedPdfContent = pdfSignerService.signPdf(pdfFileName, pdfContent);

        MdsFileKey mdsKey = uploadToMds(mdsHolder, clientId.asLong(), pdfFileName, signedPdfContent);

        DealNotificationEmailParameters parameters = new DealNotificationEmailParameters.Builder()
                .setDeal(deal)
                .setNotificationId(notificationId)
                .setContractId(contractId)
                .setContractDate(contractDate)
                .setContractAdditionalAgreementId(contractAdditionalAgreementId)
                .setPdfFileName(pdfFileName)
                .setSignedPdfContent(signedPdfContent)
                .setEmailTo(agencyService.getDealNotificationEmail(clientId))
                .build();

        String mdsUrl = getMdsUrl(mdsHolder, mdsKey);

        mailSenderService.sendDealNotificationEmail(parameters, mdsUrl);
        dealNotificationRepository.logNotification(shard, clientId, notificationId, deal.getId(),
                mdsKey.serialize(), mdsUrl);
    }

    private byte[] createPdf(Deal deal, String notificationId,
                             String contractId, Calendar contractDate,
                             List<Long> pageIds, String specialPremiumPercentage) {
        DealNotificationPdfParameters parameters = new DealNotificationPdfParameters.Builder()
                .setNotificationId(notificationId)
                .setContractId(contractId)
                .setContractDate(contractDate)
                .setDealId(deal.getId())
                .setPageIds(pageIds)
                .setSpecialPremiumPercentage(specialPremiumPercentage)
                .setEmailFrom(DealNotificationMailSenderService.EMAIL_FROM)
                .build();

        return pdfBuilder.buildPdf(PDF_TEMPLATE_FILENAME, parameters);
    }

    private MdsFileKey uploadToMds(MdsHolder mdsHolder, Long clientId, String pdfFileName, byte[] pdfContent) {
        // может статься, при выполнении задачи что-то пойдёт не так, а файл
        // в MDS останется; в таком случае при повторном запросе upload мы можем
        // получить 403 forbidden и упасть. мы могли бы обработать эту ошибку,
        // если б inside-mds давал какие-то детали в этом случае. а так мы просто
        // каждый раз создаём новое имя файла.
        String time = TIME_FORMATTER.format(LocalDateTime.now());
        String mdsPath = String.format("deal-notifications/%d/%s/%s", clientId, time, pdfFileName);

        // NB: есть юридическое требование хранить эти файлы по меньшей мере 5 лет
        MdsPostResponse uploadResponse = mdsHolder.upload(mdsPath, new ByteArrayInputStreamSource(pdfContent));
        return uploadResponse.getKey();
    }

    private String getMdsUrl(MdsHolder mdsHolder, MdsFileKey mdsKey) {
        return MdsUrlUtils.actionUrl(mdsHolder.getHosts().getHostPortForRead(),
                "get", mdsHolder.getNamespace().getName(),
                mdsKey);
    }

}
