package ru.yandex.direct.core.entity.deal.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.deal.container.CampaignDeal;
import ru.yandex.direct.core.entity.deal.container.DealStat;
import ru.yandex.direct.core.entity.deal.container.UpdateDealContainer;
import ru.yandex.direct.core.entity.deal.model.CompleteReason;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealAdfox;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.model.StatusAdfox;
import ru.yandex.direct.core.entity.deal.model.StatusAdfoxSync;
import ru.yandex.direct.core.entity.deal.model.StatusDirect;
import ru.yandex.direct.core.entity.deal.model.UpdatableDeal;
import ru.yandex.direct.core.entity.deal.repository.DealRepository;
import ru.yandex.direct.core.entity.deal.service.validation.DealValidationService;
import ru.yandex.direct.core.entity.placements.model.Placement;
import ru.yandex.direct.core.entity.placements.repository.PlacementsRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.ClientPerminfo;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytcomponents.model.DealStatsResponse;
import ru.yandex.direct.ytcomponents.statistics.service.CampaignStatService;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItemsWithoutWarnings;

@Service
@ParametersAreNonnullByDefault
public class DealService {

    private static final int STAT_SCALE = 2;
    private static final int STAT_SCALE_FOR_CTR = 4;
    private static final RoundingMode STAT_ROUNDING_MODE = RoundingMode.HALF_UP;
    private static final BigDecimal CPM_COEF = BigDecimal.valueOf(1000);
    public static final Percent AGENCY_TAX = Percent.fromPercent(BigDecimal.valueOf(15));

    private final DealRepository dealRepository;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final DealValidationService dealValidationService;
    private final DealNotificationService dealNotificationService;
    private final CampaignService campaignService;
    private final DslContextProvider dslContextProvider;
    private final CampaignStatService campaignStatService;
    private final UserService userService;
    private final PlacementsRepository placementsRepository;
    private final PpcRbac ppcRbac;
    private final boolean suppressNotifications;

    @Autowired
    public DealService(DealRepository dealRepository, ShardHelper shardHelper,
                       CampaignRepository campaignRepository,
                       DealValidationService dealValidationService,
                       DealNotificationService dealNotificationService,
                       DslContextProvider dslContextProvider,
                       CampaignStatService campaignStatService,
                       CampaignService campaignService,
                       UserService userService,
                       PlacementsRepository placementsRepository,
                       PpcRbac ppcRbac,
                       @Value("${deal_service.suppress_notifications}") boolean suppressNotifications) {
        this.dealRepository = dealRepository;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.dealValidationService = dealValidationService;
        this.dealNotificationService = dealNotificationService;
        this.campaignService = campaignService;
        this.dslContextProvider = dslContextProvider;
        this.campaignStatService = campaignStatService;
        this.userService = userService;
        this.placementsRepository = placementsRepository;
        this.ppcRbac = ppcRbac;
        this.suppressNotifications = suppressNotifications;
    }

    /**
     * получить описание сделки {@link Deal} для клиента
     */
    public List<Deal> getDealsBrief(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);

        return dealRepository.getDealsBriefByClientId(shard, clientId);
    }

    /**
     * получить список сделок клиента
     * читает все поля {@link Deal}
     */
    public List<Deal> getDeals(ClientId clientId, Collection<Long> dealIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return dealRepository.getDeals(shard, clientId, dealIds);
    }

    /**
     * Получение списка сделок клиента
     * Читает все поля {@link Deal}
     * Используется для получения сделок, когда неизвестен идентфикатор клиента агентства
     */
    public List<Deal> getDealsByClientForAgency(int shard, ClientId clientId, Collection<Long> dealIds) {
        ClientPerminfo clientPerminfo = ppcRbac.getClientsPerminfo(singletonList(clientId))
                .getOrDefault(clientId, Optional.empty())
                .orElse(null);
        if (clientPerminfo == null || clientPerminfo.agencyClientId() == null) {
            return emptyList();
        }
        Long agencyClientId = clientPerminfo.agencyClientId().asLong();
        int agencyShard = shardHelper.getShardsByClientIds(singletonList(agencyClientId)).get(agencyClientId);
        return dealRepository.getDeals(agencyShard, ClientId.fromLong(agencyClientId), dealIds);
    }

    /**
     * получить одну сделку из базы
     *
     * @return возвращает сделку или null, если её нет
     */
    @Nullable
    public Deal getSingleDeal(ClientId clientId, long dealId) {
        List<Deal> deals = getDeals(clientId, singletonList(dealId));

        if (deals.isEmpty()) {
            return null;
        }

        return deals.iterator().next();
    }

    public Collection<DealStat> getDealStatistics(List<Long> dealIds) {
        Map<Long, DealStat> dealStatResult = new HashMap<>();
        List<ru.yandex.direct.ytcomponents.model.CampaignDeal> requestsToYt = new ArrayList<>();

        shardHelper.forEachShard(shard -> {
            Map<Long, List<Long>> linkedCampaignsInShard = dealRepository.getAllCampaignsDealsByDealIds(shard, dealIds);
            List<ru.yandex.direct.ytcomponents.model.CampaignDeal> campaignDeals =
                    EntryStream.of(linkedCampaignsInShard)
                            .flatMapValues(Collection::stream)
                            .map(entry -> new ru.yandex.direct.ytcomponents.model.CampaignDeal()
                                    .withCampaignId(entry.getValue())
                                    .withDealId(entry.getKey()))
                            .toList();
            requestsToYt.addAll(campaignDeals);
        });

        if (requestsToYt.isEmpty()) {
            return emptyList();
        }
        Map<Long, DealStatsResponse> dealsStatFromYt = campaignStatService.getDealsStatistics(requestsToYt);

        for (Map.Entry<Long, DealStatsResponse> entry : dealsStatFromYt.entrySet()) {
            dealStatResult.put(entry.getKey(), convertDealStatsResponseToDealStat(entry.getValue()));
        }

        calculateStat(dealStatResult);

        dealStatResult.forEach((dealId, dealStat) -> dealStat.setId(dealId));

        return dealStatResult.values();
    }

    /**
     * Добавление сделки. Используется транспортом из adfox
     */
    //todo разбить на 2 метода: запись в ppcdict и принятие агентством
    public MassResult<Long> addDeals(ClientId clientId, List<Deal> deals) {
        ValidationResult<List<Deal>, Defect> validationResult =
                dealValidationService.validateAddDeal(clientId, deals);
        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), validationResult);
        }
        int shard = shardHelper.getShardByClientId(clientId);

        prepareSystemFields(deals);
        List<Long> dealIdsPpcDict = dealRepository.addDealsPpcDict(deals);
        List<Long> dealIdsPpc = dealRepository.addDealsPpc(shard, deals);
        checkState(deals.size() == dealIdsPpc.size() && deals.size() == dealIdsPpcDict.size(),
                "Not all deals are saved");
        return MassResult.successfulMassAction(dealIdsPpcDict, validationResult);
    }

    private void prepareSystemFields(Collection<Deal> deals) {
        deals.forEach(deal -> {
            deal.withDirectStatus(StatusDirect.RECEIVED);
            deal.withStatusAdfoxSync(StatusAdfoxSync.NO);
        });
    }

    /**
     * Обновляет параметры сделки, управляемые Adfox'ом.
     * <p>
     * <b>Внимание:</b> этот метод должен использоваться только Транспортом.
     * Директ самостоятельно эти параметры не меняет.
     */
    public MassResult<Long> updateAdfoxDeals(List<DealAdfox> deals) {
        ValidationResult<List<DealAdfox>, Defect> validationResult =
                dealValidationService.validateUpdateDealAdfox(deals);
        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), validationResult);
        }
        List<Long> dealIds = mapList(deals, DealAdfox::getId);
        // да, мы ходим за сделками два раза за один update: в валидации и тут. Планируемая нагрузка на этот метод невелика
        List<DealAdfox> existingDeals = dealRepository.getDealsAdfox(dealIds);
        Map<Long, DealAdfox> existingDealsById = listToMap(existingDeals, DealAdfox::getId);
        List<AppliedChanges<DealAdfox>> allAppliedChanges = new ArrayList<>(dealIds.size());
        for (DealAdfox updateRequest : deals) {
            Long id = updateRequest.getId();
            DealAdfox model = existingDealsById.get(id);
            AppliedChanges<DealAdfox> appliedChanges =
                    ModelChanges.build(model, DealAdfox.ADFOX_STATUS, DealAdfox.ADFOX_STATUS.get(updateRequest))
                            .applyTo(model);
            allAppliedChanges.add(appliedChanges);
        }
        dealRepository.updateDealsAdfox(allAppliedChanges);
        completeDealsIfRequired(allAppliedChanges);
        return MassResult.successfulMassAction(dealIds, validationResult);
    }

    private void completeDealsIfRequired(List<AppliedChanges<DealAdfox>> appliedChanges) {
        List<DealAdfox> dealsForComplete =
                mapList(filterList(appliedChanges, this::isAdfoxClosedDeal), AppliedChanges::getModel);
        shardHelper.groupByShard(dealsForComplete, ShardKey.CLIENT_ID, DealAdfox::getClientId).forEach(
                (shard, deals) -> changeDealsStatus(dslContextProvider.ppc(shard),
                        mapList(deals, DealAdfox::getId),
                        StatusDirect.COMPLETED, CompleteReason.BY_PUBLISHER, Applicability.FULL));
    }

    /**
     * @return {@code true}, если в {@code appliedChanges} статус сделки в AdFox меняется на Closed
     */
    private boolean isAdfoxClosedDeal(AppliedChanges<DealAdfox> appliedChanges) {
        return appliedChanges.changed(Deal.ADFOX_STATUS) &&
                (appliedChanges.getNewValue(Deal.ADFOX_STATUS) == StatusAdfox.CLOSED);
    }

    /**
     * Достать валидный список связок сделок с кампаниями
     */
    private <T extends Model> List<T> getValid(ValidationResult<UpdateDealContainer, Defect> validation,
                                               ModelProperty<? super UpdateDealContainer, List<T>> modelProperty) {
        List<T> campaignsDeals = modelProperty.get(validation.getValue());
        if (isEmpty(campaignsDeals)) {
            return emptyList();
        }
        ValidationResult<List<T>, Defect> v =
                validation.getOrCreateSubValidationResult(field(modelProperty.name()), campaignsDeals);
        return getValidItemsWithoutWarnings(v);
    }

    /**
     * Достать список связок сделок с кампаниями без errors
     */
    private <T extends Model> List<T> getItemsWithoutErrors(
            ValidationResult<UpdateDealContainer, Defect> validation,
            ModelProperty<? super UpdateDealContainer, List<T>> modelProperty) {
        List<T> campaignsDeals = modelProperty.get(validation.getValue());
        if (isEmpty(campaignsDeals)) {
            return emptyList();
        }
        ValidationResult<List<T>, Defect> v =
                validation.getOrCreateSubValidationResult(field(modelProperty.name()), campaignsDeals);
        return getValidItems(v);
    }

    private void agencyLinkCampaigns(int shard, List<CampaignDeal> campaignDeals) {
        if (campaignDeals.isEmpty()) {
            return;
        }
        List<Long> campaignIds = mapList(campaignDeals, CampaignDeal::getCampaignId);
        TransactionalRunnable transactionalTask = conf -> {
            dealRepository.linkCampaigns(conf, campaignDeals);
            campaignRepository.resetBannerSystemSyncStatus(conf, campaignIds);
        };
        dslContextProvider.ppcTransaction(shard, transactionalTask);
    }

    private void unlinkCampaigns(int shard, List<CampaignDeal> campaignDeals) {
        if (campaignDeals.isEmpty()) {
            return;
        }
        List<Long> campaignIds = mapList(campaignDeals, CampaignDeal::getCampaignId);
        TransactionalRunnable transactionalTask = conf -> {
            dealRepository.unlinkCampaigns(conf, campaignDeals);
            campaignRepository.resetBannerSystemSyncStatus(conf, campaignIds);
        };
        dslContextProvider.ppcTransaction(shard, transactionalTask);
    }

    /**
     * получение списка прилинкованных кампаний для доступных клиенту сделок
     * Если каких-то сделок из dealIds нет у клиента, их не будет в результирующей мапе
     */
    public Map<Long, List<Long>> getLinkedCampaignsByClientId(ClientId clientId) {
        List<Long> existingDealIds = mapList(getDealsBrief(clientId), Deal::getId);
        return getLinkedCampaignsByDealIds(existingDealIds);
    }

    /**
     * получение списка прилинкованных к сделкам кампаний по списку сделок
     * Если каких-то сделок из dealIds нет у клиента, их не будет в результирующей мапе
     */
    //todo можно оптимизировать количество шардов для агентства, посмотрев в каких шардах лежать клиенты агентства clientId
    public Map<Long, List<Long>> getLinkedCampaignsByDealIds(List<Long> dealIds) {
        Map<Long, List<Long>> linkedCampaigns = new HashMap<>();
        shardHelper.forEachShard(shard -> pushLinkedCampaigns(shard, dealIds, linkedCampaigns));
        return linkedCampaigns;
    }

    /**
     * Добавляет в aggregator связки из репозитория для шарда
     */
    private void pushLinkedCampaigns(int shard, List<Long> dealIds, Map<Long, List<Long>> aggregator) {
        EntryStream.of(dealRepository.getLinkedCampaigns(shard, dealIds))
                .forKeyValue((dealId, campaignIds) -> {
                    aggregator.putIfAbsent(dealId, new ArrayList<>());
                    aggregator.get(dealId).addAll(campaignIds);
                });
    }

    /**
     * Активировать список сделок
     *
     * @param clientId владелец сделки
     */
    public MassResult<Long> activateDeals(ClientId clientId, List<Long> dealIds, Applicability applicability) {
        int shard = shardHelper.getShardByClientId(clientId);
        DSLContext dslContext = dslContextProvider.ppc(shard);

        return dslContext.transactionResult(conf -> {
            DSLContext txContext = conf.dsl();

            MassResult<Long> result = changeDealsStatus(txContext, dealIds, StatusDirect.ACTIVE, null, applicability);

            for (Result<Long> dealResult : result.getResult()) {
                if (dealResult.isSuccessful()) {
                    Deal deal = getSingleDeal(clientId, dealResult.getResult());
                    if (!suppressNotifications) {
                        dealNotificationService.sendDealNotification(deal);
                    }
                }
            }

            return result;
        });
    }

    /**
     * Установить статус Completed для списка сделок
     *
     * @param clientId владелец сделки
     */
    public MassResult<Long> completeDeals(ClientId clientId, List<Long> dealIds, CompleteReason completeReason,
                                          Applicability applicability) {
        return changeDealsStatus(clientId, dealIds, StatusDirect.COMPLETED, completeReason, applicability);
    }

    /**
     * Архивировать сделки список сделок
     *
     * @param clientId владелец сделки
     */
    public MassResult<Long> archiveDeals(ClientId clientId, List<Long> dealIds, Applicability applicability) {
        return changeDealsStatus(clientId, dealIds, StatusDirect.ARCHIVED, null, applicability);
    }

    /**
     * Привязывает и отвязывает кампании от сделок, меняет название сделки и описание
     *
     * @param agencyUid           агентство, владелец сделок
     * @param agencyId            агентство, владелец сделок
     * @param updateDealContainer связки сделок с кампаниями которые нужно добавить/удалить
     */
    public Result<UpdateDealContainer> updateDeal(long agencyUid, ClientId agencyId,
                                                  UpdateDealContainer updateDealContainer) {
        int dealsShard = shardHelper.getShardByClientId(agencyId);

        ValidationResult<UpdateDealContainer, Defect> v =
                dealValidationService.validateUpdateDeal(dealsShard, agencyUid, agencyId, updateDealContainer);

        if (getItemsWithoutErrors(v, UpdateDealContainer.ADDED).isEmpty() && getItemsWithoutErrors(v,
                UpdateDealContainer.REMOVED).isEmpty() && getItemsWithoutErrors(v, UpdateDealContainer.DEALS).isEmpty()) {
            return Result.broken(v);
        }

        List<CampaignDeal> validAdded = getValid(v, UpdateDealContainer.ADDED);
        List<CampaignDeal> validRemoved = getValid(v, UpdateDealContainer.REMOVED);
        List<UpdatableDeal> validDeals = getValid(v, UpdateDealContainer.DEALS);

        if (!(validAdded.isEmpty() && validRemoved.isEmpty())) {
            agencyLinkCampaigns(validAdded, validRemoved);
        }

        if (!validDeals.isEmpty()) {
            updateDeal(dealsShard, agencyId, validDeals);
        }

        return Result.successful(
                updateDealContainer.withAdded(validAdded).withRemoved(validRemoved).withDeals(validDeals), v);
    }

    private void agencyLinkCampaigns(List<CampaignDeal> validAdded, List<CampaignDeal> validRemoved) {
        //разогреть кеш шардов по CID одним запросом в базу, а не двумя
        shardHelper.groupByShard(StreamEx.of(validAdded).append(validRemoved).toList(),
                ShardKey.CID, CampaignDeal::getCampaignId);

        shardHelper.groupByShard(validAdded, ShardKey.CID, CampaignDeal::getCampaignId)
                .chunkedByDefault()
                .stream()
                .forKeyValue(this::agencyLinkCampaigns);

        shardHelper.groupByShard(validRemoved, ShardKey.CID, CampaignDeal::getCampaignId)
                .chunkedByDefault()
                .stream()
                .forKeyValue(this::unlinkCampaigns);
    }

    private void updateDeal(int shard, ClientId agencyId, List<UpdatableDeal> updatableDealList) {
        List<Long> updateDealIds = mapList(updatableDealList, UpdatableDeal::getId);

        Map<Long, Deal> updateDealsByIds = listToMap(getDeals(agencyId, updateDealIds), Deal::getId);

        List<AppliedChanges<Deal>> appliedChanges = StreamEx.of(updatableDealList)
                .map(this::getDealChanges)
                .map(dealModelChanges -> dealModelChanges.applyTo(updateDealsByIds.get(dealModelChanges.getId())))
                .toList();

        dealRepository.updateDeals(shard, appliedChanges);
    }

    private ModelChanges<Deal> getDealChanges(UpdatableDeal deal) {
        ModelChanges<Deal> changes = new ModelChanges<>(deal.getId(), Deal.class);
        changes.process(deal.getName(), Deal.NAME);
        changes.process(deal.getDescription(), Deal.DESCRIPTION);
        return changes;
    }

    private MassResult<Long> changeDealsStatus(ClientId clientId, List<Long> dealIds, StatusDirect statusDirect,
                                               @Nullable CompleteReason completeReason, Applicability applicability) {
        int shard = shardHelper.getShardByClientId(clientId);
        return changeDealsStatus(dslContextProvider.ppc(shard),
                dealIds, statusDirect, completeReason, applicability);
    }

    private MassResult<Long> changeDealsStatus(DSLContext dslContext, List<Long> dealIds, StatusDirect statusDirect,
                                               @Nullable CompleteReason completeReason, Applicability applicability) {
        ValidationResult<List<Long>, Defect> validation =
                dealValidationService.validateChangeStatus(dslContext, dealIds, statusDirect);
        List<Long> validItems = getValidItems(validation);
        if (validItems.isEmpty() || (validation.hasAnyErrors() && isFull(applicability))) {
            return MassResult.brokenMassAction(dealIds, validation);
        }

        shardHelper.forEachShard(shard -> {
            Map<Long, List<Long>> linkedCampaignsInShard = dealRepository.getLinkedCampaigns(shard, validItems);
            List<Long> campaignIds = EntryStream.of(linkedCampaignsInShard)
                    .flatMapValues(Collection::stream)
                    .distinctValues()
                    .values()
                    .toList();
            campaignRepository.resetBannerSystemSyncStatus(shard, campaignIds);
        });

        List<AppliedChanges<Deal>> changeStatusAppliedChanges =
                generateAppliedChanges(validItems, statusDirect, completeReason);
        updateDeals(dslContext, changeStatusAppliedChanges);

        return MassResult.successfulMassAction(dealIds, validation);
    }

    @SuppressWarnings("squid:S4165")
    private int updateDeals(DSLContext dslContext, List<AppliedChanges<Deal>> appliedChanges) {
        // pre-update
        appliedChanges = resetAdfoxSyncStatusIfRequired(appliedChanges);

        return dealRepository.updateDeals(dslContext, appliedChanges);
    }

    private List<AppliedChanges<Deal>> resetAdfoxSyncStatusIfRequired(List<AppliedChanges<Deal>> appliedChanges) {
        appliedChanges.forEach(ac -> {
            if (isAnyAdfoxSyncedPropertyChanged(ac)) {
                ac.modify(Deal.STATUS_ADFOX_SYNC, StatusAdfoxSync.NO);
            }
        });
        return appliedChanges;
    }

    /**
     * @return {@code true}, если в {@code appliedChanges} менялось какое-либо из полей,
     * которые мы должны отправлять в Adfox
     */
    private boolean isAnyAdfoxSyncedPropertyChanged(AppliedChanges<Deal> appliedChanges) {
        return appliedChanges.changed(Deal.DIRECT_STATUS);
    }

    private List<AppliedChanges<Deal>> generateAppliedChanges(List<Long> dealIds, StatusDirect status,
                                                              @Nullable CompleteReason completeReason) {
        return StreamEx.of(dealIds)
                .map(dealId -> {
                    Deal deal = new Deal();
                    deal.setId(dealId);
                    ModelChanges<Deal> changes = new ModelChanges<>(deal.getId(), Deal.class);
                    changes.process(status, Deal.DIRECT_STATUS);
                    changes.processNotNull(completeReason, Deal.COMPLETE_REASON);
                    return changes.applyTo(deal);
                })
                .collect(Collectors.toList());
    }

    private DealStat convertDealStatsResponseToDealStat(DealStatsResponse relatedStat) {
        return new DealStat()
                .withClicks(relatedStat.getClicks())
                .withShows(relatedStat.getShows())
                .withSpent(relatedStat.getSpent());
    }

    private void calculateStat(Map<Long, DealStat> dealStatResult) {
        EntryStream.of(dealStatResult).forKeyValue((dealId, dealStat) -> dealStat
                .withCtr(calculateCtr(dealStat))
                .withCpm(calculateCpm(dealStat))
                .withCpc(calculateCpc(dealStat)));
    }

    private Percent calculateCtr(DealStat dealStat) {
        Long shows = dealStat.getShows();
        Long clicks = dealStat.getClicks();
        if (shows == null || shows == 0 || clicks == null) {
            return null;
        }
        return Percent.fromRatio(BigDecimal.valueOf(clicks)
                .divide(BigDecimal.valueOf(shows), STAT_SCALE_FOR_CTR, STAT_ROUNDING_MODE));
    }

    private BigDecimal calculateCpm(DealStat dealStat) {
        Long shows = dealStat.getShows();
        BigDecimal spent = dealStat.getSpent();
        if (shows == null || shows == 0 || spent == null) {
            return null;
        }
        return spent.divide(BigDecimal.valueOf(shows).divide(CPM_COEF),
                STAT_SCALE, STAT_ROUNDING_MODE);
    }


    private BigDecimal calculateCpc(DealStat dealStat) {
        Long clicks = dealStat.getClicks();
        BigDecimal spent = dealStat.getSpent();
        if (clicks == null || clicks == 0 || spent == null) {
            return null;
        }
        return spent.divide(BigDecimal.valueOf(clicks), STAT_SCALE, STAT_ROUNDING_MODE);
    }

    /**
     * получение списка прилинкованных к кампаниям сделок по списку кампаний
     * Если каких-то кампаний из  campaignIds нет у клиента, их не будет в результирующей мапе
     */
    public Map<Long, List<Long>> getLinkedDealsByCampaignsIds(List<Long> cids) {
        Map<Long, List<Long>> linkedDeals = new HashMap<>();
        shardHelper.forEachShard(shard -> dealRepository.getLinkedDeals(shard, cids)
                .forEach((cid, dealIds) -> linkedDeals.computeIfAbsent(cid, id -> new ArrayList<>()).addAll(dealIds)));
        return linkedDeals;
    }

    /**
     * получение списка прилинкованных к кампаниям сделок по списку кампаний, лежащих в одном шарде
     * Если каких-то кампаний из  campaignIds нет у клиента, их не будет в результирующей мапе
     */
    public Map<Long, List<Long>> getLinkedDealsByCampaignsIdsInShard(int shard, Collection<Long> cids) {
        return dealRepository.getLinkedDeals(shard, cids);
    }

    /**
     * Поиск кампаний, к которым можно привязывать сделки, по части id или названия для одного ClientID
     *
     * @param clientId ClientID, для которого ищем кампании
     */
    public List<CampaignSimple> searchDealCampaignsByClientId(ClientId clientId) {
        return campaignService
                .searchCampaignsByClientIdAndTypes(clientId, singletonList(CampaignType.CPM_DEALS));
    }

    /**
     * Поиск кампаний, к которым можно привязывать сделки, по части id или названия для списка ClientID
     *
     * @param clientIds список ClientID, для которых ищем кампании
     */
    public Map<Long, List<CampaignSimple>> searchDealCampaignsByClientIds(Collection<Long> clientIds) {
        return campaignService
                .searchCampaignsByClientIdsAndTypes(clientIds, singletonList(CampaignType.CPM_DEALS));
    }

    /**
     * По списку сделок возвращает информацию о владельцах сделок
     *
     * @param dealIds список сделок для поиска владельцев
     */
    public Map<Long, LoginAndClientId> getLoginAndClientIdsByDealIds(Collection<Long> dealIds) {
        Map<Long, Long> clientIdsByDealIds = dealRepository.getClientIdsByDealIdsFromDealsAdfox(dealIds);
        Map<ClientId, String> loginsByClientIds =
                userService.getChiefsLoginsByClientIds(mapList(clientIdsByDealIds.values(), ClientId::fromLong));
        return EntryStream.of(clientIdsByDealIds).mapValues(
                c -> LoginAndClientId.of(loginsByClientIds.get(ClientId.fromLong(c)), ClientId.fromLong(c))
        ).toMap();
    }

    /**
     * Получение подробной информации о площадках из списка сделок
     *
     * @param deals - сделки
     * @return маппинг PageID в подробную информацию о данной площадке
     */
    public Map<Long, Placement> getPlacementsMapByDeals(Collection<Deal> deals) {
        List<Long> pageIds = mapList(flatMap(deals, DealAdfox::getPlacements), DealPlacement::getPageId);
        List<Placement> placements = placementsRepository.getPlacements(pageIds);
        return listToMap(placements, Placement::getPageId);
    }

    /**
     * Извлекает из информации о площадке только те форматы, которые применимы к переданной сделке
     *
     * @param placement     - информация о площадке
     * @param dealPlacement - площадка и набор блоков из сделки
     * @return набор форматов
     */
    @SuppressWarnings("unchecked")
    public static Set<String> extractFormatsFromPlacement(Placement placement, DealPlacement dealPlacement) {
        Set<String> formats = new HashSet<>();
        if (placement.getBlocks() != null) {
            Map<String, List<String>> blocks = fromJson(placement.getBlocks(), Map.class);
            dealPlacement.getImpId()
                    .forEach(impId -> formats.addAll(blocks.getOrDefault(impId.toString(), emptyList())));
        }
        return formats;
    }

}
